package com.huawei.hwmdemo.view.sdk;

import android.graphics.Color;
import android.os.Bundle;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.widget.TextView;

import androidx.annotation.NonNull;

import com.huawei.hwmbiz.HWMBizSdk;
import com.huawei.hwmdemo.R;
import com.huawei.hwmdemo.view.base.BaseDialogFragment;
import com.huawei.hwmfoundation.callback.HwmCallback;
import com.huawei.hwmfoundation.utils.StringUtil;
import com.huawei.hwmsdk.model.result.AccessAccountResource;
import com.huawei.hwmsdk.model.result.ConcurrentResource;
import com.huawei.hwmsdk.model.result.DeviceResource;
import com.huawei.hwmsdk.model.result.VmrResource;

public class DeviceResourceFragment extends BaseDialogFragment {

    public final static String TAG = DeviceResourceFragment.class.getSimpleName();

    private View vRoot;

    private TextView mPayStateDesc;
    // 接入帐号
    private TextView mAccessAccountAvailable;
    private TextView mConfResourceName;
    // 截止时间
    private TextView mConfResourceExpireDate;
    // 单次会议时长
    private View mConfResourceDurationLayout;
    private TextView mConfResourceDuration;
    private View mConfResourceConfIdLayout;
    private TextView mConfResourceConfId;

    // 会议资源
    private DeviceResource mDeviceResource;

    // 并发资源
    private ConcurrentResource mConcurrentResource;

    // 并发资源是否下发
    private boolean mConcurrentResourceValid;

    // vmr资源
    private VmrResource mVmrResource;

    // vmr 资源是否下发
    private boolean mVmrResourceValid;

    // vmr 资源状态
    private int mVmrResourceStatus;
    // 接入帐号资源
    private AccessAccountResource mAccessAccountResource;

    // 接入帐号是否可用
    private boolean mAccessAccountResourceValid;

    // 接入帐号状态状态
    private int mAccessAccountResourceStatus;

    // 共享VMR资源是否可用
    private boolean mShareVmrResourceValid;

    private void initDeviceResource(DeviceResource deviceResource) {
        if (deviceResource != null) {
            mDeviceResource = deviceResource;

            // 接入帐号资源
            mAccessAccountResource = deviceResource.getAccessAccountResource();
            mAccessAccountResourceStatus = mAccessAccountResource.getStatus();
            mAccessAccountResourceValid = mAccessAccountResourceStatus == 0;

            // 云会议资源
            mVmrResource = deviceResource.getVmrResource();
            mVmrResourceValid = mVmrResource.getIsValid();
            mVmrResourceStatus = mVmrResource.getStatus();

            // 并发资源
            mConcurrentResource = deviceResource.getConcurrentResource();
            mConcurrentResourceValid = mConcurrentResource.getIsValid();

            // 共享VMR
            mShareVmrResourceValid = mDeviceResource.getIsEnableShareVmr();
        }
    }

    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        if (getDialog() != null) {
            getDialog().requestWindowFeature(Window.FEATURE_NO_TITLE);
        }
        vRoot = inflater.inflate(R.layout.sdk_device_resource, container, false);
        initViews();
        refreshDeviceResource();
        return vRoot;
    }

    private void refreshDeviceResource() {
        HWMBizSdk.getBizOpenApi().getDeviceResource(new HwmCallback<DeviceResource>() {
            @Override
            public void onFailed(int retCode, String desc) {

            }

            @Override
            public void onSuccess(DeviceResource deviceResource) {
                initDeviceResource(deviceResource);
                initResourceLayout();
                initPayStateDesc();
            }
        });
    }

    private void initViews() {
        mPayStateDesc = vRoot.findViewById(R.id.device_info_pay_state_desc);
        mAccessAccountAvailable = vRoot.findViewById(R.id.conf_access_account_available);
        mConfResourceName = vRoot.findViewById(R.id.conf_name);
        mConfResourceExpireDate = vRoot.findViewById(R.id.conf_expire_date);
        mConfResourceDurationLayout = vRoot.findViewById(R.id.conf_duration_layout);
        mConfResourceDuration = vRoot.findViewById(R.id.conf_duration_text);
        mConfResourceConfIdLayout = vRoot.findViewById(R.id.conf_id_layout);
        mConfResourceConfId = vRoot.findViewById(R.id.conf_id);

        initResourceLayout();
    }

    private void initResourceLayout() {
        if (mAccessAccountAvailable != null) {
            mAccessAccountAvailable.setText("状态获取中…");
        }
        if (mDeviceResource == null) {
            resetData();
            if (mAccessAccountAvailable != null) {
                mAccessAccountAvailable.setText("状态获取失败");
            }
            setNoResource();
        } else {
            // 帐号状态
            if (mAccessAccountAvailable != null) {
                mAccessAccountAvailable.setText(mAccessAccountResourceValid ? "生效中" : "已到期");
            }
            // 设置显示资源
            setDeviceResources();
        }
    }

    private void initPayStateDesc() {
        // 会议服务状态（已到期,已激活,资源已到期）
        if (!mAccessAccountResourceValid) {
            // 已到期
            mPayStateDesc.setText("已到期");
        } else {
            if (mVmrResourceValid || mConcurrentResourceValid || mShareVmrResourceValid) {
                // 已激活
                mPayStateDesc.setText("已激活");
            } else {
                // 资源已到期
                mPayStateDesc.setText("资源已到期");
            }
        }
    }

    /**
     * 资源优先级: 专属VMR > 会议并发 > 共享VMR
     */
    private void setDeviceResources() {
        // 无并发 无vmr 无共享VMR
        if (!mVmrResourceValid && !mConcurrentResourceValid && !mShareVmrResourceValid) {
            // 无可用资源
            setNoResource();
            return;
        }
        // 优先使用专属VMR资源
        if (mVmrResourceValid && mVmrResourceStatus != VmrResourceStatus.DISABLED.getValue()) {
            setVmrResource();
        } else if (mConcurrentResourceValid) {
            // 专属VMR不可用，使用并发资源
            setConcurrentResource();
        } else if (mShareVmrResourceValid) {
            // 专属VMR不可用，且无并发资源，使用共享VMR资源
            setShareVmrResource();
        } else {
            setVmrResource();
        }
    }

    // 显示并发资源
    private void setConcurrentResource() {
        mConfResourceConfId.setText("随机会议号");
        mConfResourceName.setText("会议并发(企业共享)");
        String expireDateStr = mConcurrentResource.getExpireDate();
        mConfResourceExpireDate.setTextColor(Color.parseColor("#333333"));
        if (!TextUtils.isEmpty(expireDateStr)) {
            mConfResourceExpireDate.setText(String.format("截止%1$s", expireDateStr));
            mConfResourceExpireDate.setVisibility(View.VISIBLE);
        }

        mConfResourceDurationLayout.setVisibility(View.GONE);
        mConfResourceConfIdLayout.setVisibility(View.VISIBLE);
    }

    // 显示云会议资源
    private void setVmrResource() {
        // 会议ID
        mConfResourceConfId.setText(StringUtil.formatConfId(mVmrResource.getVmrConfId()));
        String expireDateStr = mVmrResource.getExpireDate();

        String expiredStr = "";
        // 已到期需要设置不同颜色
        if (mVmrResourceStatus == 2) {
            expiredStr = "(已到期)";
            mConfResourceExpireDate.setTextColor(Color.parseColor("#999999"));
        } else if (mVmrResourceStatus == 1) {
            mConfResourceExpireDate.setTextColor(Color.parseColor("#999999"));
            mConfResourceExpireDate.setText("已停用");
            mConfResourceExpireDate.setVisibility(View.VISIBLE);
        } else {
            mConfResourceExpireDate.setTextColor(Color.parseColor("#333333"));
        }
        // 到期时间
        if (!TextUtils.isEmpty(expireDateStr)) {
            mConfResourceExpireDate.setText(String.format("截止%1$s", expireDateStr) + expiredStr);
            mConfResourceExpireDate.setVisibility(View.VISIBLE);
        }
        // 资源名
        mConfResourceName.setText(String.format("%1$s方云会议室", mVmrResource.getMaxParties()));
        // 限时时间
        int duration = mVmrResource.getDuration();
        if (duration > 0) {
            mConfResourceDuration.setText(String.format("限时%1$s分钟", "" + duration));
        } else {
            mConfResourceDuration.setText("不限时");
        }

        mConfResourceDurationLayout.setVisibility(View.VISIBLE);
        mConfResourceConfIdLayout.setVisibility(View.VISIBLE);
    }

    // 显示并发资源
    private void setShareVmrResource() {
        mConfResourceConfId.setText("随机会议号");
        mConfResourceName.setText("会议并发(企业共享)");
        mConfResourceName.setText("共享云会议室(企业共享)");
        mConfResourceExpireDate.setVisibility(View.VISIBLE);
        mConfResourceDuration.setText("不限时");
        mConfResourceDurationLayout.setVisibility(View.VISIBLE);
        mConfResourceConfIdLayout.setVisibility(View.VISIBLE);
    }

    private void setNoResource() {
        // 无可用资源
        mConfResourceName.setText("暂无会议资源");
        mConfResourceExpireDate.setText("");
        mConfResourceDurationLayout.setVisibility(View.GONE);
        mConfResourceConfIdLayout.setVisibility(View.GONE);
    }

    private void resetData() {
        mDeviceResource = null;
        mConcurrentResource = null;
        mConcurrentResourceValid = false;
        mShareVmrResourceValid = false;
        mVmrResource = null;
        mVmrResourceValid = false;
        mVmrResourceStatus = 0;
        mAccessAccountResource = null;
        mAccessAccountResourceValid = false;
        mAccessAccountResourceStatus = 0;
    }

    enum VmrResourceStatus {

        // 0 正常状态
        NORMAL(0),
        // 1 停用状态
        DISABLED(1),
        // 2 已到期
        EXPIRED(2);

        private final int value;

        VmrResourceStatus(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }
    }
}
