import { SDKERR} from './enum';
import { BookConfParam, CancelConfParam, ConfListInfo, EditConfParam, ChangeVmrInfoParam, GetConfDetailParam, CancelCycleConfParam, ModifySubCycleConfParam, CancelSubCycleConfParam, BookCycleConfParam, ConfIncomingInfo, ConfDetail, VmrListInfo, CreateConfResult, ModifyCycleConfParam} from './model';
import { needCallWithIPC } from './util';
const electron = window.require('electron');

export class IConfMgr {
	addOn!: IConfMgr;
	_onConfIncomingNotifyCallback!: (confIncomingInfo: ConfIncomingInfo) => void;
	_onConfListChangedCallback!: (confListInfo: ConfListInfo) => void;
	_onCreateConfFinishedNotifyCallback!: (ret: SDKERR, reason: string, createConfResult: CreateConfResult) => void;
	_bookConfCallback!: (ret: SDKERR, reason: string, confDetail: ConfDetail) => void;
	_cancelConfCallback!: (ret: SDKERR, reason: string) => void;
	_editConfCallback!: (ret: SDKERR, reason: string) => void;
	_getVmrListCallback!: (ret: SDKERR, reason: string, vmrListInfo: VmrListInfo) => void;
	_getConfDetailCallback!: (ret: SDKERR, reason: string, confDetail: ConfDetail) => void;
	_changeVmrInfoCallback!: (ret: SDKERR, reason: string) => void;
	_getConfListCallback!: (ret: SDKERR, reason: string, confListInfo: ConfListInfo) => void;
	_bookCycleConfCallback!: (ret: SDKERR, reason: string, confDetail: ConfDetail) => void;
	_modifyCycleConfCallback!: (ret: SDKERR, reason: string) => void;
	_modifySubCycleConfCallback!: (ret: SDKERR, reason: string) => void;
	_cancelCycleConfCallback!: (ret: SDKERR, reason: string) => void;
	_cancelSubCycleConfCallback!: (ret: SDKERR, reason: string) => void;
	_inProcessCallbackSet!: Set<string>;
	callWithIPC: boolean = false;
	constructor(addOn: IConfMgr) {
		this.addOn = addOn;
		this._inProcessCallbackSet = new Set<string>();
		this.callWithIPC = needCallWithIPC();
	}

	init(): void {
		this._inProcessCallbackSet.clear();
	}

	/**
	* [en] Notify of the conf incoming [cn] 会议来电通知
	*/
	setOnConfIncomingNotifyCB(onConfIncomingNotify: (confIncomingInfo: ConfIncomingInfo) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnConfIncomingNotifyCB', {});
			electron.ipcRenderer.on('onConfIncomingNotify', (event: any, arg: any) => {
				if (onConfIncomingNotify) {
					onConfIncomingNotify(arg["confIncomingInfo"])
				}
			});
			return;
		}

		this._onConfIncomingNotifyCallback = (confIncomingInfo: ConfIncomingInfo) => {
			if (onConfIncomingNotify) {
				onConfIncomingNotify(confIncomingInfo);
			}
		}
		this.addOn.setOnConfIncomingNotifyCB(this._onConfIncomingNotifyCallback);
	}

	/**
	* [en] Conf list notify. when login/bookconf/cancelconf, will receive this notify [cn] 会议列表通知
	*/
	setOnConfListChangedCB(onConfListChanged: (confListInfo: ConfListInfo) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnConfListChangedCB', {});
			electron.ipcRenderer.on('onConfListChanged', (event: any, arg: any) => {
				if (onConfListChanged) {
					onConfListChanged(arg["confListInfo"])
				}
			});
			return;
		}

		this._onConfListChangedCallback = (confListInfo: ConfListInfo) => {
			if (onConfListChanged) {
				onConfListChanged(confListInfo);
			}
		}
		this.addOn.setOnConfListChangedCB(this._onConfListChangedCallback);
	}

	/**
	* [en] Create Conf Finished Notify. when create conf, will receive this notify [cn] 完成创建会议通知
	*/
	setOnCreateConfFinishedNotifyCB(onCreateConfFinishedNotify: (ret: SDKERR, reason: string, createConfResult: CreateConfResult) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnCreateConfFinishedNotifyCB', {});
			electron.ipcRenderer.on('onCreateConfFinishedNotify', (event: any, arg: any) => {
				if (onCreateConfFinishedNotify) {
					onCreateConfFinishedNotify(arg["ret"], arg["reason"], arg["createConfResult"])
				}
			});
			return;
		}

		this._onCreateConfFinishedNotifyCallback = (ret: SDKERR, reason: string, createConfResult: CreateConfResult) => {
			if (onCreateConfFinishedNotify) {
				onCreateConfFinishedNotify(ret, reason, createConfResult);
			}
		}
		this.addOn.setOnCreateConfFinishedNotifyCB(this._onCreateConfFinishedNotifyCallback);
	}

	/**
	* [en] This interface is book a conference. [cn] 预定会议
	*/
	bookConf(bookConfParam: BookConfParam, onBookConfResult?: (ret: SDKERR, reason: string, confDetail: ConfDetail) => void): void {
		if (this._inProcessCallbackSet.has("bookConf")) {
			if (onBookConfResult) {
				let nullResult = <ConfDetail>{};
				onBookConfResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("bookConf");
		this._bookConfCallback = (ret: SDKERR, reason: string, confDetail: ConfDetail) => {
			this._inProcessCallbackSet.delete("bookConf");
			if (onBookConfResult) {
				onBookConfResult(ret, reason, confDetail);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('bookConf', {"bookConfParam":bookConfParam}).then((ret: any) => {
				this._inProcessCallbackSet.delete('bookConf');
				if (onBookConfResult) {
					onBookConfResult(ret["ret"], ret["reason"], ret["confDetail"]);
				}
			});
		}

		this.addOn.bookConf(bookConfParam, this._bookConfCallback);
	}

	/**
	* [en] This interface is used to cancel a conference booked before. [cn] 取消会议
	*/
	cancelConf(cancelConfParam: CancelConfParam, onCancelConfResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("cancelConf")) {
			if (onCancelConfResult) {
				onCancelConfResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("cancelConf");
		this._cancelConfCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("cancelConf");
			if (onCancelConfResult) {
				onCancelConfResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('cancelConf', {"cancelConfParam":cancelConfParam}).then((ret: any) => {
				this._inProcessCallbackSet.delete('cancelConf');
				if (onCancelConfResult) {
					onCancelConfResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.cancelConf(cancelConfParam, this._cancelConfCallback);
	}

	/**
	* [en] This interface is used to modify a conf booked before. [cn] 编辑会议
	*/
	editConf(editConfParam: EditConfParam, onEditConfResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("editConf")) {
			if (onEditConfResult) {
				onEditConfResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("editConf");
		this._editConfCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("editConf");
			if (onEditConfResult) {
				onEditConfResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('editConf', {"editConfParam":editConfParam}).then((ret: any) => {
				this._inProcessCallbackSet.delete('editConf');
				if (onEditConfResult) {
					onEditConfResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.editConf(editConfParam, this._editConfCallback);
	}

	/**
	* [en] This interface is used to get vmr list. [cn] 获取VMR列表
	*/
	getVmrList(onGetVmrListResult?: (ret: SDKERR, reason: string, vmrListInfo: VmrListInfo) => void): void {
		if (this._inProcessCallbackSet.has("getVmrList")) {
			if (onGetVmrListResult) {
				let nullResult = <VmrListInfo>{};
				onGetVmrListResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("getVmrList");
		this._getVmrListCallback = (ret: SDKERR, reason: string, vmrListInfo: VmrListInfo) => {
			this._inProcessCallbackSet.delete("getVmrList");
			if (onGetVmrListResult) {
				onGetVmrListResult(ret, reason, vmrListInfo);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('getVmrList', {}).then((ret: any) => {
				this._inProcessCallbackSet.delete('getVmrList');
				if (onGetVmrListResult) {
					onGetVmrListResult(ret["ret"], ret["reason"], ret["vmrListInfo"]);
				}
			});
		}

		this.addOn.getVmrList(this._getVmrListCallback);
	}

	/**
	* [en] This interface is used to get conf detail. [cn] 获取会议详情
	*/
	getConfDetail(getConfDetailParam: GetConfDetailParam, onGetConfDetailResult?: (ret: SDKERR, reason: string, confDetail: ConfDetail) => void): void {
		if (this._inProcessCallbackSet.has("getConfDetail")) {
			if (onGetConfDetailResult) {
				let nullResult = <ConfDetail>{};
				onGetConfDetailResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("getConfDetail");
		this._getConfDetailCallback = (ret: SDKERR, reason: string, confDetail: ConfDetail) => {
			this._inProcessCallbackSet.delete("getConfDetail");
			if (onGetConfDetailResult) {
				onGetConfDetailResult(ret, reason, confDetail);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('getConfDetail', {"getConfDetailParam":getConfDetailParam}).then((ret: any) => {
				this._inProcessCallbackSet.delete('getConfDetail');
				if (onGetConfDetailResult) {
					onGetConfDetailResult(ret["ret"], ret["reason"], ret["confDetail"]);
				}
			});
		}

		this.addOn.getConfDetail(getConfDetailParam, this._getConfDetailCallback);
	}

	/**
	* [en] This interface is used to change virtual meeting room information. [cn] 修改云会议室信息
	*/
	changeVmrInfo(vmrInfoParam: ChangeVmrInfoParam, onChangeVmrInfoResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("changeVmrInfo")) {
			if (onChangeVmrInfoResult) {
				onChangeVmrInfoResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("changeVmrInfo");
		this._changeVmrInfoCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("changeVmrInfo");
			if (onChangeVmrInfoResult) {
				onChangeVmrInfoResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('changeVmrInfo', {"vmrInfoParam":vmrInfoParam}).then((ret: any) => {
				this._inProcessCallbackSet.delete('changeVmrInfo');
				if (onChangeVmrInfoResult) {
					onChangeVmrInfoResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.changeVmrInfo(vmrInfoParam, this._changeVmrInfoCallback);
	}

	/**
	* [en]This interface is used to get conf list [cn]获取会议列表
	*/
	getConfList(onGetConfListResult?: (ret: SDKERR, reason: string, confListInfo: ConfListInfo) => void): void {
		if (this._inProcessCallbackSet.has("getConfList")) {
			if (onGetConfListResult) {
				let nullResult = <ConfListInfo>{};
				onGetConfListResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("getConfList");
		this._getConfListCallback = (ret: SDKERR, reason: string, confListInfo: ConfListInfo) => {
			this._inProcessCallbackSet.delete("getConfList");
			if (onGetConfListResult) {
				onGetConfListResult(ret, reason, confListInfo);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('getConfList', {}).then((ret: any) => {
				this._inProcessCallbackSet.delete('getConfList');
				if (onGetConfListResult) {
					onGetConfListResult(ret["ret"], ret["reason"], ret["confListInfo"]);
				}
			});
		}

		this.addOn.getConfList(this._getConfListCallback);
	}

	/**
	* [en] [cn]预约周期会议
	*/
	bookCycleConf(bookCycleConfParam: BookCycleConfParam, onBookCycleConfResult?: (ret: SDKERR, reason: string, confDetail: ConfDetail) => void): void {
		if (this._inProcessCallbackSet.has("bookCycleConf")) {
			if (onBookCycleConfResult) {
				let nullResult = <ConfDetail>{};
				onBookCycleConfResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("bookCycleConf");
		this._bookCycleConfCallback = (ret: SDKERR, reason: string, confDetail: ConfDetail) => {
			this._inProcessCallbackSet.delete("bookCycleConf");
			if (onBookCycleConfResult) {
				onBookCycleConfResult(ret, reason, confDetail);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('bookCycleConf', {"bookCycleConfParam":bookCycleConfParam}).then((ret: any) => {
				this._inProcessCallbackSet.delete('bookCycleConf');
				if (onBookCycleConfResult) {
					onBookCycleConfResult(ret["ret"], ret["reason"], ret["confDetail"]);
				}
			});
		}

		this.addOn.bookCycleConf(bookCycleConfParam, this._bookCycleConfCallback);
	}

	/**
	* [en] [cn]编辑周期会议
	*/
	modifyCycleConf(modifyParam: ModifyCycleConfParam, onModifyCycleConfResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("modifyCycleConf")) {
			if (onModifyCycleConfResult) {
				onModifyCycleConfResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("modifyCycleConf");
		this._modifyCycleConfCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("modifyCycleConf");
			if (onModifyCycleConfResult) {
				onModifyCycleConfResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('modifyCycleConf', {"modifyParam":modifyParam}).then((ret: any) => {
				this._inProcessCallbackSet.delete('modifyCycleConf');
				if (onModifyCycleConfResult) {
					onModifyCycleConfResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.modifyCycleConf(modifyParam, this._modifyCycleConfCallback);
	}

	/**
	* [en] [cn]编辑周期子会议
	*/
	modifySubCycleConf(modifySubCycleConfParam: ModifySubCycleConfParam, onModifySubCycleConfResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("modifySubCycleConf")) {
			if (onModifySubCycleConfResult) {
				onModifySubCycleConfResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("modifySubCycleConf");
		this._modifySubCycleConfCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("modifySubCycleConf");
			if (onModifySubCycleConfResult) {
				onModifySubCycleConfResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('modifySubCycleConf', {"modifySubCycleConfParam":modifySubCycleConfParam}).then((ret: any) => {
				this._inProcessCallbackSet.delete('modifySubCycleConf');
				if (onModifySubCycleConfResult) {
					onModifySubCycleConfResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.modifySubCycleConf(modifySubCycleConfParam, this._modifySubCycleConfCallback);
	}

	/**
	* [en] [cn]取消周期会议
	*/
	cancelCycleConf(cancelCycleConfParam: CancelCycleConfParam, onCancelCycleConfResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("cancelCycleConf")) {
			if (onCancelCycleConfResult) {
				onCancelCycleConfResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("cancelCycleConf");
		this._cancelCycleConfCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("cancelCycleConf");
			if (onCancelCycleConfResult) {
				onCancelCycleConfResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('cancelCycleConf', {"cancelCycleConfParam":cancelCycleConfParam}).then((ret: any) => {
				this._inProcessCallbackSet.delete('cancelCycleConf');
				if (onCancelCycleConfResult) {
					onCancelCycleConfResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.cancelCycleConf(cancelCycleConfParam, this._cancelCycleConfCallback);
	}

	/**
	* [en] [cn]取消周期子会议
	*/
	cancelSubCycleConf(cancelSubCycleConfParam: CancelSubCycleConfParam, onCancelSubCycleConfResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("cancelSubCycleConf")) {
			if (onCancelSubCycleConfResult) {
				onCancelSubCycleConfResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("cancelSubCycleConf");
		this._cancelSubCycleConfCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("cancelSubCycleConf");
			if (onCancelSubCycleConfResult) {
				onCancelSubCycleConfResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('cancelSubCycleConf', {"cancelSubCycleConfParam":cancelSubCycleConfParam}).then((ret: any) => {
				this._inProcessCallbackSet.delete('cancelSubCycleConf');
				if (onCancelSubCycleConfResult) {
					onCancelSubCycleConfResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.cancelSubCycleConf(cancelSubCycleConfParam, this._cancelSubCycleConfCallback);
	}

}

