﻿/**
* @file HwmUISDK.h
*
* Copyright(C), 2019-2020, Huawei Tech. Co., Ltd. ALL RIGHTS RESERVED.
*
* @brief [en] Interface file.
*        [cn] 接口文件
*/

#ifndef __HWM_UISDK_H__
#define __HWM_UISDK_H__

#include "HwmUISdkLoginService.h"
#include "HwmUISdkConfService.h"
#include "HwmUISdkConfigService.h"
#include "HwmUISdkCallService.h"
#include "HwmUISdkDef.h"

BEGIN_HWM_UISDK_NAMESPACE

/**
* @brief 
* [en] This class is about UISDK notify callback.
* [cn] UISDK通知
*/
class HWM_API HwmUISDKNotifyCallback
{
public:
    /**
    * @brief 
    * [en] Notify of sdk disconnected.
    * [cn] 连接断开通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示与SDK的连接断开了。
    *
    * @mac:disable
    **/
    virtual HWM_VOID OnSdkDisconnectedNotify() {};
};


/**
* @brief 
* [en]This class is about UISDK
* [cn]UISDK接口管理对象
*
* @GetInstance
*        HWM_UISDK_NAMESPACE::HwmUISDK
*/
class HWM_API HwmUISDK
{
public:
    /**
    * @brief 
    * [en] This interface is used to set the the global callback for the notifications.
    * [cn] 设置通知类全局回调
    *
    * @param callback [type:HwmUISDKNotifyCallback*][cs_allowNull:N]
    * [en]
    * [cn] 通知类全局回调
    **/
    static HWM_VOID SetUISDKNotifyCallback(HwmUISDKNotifyCallback* callback);

    /**
    * @brief 
    * [en] This interface is used to init sdk.
    * [cn] 初始化
    *
    * @detail
    * [en]
    * [cn] 该接口用于启动并且初始化SDK。注意事项：1、调用该接口之前请先修改HwmSdk文件夹下HwmSdk.exe文件的名称；2、该接口用于启动并初始化SDK，在调用其他接口之前必须先调用该接口并等待初始化回调结果；3、每次调用初始化接口会重新启动SDK，之前所有设置与操作将不复存在。
    *
    * @param initInfo [type:HwmInitInfo][cs_allowNull:N]
    * [en] Init info
    * [cn] 初始化信息
    *
    * @synchronized
    **/
    static SDKERR Init(const HwmInitInfo* initInfo);

    /**
    * @brief 
    * [en] This interface is used to exit sdk.
    * [cn] 退出
    *
    * @detail
    * [en]
    * [cn] 该接口用于退出SDK。注意事项：1、该接口用于退出SDK，调用该接口之后SDK不可使用；2、应用退出时请调用该接口，若不调用该接口，则SDK无法退出，应用退出后SDK进程将继续存在；3、mac请在杀进程场景调用。
    * 
    * @synchronized
    **/
    static SDKERR Exit();

    /**
    * @brief 
    * [en] This interface is used to get the login service object.
    * [cn] 获取登录服务对象
    *
    * @param [cs_allowNull:Y]
    *
    * @retval IHwmLogin*        [en] If it's success return the pointer of class IHwmLogin.Otherwise,return HWM_NULL_PTR.
    *                           [cn] 成功返回 IHwmLogin     对象，失败返回 HWM_NULL_PTR
    *
    * @attention [en] The "login service object" is the basic service class object of the SDK, which must be acquired and used by the application.
    *            [cn] "登录服务对象”是SDK的基础服务类对象，应用程序必需获取并使用。
    **/
    static IHwmLogin* GetLoginApi(HWM_VOID);

    /**
    * @brief 
    * [en] This interface is used to get the conf control service object.
    * [cn] 获取会议控制服务对象
    *
    * @param NULL [cs_allowNull:Y]
    *
    * @retval IHwmConfCtrl*        [en] If it's success return the pointer of class IHwmConfCtrl.Otherwise,return HWM_NULL_PTR.
    *                              [cn] 成功返回 IHwmConfCtrl     对象，失败返回 HWM_NULL_PTR
    *
    * @attention [en] When using a conf control service, the application must get and use it.
    *            [cn] 在使用会议控制服务时，应用程序必需获取并使用。
    **/
    static IHwmConfCtrl* GetConfCtrlApi(HWM_VOID);

    /**
    * @brief 
    * [en] This interface is used to get the conf management service object.
    * [cn] 获取会议管理服务对象
    *
    * @param NULL [cs_allowNull:Y]
    *
    * @retval IHwmConfMgr*        [en] If it's success return the pointer of class IHwmConfMgr.Otherwise,return HWM_NULL_PTR.
    *                             [cn] 成功返回 IHwmConfMgr     对象，失败返回 HWM_NULL_PTR
    *
    * @attention [en] When using a conf management service, the application must get and use it.
    *            [cn] 在使用会议管理服务时，应用程序必需获取并使用。
    **/
    static IHwmConfMgr* GetConfMgrApi(HWM_VOID);

    /**
    * @brief 
    * [en] This interface is used to get the config service object.
    * [cn] 获取配置服务对象
    *
    * @param NULL [cs_allowNull:Y]
    *
    * @retval IHwmConfig*        [en] If it's success return the pointer of class IHwmConfig.Otherwise,return HWM_NULL_PTR.
    *                            [cn] 成功返回 IHwmConfig     对象，失败返回 HWM_NULL_PTR
    *
    * @attention [en] When using a config service, the application must get and use it.
    *            [cn] 在使用配置服务时，应用程序必需获取并使用。
    **/
    static IHwmConfig* GetConfigApi(HWM_VOID);

    /**
    * @brief 
    * [en] This interface is used to get the UI config service object.
    * [cn] 获取界面配置服务对象
    *
    * @param NULL [cs_allowNull:Y]
    *
    * @retval IHwmUIConfig*        [en] If it's success return the pointer of class IHwmUIConfig.Otherwise,return HWM_NULL_PTR.
    *                              [cn] 成功返回 IHwmUIConfig     对象，失败返回 HWM_NULL_PTR
    *
    * @attention [en] When using a UI config service, the application must get and use it.
    *            [cn] 在使用界面配置服务时，应用程序必需获取并使用。
    **/
    static IHwmUIConfig* GetUIConfigApi(HWM_VOID);

    /**
    * @brief 
    * [en] This interface is used to get the call service object.
    * [cn] 获取呼叫服务对象
    *
    * @param NULL [cs_allowNull:Y]
    *
    * @retval IHwmCall*        [en] If it's success return the pointer of class IHwmCall.Otherwise,return HWM_NULL_PTR.
    *                          [cn] 成功返回 IHwmCall     对象，失败返回 HWM_NULL_PTR
    *
    * @attention [en] When using a call service, the application must get and use it.
    *            [cn] 在使用呼叫服务时，应用程序必需获取并使用。
    *
    * @doc:disable
    **/
    static IHwmCall* GetCallApi(HWM_VOID);

};

END_HWM_UISDK_NAMESPACE


#endif /* __HWM_UISDK_H__ */