﻿/**
 * @file HwmUISdkConfService.h
 *
 * Copyright(C), 2019-2020, Huawei Tech. Co., Ltd. ALL RIGHTS RESERVED.
 *
 * @brief 会议模块接口文件.
 */

#ifndef __HWM_UISDK_CONF_SERVICE_H__
#define __HWM_UISDK_CONF_SERVICE_H__

#include "HwmUISdkConfServiceDef.h"
#include "HwmGlobalErrorCodeDef.h"
#include "HwmUISdkCallServiceDef.h"

BEGIN_HWM_UISDK_NAMESPACE

/**
* @brief 
* [en] This class is about confctrl result callback.
* [cn] 会议控制操作类
*/
class HWM_API IHwmConfCtrlResultCallback
{
public:
    /**
    * @brief 
    * [en] Callback of CreateConf interface.
    * [cn] 创建会议回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param createConfResult [type:HwmCreateConfResult][cs_allowNull:N]
    * [en] Create conf result
    * [cn] 创建会议结果
    **/
    virtual HWM_VOID OnCreateConfResult(SDKERR ret, const HWM_CHAR* reason, const HwmCreateConfResult* createConfResult) {};

    /**
    * @brief [en] Callback of StartP2pConf interface.
    *        [cn] 发起立即会议回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param createConfResult [type:HwmCreateConfResult][cs_allowNull:N]
    * [en] Create conf result
    * [cn] 创建会议结果
    **/
    virtual HWM_VOID OnStartP2pConfResult(SDKERR ret, const HWM_CHAR* reason, const HwmCreateConfResult* createConfResult) {};

    /**
    * @brief
    * [en] Callback of JoinConfById interface.
    * [cn] 通过会议id入会回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnJoinConfByIdResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of JoinConfByRandom interface.
    * [cn] 通过会议random入会回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnJoinConfByRandomResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of AddAttendee interface.
    * [cn] 添加与会者回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnAddAttendeeResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of LeaveConf interface.
    * [cn] 离开会议回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnLeaveConfResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of EndConf interface.
    * [cn] 结束会议回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnEndConfResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of StartShare interface.
    * [cn] 开启共享回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnStartShareResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of StopShare interface.
    * [cn] 停止共享回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnStopShareResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en] Callback of GetAttendeeList interface.
    * [cn] 获取与会者列表接口回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param attendeeList [type:HwmAttendeeListInfo][cs_allowNull:N]
    * [en] Attendee list
    * [cn] 与会者列表
    **/
    virtual HWM_VOID OnGetAttendeeListResult(SDKERR ret, const HWM_CHAR* reason, const HwmAttendeeListInfo* attendeeList) {};

    /**
    * @brief 
    * [en] Callback of GetAudienceList interface.
    * [cn] 获取观众列表接口回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param audidenceList [type:HwmAttendeeListInfo]
    * [en] Audience list
    * [cn] 观众列表
    **/
    virtual HWM_VOID OnGetAudienceListResult(SDKERR ret, const HWM_CHAR* reason, const HwmAttendeeListInfo* audidenceList) {};

    /**
    * @brief 
    * [en] Callback of GetUserState interface.
    * [cn] 获取用户状态结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param userState [type:HwmUserState]
    * [en] User state
    * [cn] 用户当前的状态,判断当前用户是否处于忙碌状态
    **/
    virtual HWM_VOID OnGetUserStateResult(SDKERR ret, const HWM_CHAR* reason, HwmUserState userState) {};

    /**
    * @brief 
    * [en] Callback of GetSelfRole interface.
    * [cn] 获取自己会中角色结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param selfRole [type:HwmConfRole]
    * [en] Self role in conf
    * [cn] 自己在会议中的角色
    **/
    virtual HWM_VOID OnGetSelfRoleResult(SDKERR ret, const HWM_CHAR* reason, HwmConfRole selfRole) {};

    /**
    * @brief 
    * [en] Callback of RenameSelf interface.
    * [cn] 设置会场名回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual void OnRenameSelfResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en] Callback of GeneralWatch interface.
    * [cn] 选看接口回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnGeneralWatchResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en] Callback of SetAnnotationWindow interface.
    * [cn] 设置标注窗口接口回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetAnnotationWindowResult(SDKERR ret, const HWM_CHAR* reason) {};
   
    /**
    * @brief
    * [en] Callback of StartAnnotation interface.
    * [cn] 开启标注接口回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnStartAnnotationResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en] Callback of SaveAnnotation interface.
    * [cn] 保存标注接口回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSaveAnnotationResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en] Callback of StopAnnotation interface.
    * [cn] 停止标注接口回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnStopAnnotationResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief [en] Callback of StartProjection interface.
    *        [cn] 发起投屏回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param startProjectionResult [type:HwmStartProjectionResult]
    * [en] Start projection result
    * [cn] 发起投屏结果
    **/
    virtual HWM_VOID OnStartProjectionResult(SDKERR ret, const HWM_CHAR* reason, const HwmStartProjectionResult* startProjectionResult) {};

    /**
    * @brief
    * [en] Callback of AcceptConf interface.
    * [cn] 接受会议邀请回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    */
    virtual HWM_VOID OnAcceptConfResult(SDKERR ret, const HWM_CHAR* reason) = 0;

    /**
    * @brief
    * [en] Callback of RejectConf interface.
    * [cn] 拒绝会议邀请回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    */
    virtual HWM_VOID OnRejectConfResult(SDKERR ret, const HWM_CHAR* reason) = 0;

    /**
    * @brief
    * [en] Callback of GetSelfShareState interface.
    * [cn] 获取自己的开启共享状态接口回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param selfShareStateInfo [type:HwmSelfShareStateInfo]
    * [en] Self share state info
    * [cn] 自己开启共享的状态信息
    **/
    virtual HWM_VOID OnGetSelfShareStateResult(SDKERR ret, const HWM_CHAR* reason, const HwmSelfShareStateInfo* selfShareStateInfo) {};

    /**
    * @brief
    * [en] Callback of OpenSubtitle interface.
    * [cn] 开启/关闭字幕回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param isOpen [type:HWM_BOOL]
    * [en] Is open
    * [cn] 是否开启字幕
    */
    virtual HWM_VOID OnOpenSubtitleResult(SDKERR ret, const HWM_CHAR* reason, HWM_BOOL isOpen) {};
    
    /**
    * @brief
    * [en] Callback of OpenWaitingRoom interface.
    * [cn] 开启/关闭等候室回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param isOpen [type:HWM_BOOL]
    * [en] Is open
    * [cn] 是否开启等候室
    */
    virtual HWM_VOID OnOpenWaitingRoomResult(SDKERR ret, const HWM_CHAR* reason, HWM_BOOL isOpen) {};

    /**
    * @brief
    * [en] Callback of OperateLocalRecord interface.
    * [cn] 操作本地录制回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    */
    virtual HWM_VOID OnOperateLocalRecordResult(SDKERR ret, const HWM_CHAR* reason, HwmLocalRecordOperateType type) {};

    /**
    * @brief
    * [en] Callback of OpenVirtualBackgroundSettingPage interface.
    * [cn] 打开虚拟背景设置回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    */
    virtual HWM_VOID OnOpenVirtualBackgroundSettingPageResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en] Callback of GetConfLocalRecordStatus interface.
    * [cn] 获取本地录制状态结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param state [type:HwmLocalRecordStatusInfo]
    * [en] Local record status
    * [cn] 本地录制状态
    **/
    virtual HWM_VOID OnGetConfLocalRecordStatusResult(SDKERR ret, const HWM_CHAR* reason, const HwmLocalRecordStatusInfo* state) {};

    /**
    * @brief
    * [en] Callback of GetConfLocalRecordPermission interface.
    * [cn] 获取本地录制权限结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param permission [type:HwmLocalRecordPermissionInfo]
    * [en] Local record permission
    * [cn] 本地录制状态
    **/
    virtual HWM_VOID OnGetSelfHasLocalRecordPermissionResult(SDKERR ret, const HWM_CHAR* reason, const HwmLocalRecordPermissionInfo* permission) {};

    /**
    * @brief
    * [en] Callback of RequestLocalRecordPermission interface.
    * [cn] 申请本地录制权限回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    */
    virtual HWM_VOID OnRequestLocalRecordPermissionResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en] Callback of ApplyHostRole interface.
    * [cn] 申请主持人回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    */
    virtual HWM_VOID OnApplyHostRoleResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en] Callback of GetSubtitlePageOpened interface.
    * [cn] 获取字幕状态回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param isOpen [type:HWM_BOOL]
    * [en] Is open
    * [cn] 是否开启
    */
    virtual HWM_VOID OnGetSubtitlePageOpenedResult(SDKERR ret, const HWM_CHAR* reason, HWM_BOOL isOpen) {};

    /**
    * @brief
    * [en] Callback of GetWaitingRoomOpened interface.
    * [cn] 获取等候室状态回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param isOpen [type:HWM_BOOL]
    * [en] Is open
    * [cn] 是否开启
    */
    virtual HWM_VOID OnGetWaitingRoomOpenedResult(SDKERR ret, const HWM_CHAR* reason, HWM_BOOL isOpen) {};

    /**
    * @brief
    * [en] Callback of GetVirtualBackgroundSettingPageOpened interface.
    * [cn] 获取虚拟背景设置界面状态回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param isOpen [type:HWM_BOOL]
    * [en] Is open
    * [cn] 是否开启
    */
    virtual HWM_VOID OnGetVirtualBackgroundSettingPageOpenedResult(SDKERR ret, const HWM_CHAR* reason, HWM_BOOL isOpen) {};

    /**
    * @brief
    * [en] Callback of SubscribeSubtitleLanguage interface.
    * [cn] 订阅字幕语言
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param targetLanguageType [type:HwmTargetLanguageType]
    * [en] Target Language Type
    * [cn] 目标语言类型
    */
    virtual HWM_VOID OnSubscribeSubtitleLanguageResult(SDKERR ret, const HWM_CHAR* reason, HwmTargetLanguageType targetLanguageType) {};

    /**
    * @brief
    * [en] Callback of SubscribeSubtitle interface.
    * [cn] 订阅/取消订阅字幕
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param isSubscirbe [type:HWM_BOOL]
    * [en] Is Subscirbe
    * [cn] 是否订阅
    */
    virtual HWM_VOID OnSubscribeSubtitleResult(SDKERR ret, const HWM_CHAR* reason, HWM_BOOL isSubscirbe) {};

    /**
    * @brief
    * [en] Callback of GetSubtitleSupportLanguage interface.
    * [cn] 获取字幕语言列表
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param targetLanguageTypeList [type:HwmTargetLanguageTypeList*]
    * [en] Target Language Type List
    * [cn] 目标语言类型列表
    */
    virtual HWM_VOID OnGetSubtitleSupportLanguageResult(SDKERR ret, const HWM_CHAR* reason, const HwmTargetLanguageTypeList* targetLanguageTypeList) {};

};

/**
* @brief 
* [en] This class is about confctrl notify callback.
* [cn] 会议控制通知回调
*/
class HWM_API IHwmConfCtrlNotifyCallback
{
public:

    /**
    * @brief 
    * [en] Notify of conf state.
    * [cn] 会议状态通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示会议状态发生变更。
    *
    * @param confStateInfo [type:HwmConfStateInfo*]
    * [en] Conf state info
    * [cn] 会议状态信息
    **/
    virtual HWM_VOID OnConfStateChanged(const HwmConfStateInfo *confStateInfo) {};

    /**
    * @brief
    * [en] This class is about the notification of an ended conference.
    * [cn] 会议结束通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示会议已结束。
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param confEndInfo [type:HwmConfEndInfo*]
    * [en] Conf end info
    * [cn] 已结束的会议信息
    */
    virtual HWM_VOID OnConfEndedNotify(const HwmConfEndInfo* confEndInfo) {};

    /**
    * @brief
    * [en] This class is about the callback of transfering point to point call to conference.
    * [cn] 立即会议记录回调
    *
    * @param recordInfo [type:HwmP2PConfRecordInfo*]
    * [en] Conf record info
    * [cn] 会议记录信息
    */
    virtual HWM_VOID OnP2PConfRecordNotify(HwmP2PConfRecordInfo *recordInfo) {};
    
    /**
    * @brief 
    * [en] Notify of conf info.
    * [cn] 会议信息通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示会议中的信息发生变更。
    *
    * @param confInfo [type:HwmConfInfo*]
    * [en] Conf info
    * [cn] 会议信息
    **/
    virtual HWM_VOID OnConfInfoNotify(const HwmConfInfo* confInfo) {};

    /**
    * @brief 
    * [en] Notify of share state.
    * [cn] 共享状态通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示当前共享状态发生变更。
    *
    * @param shareStateInfo [type:HwmShareStateInfo*]
    * [en] Share state info
    * [cn] 共享状态信息
    *
    **/
    virtual HWM_VOID OnShareStateChanged(const HwmShareStateInfo* shareStateInfo) {};

    /**
    * @brief 
    * [en] Notify of receive share state.
    * [cn] 收到共享状态通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示当前接收共享的状态发生变更。
    *
    * @param shareStateInfo [type:HwmRecvShareStateInfo*]
    * [en] Recv share state info
    * [cn] 收到共享状态信息
    *
    * @mac:disable
    * @doc:disable
    **/
    virtual HWM_VOID OnRecvShareStateChanged(const HwmRecvShareStateInfo* shareStateInfo) {};

    /**
    * @brief 
    * [en] Notify of remote control state.
    * [cn] 得到远程控制状态通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示当前远程控制权限状态发生变化：state为1时，从name处获得远程控制权限，state为0时，name的远程控制权限被释放、收回。
    *
    * @param remoteControlStateInfo [type:HwmRemoteControlStateInfo*]
    * [en] Remote control state
    * [cn] 远程控制状态信息
    *
    * @mac:disable
    * @js:disable
    **/
    virtual HWM_VOID OnRemoteControlStateChanged(const HwmRemoteControlStateInfo* remoteControlStateInfo) {};

    /**
    * @brief 
    * [en] Notify of be remote controled state.
    * [cn] 授予他人远程控制状态通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示当前授予他人远程控制权限状态发生变化：state为1时，把远程控制权限授予给name，state为0时，name的远程控制权限被释放、收回。
    *
    * @param remoteControlStateInfo [type:HwmRemoteControlStateInfo*]
    * [en] Be remote controled state
    * [cn] 被他人远程控制状态信息
    *
    * @mac:disable
    * @js:disable
    **/
    virtual HWM_VOID OnBeRemoteControledStateChanged(const HwmRemoteControlStateInfo* remoteControlStateInfo) {};

    /**
    * @brief 
    * [en] Self role changed.
    * [cn] 个人角色变化通知
    *
    * @detail
    * [en] (only for mac)
    * [cn] 收到该通知时，表示个人角色发生变化(仅对mac平台生效)。
    *
    * @param role [type:HwmConfRole]
    * [en]self role in conf
    * [cn]自己在会议中的角色
    *
    * @win:todo
    **/
    virtual HWM_VOID OnSelfRoleChanged(HwmConfRole role) {};

    /**
    * @brief 
    * [en] Notify of conf share locked state changed
    * [cn] 会议是否共享锁定状态变化通知
    *
    * @detail
    * [en] (only for windows)
    * [cn] 收到该通知时，表示会议共享锁定状态发生变化(仅对windwos平台生效)。
    *
    * @param isShareLocked [type:HWM_BOOL]
    * [en] Is share locked
    * [cn] 是否共享锁定
    *
    * @mac:disable
    * @js:disable
    */
    virtual HWM_VOID OnConfIsShareLockedChanged(HWM_BOOL isShareLocked) {};

    /**
    * @brief
    * [en] This class is about the notification of low quality network.
    * [cn] 弱网提示通知
    *
    * @param poorNetWorkInfo [type:HwmPoorNetWorkQualityInfo*]
    * [en] Weak Network Type
    * [cn] 弱网类型
    */
    virtual HWM_VOID OnPoorNetworkQualityInfoNotify(const HwmPoorNetWorkQualityInfo* poorNetWorkInfo) {};

    /**
    * @brief
    * [en]Notify of speaker list
    * [cn]发言人变更的通知
    *
    * @param speakerList [type:HwmSpeakerList*]
    * [en]Speaker list
    * [cn]发言人列表
    *
    * @js:disable
    */
    virtual HWM_VOID OnSpeakerListChanged(const HwmSpeakerList* speakerList) {};

    /**
    * @brief
    * [en]Notify of video stream type info
    * [cn]视频流类型的变化通知
    *
    * @param videoStreamTypeInfo [type:HwmVideoStreamTypeInfo*]
    * [en]video stream type info
    * [cn]视频流类型信息
    *
    * @js:disable
    */
    virtual HWM_VOID OnVideoStreamTypeChanged(const HwmVideoStreamTypeInfo* videoStreamTypeInfo) {};

    /**
    * @brief
    * [en]Notify of video attendee list change
    * [cn]视频与会者列表变更的通知
    *
    * @js:disable
    */
    virtual HWM_VOID OnVideoAttendeeListChanged() {};

    /**
    * @brief
    * [en]Notify of annotation state change
    * [cn]标注状态变化通知
    *
    * @param stateInfo [type:HwmAnnotationStateInfo*]
    * [en]Annotation state info
    * [cn]标注状态信息
    *
    * @js:disable
    */
    virtual HWM_VOID OnAnnotationStateChanged(const HwmAnnotationStateInfo* stateInfo){};

    /**
    * @brief
    * [en]Notify of conf info
    * [cn]会中信息更新通知
    *
    * @param meetingInfo [type:HwmMeetingInfo*]
    * [en]Conf info
    * [cn]会中信息
    */
    virtual HWM_VOID OnMeetingInfoChanged(const HwmMeetingInfo* meetingInfo) {};

    /**
    * @brief
    * [en] Notify of the waiting list
    * [cn] 等候室成员列表变更的通知
    *
    * @param waitingList [type:HwmAttendeeListInfo*]
    * [en] Waiting list
    * [cn] 等候室成员列表
    **/
    virtual HWM_VOID OnWaitingListChanged(const HwmAttendeeListInfo* waitingList) {};

    /**
    * @brief
    * [en] Notify of the add attendee
    * [cn] 添加与会者结果通知
    *
    * @detail
    * [en]
    * [cn] 添加多人时，可能会上报多次。
    *
    * @param addAttendeeResultList [type:HwmAddAttendeeResultList*]
    * [en] add attendee result list
    * [cn] 添加与会者结果列表
    **/
    virtual HWM_VOID OnAddAttendeeNotify(const HwmAddAttendeeResultList* addAttendeeResultList) {};

    /**
    * @brief
    * [en] Notify of local record state
    * [cn] 本地录制状态通知
    *
    * @param localRecordStateInfo [type:HwmLocalRecordStateInfo*]
    * [en] Local record state info
    * [cn] 本地录制状态信息
    */
    virtual HWM_VOID OnSelfLocalRecordStateNotify(const HwmLocalRecordStateInfo* localRecordStateInfo) {};

    /**
    * @brief
    * [en] Notify of the attendee list
    * [cn] 与会者列表变更的通知
    *
    * @param attendeeList [type:HwmAttendeeListInfo*]
    * [en] Attendee list
    * [cn] 与会者列表
    **/
    virtual HWM_VOID OnAttendeeListChanged(const HwmAttendeeListInfo* attendeeList) {};

    /**
    * @brief
    * [en] Notify of the real time subtitle info
    * [cn] 实时字幕信息的通知
    *
    * @param realTimeSubtitleInfo [type:HwmRealTimeSubtitleInfo*]
    * [en] Real Time Subtitle Info
    * [cn] 实时字幕信息
    **/
    virtual HWM_VOID OnRealTimeSubtitleInfoNotify(const HwmRealTimeSubtitleInfo* realTimeSubtitleInfo) {};

    /**
    * @brief
    * [en] Notify of the subtitle record info
    * [cn] 字幕记录消息的通知
    *
    * @param subtitleRecordInfo [type:HwmSubtitleRecordInfo*]
    * [en] Subtitle Record Info
    * [cn] 字幕记录信息
    **/
    virtual HWM_VOID OnSubtitleRecordNotify(const HwmSubtitleRecordInfo* subtitleRecordInfo) {};

    /**
    * @brief
    * [en] Notify of subtitle state change
    * [cn] 字幕状态变更通知
    *
    * @detail
    * [en] Notify of subtitle state change
    * [cn] 收到该通知时，表示当前字幕状态发生变化。
    *
    * @param isOpen [type:HWM_BOOL]
    * [en] Is open
    * [cn] 是否开启
    **/
    virtual HWM_VOID OnSubtitleStateChanged(HWM_BOOL isOpen) {};

    /**
    * @brief
    * [en] Notify of waiting room state change
    * [cn] 等候室状态变更通知
    *
    * @detail
    * [en] Notify of waiting room state change
    * [cn] 收到该通知时，表示当前等候室状态发生变化。
    *
    * @param isOpen [type:HWM_BOOL]
    * [en] Is open
    * [cn] 是否开启
    **/
    virtual HWM_VOID OnWaitingRoomStateChanged(HWM_BOOL isOpen) {};

    /**
    * @brief
    * [en] Notify of local record operation interrupt notify
    * [cn] 本地录制操作中断通知
    *
    * @detail
    * [en] Notify of waiting room state change
    * [cn] 收到该通知时，表示当前本地录制操作中断。
    *
    * @param info [type:HwmInterruptInfo]
    * [en] Interrupt info
    * [cn] 中断信息
    **/
    virtual HWM_VOID OnStartLocalRecordOperationInterruptNotify(const HwmInterruptInfo* info) {};

    /**
    * @brief
    * [en]Notify of forbidden screenshots enabletype changed
    * [cn]禁止截屏开关状态变更
    *
    * @param enableType [type:HwmForbiddenScreenShotsEnableType]
    * [en]
    * [cn]禁止截屏开关状态
    *
    * @mac:disable
    */
    virtual HWM_VOID OnConfForbiddenScreenShotsEnableChanged(HwmForbiddenScreenShotsEnableType enableType) {};
};


/**
* @brief 
* [en] This class is about the object of conference control operation.
* [cn] 会议控制操作对象
*
* @detail
* [en]
* [cn] 获取该对象的方法HwmUISDK.getConfCtrlApi()
*
* @GetInstance
*        HWM_UISDK_NAMESPACE::HwmUISDK::GetConfCtrlApi()
*
*/
class IHwmConfCtrl
{
public:

    /**
    * @brief
    * [en] This class is about the global callback for the operating results of setting the conference control.
    * [cn] 设置会议控制操作类全局回调
    *
    * @param callback [type:IHwmConfCtrlResultCallback][cs_allowNull:N]
    * [en]
    * [cn] 会议控制操作类全局回调
    */
    virtual HWM_VOID SetConfCtrlResultCallback(IHwmConfCtrlResultCallback* callback) = 0;

    /**
    * @brief
    * [en] This class is about the global callback for the notifications of conference control.
    * [cn] 设置会议控制通知类全局回调
    *
    * @param callback [type:IHwmConfCtrlNotifyCallback][cs_allowNull:N]
    * [en]
    * [cn] 会议控制通知类全局回调
    */
    virtual HWM_VOID SetConfCtrlNotifyCallback(IHwmConfCtrlNotifyCallback* callback) = 0;

    /**
    * @brief 
    * [en] This interface is used to create conf.
    * [cn] 创建会议
    *
    * @detail
    * [en]
    * [cn] 该接口用于创建即时会议。注意事项：1、该接口默认已将自己作为主持人加入会议，调用接口时，在与会者参数中不需要带上自己的信息；
           2、若需要在创建会议的同时邀请其他与会者，需要传入其他与会者信息。若不需要，与会者信息和与会者个数可以为空。
    *
    * @param createConfInfo [type:HwmCreateConfInfo*][cs_allowNull:N]
    * [en] Create conf info
    * [cn] 创会信息
    **/
    virtual SDKERR CreateConf(const HwmCreateConfInfo* createConfInfo) = 0;

    /**
    * @brief [en] This interface is used to start p2p conf.
    *        [cn] 发起立即会议
    *
    * @param startCallInfo [type:HwmStartCallInfo*][cs_allowNull:N]
    * [en] Create p2p conf info
    * [cn] 创建立即会议信息
    **/
    virtual SDKERR StartP2pConf(const HwmStartCallInfo* startCallInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to join conf by id.
    * [cn] 加入会议
    *
    * @detail
    * [en]
    * [cn] 该接口用于通过会议ID和密码加入已经存在的会议。在登录和未登录状态下都可以调用该接口。
    *
    * @param joinConfByIdInfo [type:HwmJoinConfByIdInfo*][cs_allowNull:N]
    * [en] Join conf by Id info
    * [cn] id入会信息
    **/
    virtual SDKERR JoinConfById(const HwmJoinConfByIdInfo* joinConfByIdInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to join conf by random.
    * [cn] Random入会
    *
    * @detail
    * [en]
    * [cn] 通过邮件会议通知中的会议链接入会时（会议链接如：https://bmeeting.huaweicloud.com/#/j/981924330/e11ddec62ee760734fcf2ba55b79937eac2aa68677caf659），
           链接里面的是会议ID（即举例中的981924330）和会议的Random码（即举例中的e11ddec62ee760734fcf2ba55b79937eac2aa68677caf659）。第三方App需要调用本接口加入会议。
    *
    * @param joinConfByRandomInfo [type:HwmJoinConfByRandomInfo*][cs_allowNull:N]
    * [en] Join conf by random info
    * [cn] random入会信息
    **/
    virtual SDKERR JoinConfByRandom(const HwmJoinConfByRandomInfo* joinConfByRandomInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to add attendee.
    * [cn] 添加与会者
    *
    * @detail
    * [en]
    * [cn] 该接口用于在会议或呼叫中添加与会者。注意事项：1、不在会议或呼叫中，调用该接口将返回失败；2、呼叫中调用该接口，呼叫则转成会议。
    *
    * @param addAttendeeInfo [type:HwmAddAttendeeInfo*][cs_allowNull:N]
    * [en] Add attendee info
    * [cn] 添加与会者信息
    **/
    virtual SDKERR AddAttendee(const HwmAddAttendeeInfo* addAttendeeInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to leave conf.
    * [cn] 离开会议
    *
    * @detail
    * [en]
    * [cn] 该接口用于离开会议。注意事项：1、不在会议中不应调用此接口；2、调用该接口离开会议，会议不会结束，在会议存在期间可以再次入会。
    *
    * @param [cs_allowNull:Y]
    **/
    virtual SDKERR LeaveConf() = 0;

    /**
    * @brief 
    * [en] This interface is used to end conf.
    * [cn] 结束会议
    *
    * @detail
    * [en]
    * [cn] 该接口用于结束会议。注意事项：1、不在会议中不应调用此接口；2、调用该接口，会议将结束，无法再次加入该会议。
    *
    * @param [cs_allowNull:Y]
    **/
    virtual SDKERR EndConf() = 0;

    /**
    * @brief 
    * [en] This interface is used to start share.
    * [cn] 开启共享
    *
    * @detail
    * [en]
    * [cn] 该接口用于开启会议共享。SDK的UI已经实现了“共享”功能，如果第三方App不需要通过程序控制开启共享，则不用调用该接口。注意事项：1、不在会议或呼叫中调用该接口将返回失败；2、呼叫中调用该接口，呼叫转成会议。
    *
    * @param startShareInfo [type:HwmStartShareInfo*][cs_allowNull:N]
    * [en] Start share info
    * [cn] 开启共享参数    
    **/
    virtual SDKERR StartShare(const HwmStartShareInfo* startShareInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to stop share.
    * [cn] 结束共享
    *
    * @detail
    * [en]
    * [cn] 该接口用于结束会中共享。SDK的UI已经实现了“停止共享”功能，如果第三方App不需要通过程序控制停止共享，则不用调用该接口。注意事项：1、不在共享中不应调用此接口。
    *
    * @param [cs_allowNull:Y]
    *
    * @mac:transToSync
    **/
    virtual SDKERR StopShare() = 0;

    /**
    * @brief 
    * [en] This interface is used to get attendee List.
    * [cn] 获取与会者列表
    *
    * @detail
    * [en]
    * [cn] 该接口用于获取会议中与会者列表。注意事项：1、只能在会议中使用。
    *
    * @param [cs_allowNull:Y]
    *
    * @mac:transToSync
    **/
    virtual SDKERR GetAttendeeList() = 0;

    /**
    * @brief 
    * [en] This interface is used to get audience List.
    * [cn] 获取观众列表
    *
    * @detail
    * [en]
    * [cn] 该接口用于获取会议中观众列表。注意事项：1、只能在会议中使用；2、观众身份（可以通过会议详情消息推送获取）不能使用此接口。
    *
    * @param [cs_allowNull:Y]
    *
    * @mac:transToSync
    **/
    virtual SDKERR GetAudienceList() = 0;

    /**
    * @brief 
    * [en] This interface is used to get user state.
    * [cn] 获取用户状态
    *
    * @detail
    * [en]
    * [cn] 该接口用于获取用户当前的状态(是否在呼叫/会议中)。
    *
    * @param [cs_allowNull:Y]
    *
    * @mac:transToSync
    **/
    virtual SDKERR GetUserState() = 0;

    /**
    * @brief 
    * [en] This interface is used to get self role.
    * [cn] 获取自己会中角色
    *
    * @detail
    * [en] (only for mac)
    * [cn] 该接口用于获取自己会中角色(仅对mac平台生效)
    *
    * @param [cs_allowNull:Y]
    *
    * @win:todo
    * @mac:transToSync
    **/
    virtual SDKERR GetSelfRole() = 0;

    /**
    * @brief 
    * [en] This interface is used to rename self.
    * [cn] 设置会场名
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置会场名。
    *
    * @param renameSelfInfo [type:HwmRenameSelfInfo*][cs_allowNull:N]
    * [en] Set self name info
    * [cn] 设置自己会场名信息
    **/
    virtual SDKERR RenameSelf(const HwmRenameSelfInfo* renameSelfInfo) = 0;

    /**
    * @brief
    * [en]This interface is about general watch
    * [cn]选看
    *
    * @param watchListInfo [type:HwmWatchListInfo*][cs_allowNull:N]
    * [en]
    * [cn]选看列表
    *
    * @js:disable
    * @mac:transToSync
    */
    virtual SDKERR GeneralWatch(const HwmWatchListInfo* watchListInfo) = 0;

    /**
    * @brief
    * [en]This interface is about set annotation window
    * [cn]设置标注窗口
    *
    * @param annoWndInfo [type:HwmAnnotationWndInfo*][cs_allowNull:N]
    * [en]
    * [cn]标注窗口信息
    *
    * @js:disable
    * @mac:transToSync
    */
    virtual SDKERR SetAnnotationWindow(const HwmAnnotationWndInfo* annoWndInfo) = 0;

    /**
    * @brief
    * [en]This interface is about start annotation
    * [cn]开启标注
    *
    * @param annoWndInfo [type:HwmAnnotationWndInfo*][cs_allowNull:N]
    * [en]
    * [cn]标注窗口信息
    *
    * @js:disable
    * @mac:transToSync
    */
    virtual SDKERR StartAnnotation(const HwmAnnotationWndInfo* annoWndInfo) = 0;

    /**
    * @brief
    * [en]This interface is about save annotation
    * [cn]保存标注
    *
    * @param configInfo [type:HwmAnnotationSaveConfigInfo*][cs_allowNull:N]
    * [en]
    * [cn]标注保存配置信息
    *
    * @js:disable
    * @mac:transToSync
    */
    virtual SDKERR SaveAnnotation(const HwmAnnotationSaveConfigInfo* configInfo) = 0;

    /**
    * @brief
    * [en]This interface is about stop annotation
    * [cn]停止标注
    *
    * @js:disable
    * @mac:disable
    */
    virtual SDKERR StopAnnotation() = 0;

    /**
    * @brief
    * [en] This interface is used to start projection.
    * [cn] 发起投屏
    *
    * @detail
    * [en] This interface is used to start projection.
    * [cn] 该接口用于发起投屏。注意事项：1、该接口在已登录状态下才能调用。
    *
    * @param startProjectionParam [type:HwmStartProjectionParam*][cs_allowNull:N]
    * [en] Start projection param
    * [cn] 发起投屏的参数体
    **/
    virtual SDKERR StartProjection(const HwmStartProjectionParam* startProjectionParam) = 0;

    /**
    * @brief
    * [en] This interface is used to set auto hide tool bar.
    * [cn] 设置会控栏自动隐藏
    *
    * @detail
    * [en] This interface is used to set auto hide tool bar.
    * [cn] 该接口用于设置会控栏是否自动隐藏。注意事项：1、当次会议生效，离会后恢复默认体验，即鼠标无操作后一段时间后自动隐藏会控栏。
    *
    * @param autoHideToolBarParam [type:HwmAutoHideToolBarParam*][cs_allowNull:N]
    * [en] Auto Hide ToolBar param
    * [cn] 自动隐藏会控栏参数体
    *
    * @synchronized
    * @mac:disable
    **/
    virtual SDKERR SetAutoHideToolBar(const HwmAutoHideToolBarParam* autoHideToolBarParam) = 0;

    /**
    * @brief
    * [en] This interface is used to accept conf.
    * [cn] 接受会议邀请
    *
    * @detail
    * [en] This interface is used to accept conf.
    * [cn] 该接口用于接受会议邀请
    *
    * @param acceptConfParam [type:HwmAcceptConfParam*]
    * [en] accept conf param
    * [cn] 接受会议邀请的参数体
    *
    */
    virtual SDKERR AcceptConf(const HwmAcceptConfParam* acceptConfParam) { return SDKERR::UISDK_NOT_SUPPORT_END; }

    /**
    * @brief
    * [en] This interface is used to reject conf.
    * [cn] 拒绝会议邀请
    *
    * @detail
    * [en] This interface is used to reject conf.
    * [cn] 该接口用于拒绝会议邀请
    *
    */
    virtual SDKERR RejectConf() { return SDKERR::UISDK_NOT_SUPPORT_END; }

    /**
    * @brief
    * [en] This interface is used to get self share state.
    * [cn] 获取自己的开启共享状态
    *
    * @detail
    * [en] This interface is used to get self share state.
    * [cn] 获取自己的开启共享状态
    *
    * @mac:transToSync
    **/
    virtual SDKERR GetSelfShareState() = 0;

    /**
    * @brief
    * [en] This interface is used to operate subtitle.
    * [cn] 开启/关闭字幕
    *
    * @detail
    * [en] This interface is used to operate subtitle.
    * [cn] 该接口用于开启/关闭字幕
    *
    * @param isOpen [type:HWM_BOOL]
    * [en] isOpen
    * [cn] 是否开启
    */
    virtual SDKERR OpenSubtitle(HWM_BOOL isOpen) = 0;

    /**
    * @brief
    * [en] This interface is used to operate waiting room.
    * [cn] 开启/关闭等候室
    *
    * @detail
    * [en] This interface is used to operate waiting room.
    * [cn] 该接口用于开启/关闭等候室
    *
    * @param isOpen [type:HWM_BOOL]
    * [en] isOpen
    * [cn] 是否开启
    */
    virtual SDKERR OpenWaitingRoom(HWM_BOOL isOpen) = 0;

    /**
    * @brief
    * [en] This interface is used to operate local record.
    * [cn] 操作本地录制
    *
    * @detail
    * [en] This interface is used to operate local record.
    * [cn] 该接口用于操作本地录制
    *
    * @param type [type:HwmLocalRecordOperateType]
    * [en] type
    * [cn] 本地录制操作类型
    */
    virtual SDKERR OperateLocalRecord(HwmLocalRecordOperateType type) = 0;

    /**
    * @brief
    * [en] This interface is used to open virtual background setting page.
    * [cn] 打开虚拟背景设置
    *
    * @detail
    * [en] This interface is used to open virtual background setting page.
    * [cn] 该接口用于打开虚拟背景设置
    */
    virtual SDKERR OpenVirtualBackgroundSettingPage() = 0;

    /**
    * @brief
    * [en] This interface is used to get local record status.
    * [cn] 获取本地录制状态
    *
    * @detail
    * [en] This interface is used to get local record status.
    * [cn] 该接口用于获取本地录制状态
    */
    virtual SDKERR GetConfLocalRecordStatus() = 0;

    /**
    * @brief
    * [en] This interface is used to get local record permission.
    * [cn] 获取本地录制权限
    *
    * @detail
    * [en] This interface is used to get local record permission.
    * [cn] 该接口用于获取本地录制权限
    */
    virtual SDKERR GetSelfHasLocalRecordPermission() = 0;

    /**
    * @brief
    * [en] This interface is used to request local record permission.
    * [cn] 申请本地录制权限
    *
    * @detail
    * [en] This interface is used to request local record permission.
    * [cn] 该接口用于申请本地录制权限
    */
    virtual SDKERR RequestLocalRecordPermission() = 0;

    /**
    * @brief
    * [en] This interface is used to request host role.
    * [cn] 申请主持人
    *
    * @detail
    * [en] This interface is used to request host role.
    * [cn] 该接口用于申请主持人
    *
    * @param password [type:HWM_CHAR*]
    * [en] password
    * [cn] 主持人密码
    */
    virtual SDKERR ApplyHostRole(const HWM_CHAR* password) = 0;

    /**
    * @brief
    * [en] This interface is used to get subtitle state.
    * [cn] 获取字幕状态
    *
    * @detail
    * [en] This interface is used to get subtitle state.
    * [cn] 获取字幕状态
    *
    * @mac:transToSync
    **/
    virtual SDKERR GetSubtitlePageOpened() = 0;

    /**
    * @brief
    * [en] This interface is used to get waiting room state.
    * [cn] 获取等候室状态
    *
    * @detail
    * [en] This interface is used to get waiting room state.
    * [cn] 获取等候室状态
    *
    * @mac:transToSync
    **/
    virtual SDKERR GetWaitingRoomOpened() = 0;

    /**
    * @brief
    * [en] This interface is used to get virtual background setting page state.
    * [cn] 获取是否已经打开虚拟背景设置
    *
    * @detail
    * [en] This interface is used to get virtual background setting page state.
    * [cn] 获取是否已经打开虚拟背景设置
    *
    * @mac:transToSync
    **/
    virtual SDKERR GetVirtualBackgroundSettingPageOpened() = 0;

    /**
    * @brief
    * [en] This interface is used to subscribe subtitle language.
    * [cn] 订阅字幕语言
    *
    * @detail
    * [en] This interface is used to subscribe subtitle languague.
    * [cn] 该接口用于订阅字幕语言
    */
    virtual SDKERR SubscribeSubtitleLanguage(HwmTargetLanguageType targetLanguageType) = 0;

    /**
    * @brief
    * [en] This interface is used to subscribe subtitle.
    * [cn] 订阅/取消订阅字幕
    *
    * @detail
    * [en] This interface is used to subscribe subtitle.
    * [cn] 该接口用于订阅/取消订阅字幕
    */
    virtual SDKERR SubscribeSubtitle(HWM_BOOL isSubscribe) = 0;

    /**
    * @brief
    * [en] This interface is used to get subtitle support languagues.
    * [cn] 获取字幕支持的语言
    *
    * @detail
    * [en] This interface is used to get subtitle support languagues.
    * [cn] 该接口用于获取字幕支持的语言
    *
    * @mac:transToSync
    */
    virtual SDKERR GetSubtitleSupportLanguage() = 0;

};

/**
* @brief 
* [en] This class is about confMgr result callback.
* [cn] 会议管理的结果回调类回调
*/
class HWM_API IHwmConfMgrResultCallback
{
public:
    /**
    * @brief 
    * [en] Callback of BookConf interface.
    * [cn] 预约会议回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param confDetail [type:HwmConfDetail*][cs_allowNull:N]
    * [en] Conf detail
    * [cn] 会议详情
    **/
    virtual HWM_VOID OnBookConfResult(SDKERR ret, const HWM_CHAR* reason, const HwmConfDetail* confDetail) {};

    /**
    * @brief 
    * [en] Callback of CancelConf interface.
    * [cn] 取消会议预约回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnCancelConfResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of EditConf interface.
    * [cn] 修改会议预约回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnEditConfResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of GetVmrList interface
    * [cn] Vmr列表通知，获取Vmr列表请求返回
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param vmrListInfo [type:HwmVmrListInfo*]
    * [en] Vmr list info
    * [cn] Vmr列表信息
    */
    virtual HWM_VOID OnGetVmrListResult(SDKERR ret, const HWM_CHAR* reason, const HwmVmrListInfo* vmrListInfo) {};

    /**
    * @brief 
    * [en] Callback of GetConfDetail interface.
    * [cn] 会议详情通知，获取会议详情返回
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param confDetail [type:HwmConfDetail*][cs_allowNull:N]
    * [en] Conf detail
    * [cn] 会议详情
    */
    virtual HWM_VOID OnGetConfDetailResult(SDKERR ret, const HWM_CHAR* reason, const HwmConfDetail* confDetail) {};

    /**
    * @brief 
    * [en] Callback of ChangeVmrInfo interface.
    * [cn] 修改虚拟会议室信息回调
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnChangeVmrInfoResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of GetConfList interface.
    * [cn] 会议列表通知，获取会议列表请求返回
    *
    * @param ret [type:SDKERR][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param confListInfo [type:HwmConfListInfo*]
    * [en] Conf list info
    * [cn] 会议列表信息
    */
    virtual HWM_VOID OnGetConfListResult(SDKERR ret, const HWM_CHAR* reason, const HwmConfListInfo* confListInfo) {};

    /**
    * @brief
    * [en]
    * [cn]创建周期会议的结果回调
    *
    * @param ret [type:HwmErrCode][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param confDetail [type:HwmConfDetail*][cs_allowNull:N]
    * [en] Conf detail
    * [cn] 会议详情
    */
    virtual void OnBookCycleConfResult(SDKERR ret, const HWM_CHAR* reason, const HwmConfDetail* confDetail) {};

    /**
    * @brief
    * [en]
    * [cn]编辑周期会议的结果回调
    *
    * @param ret [type:HwmErrCode][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    */
    virtual void OnModifyCycleConfResult(SDKERR  ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en]
    * [cn]编辑周期子会议的结果回调
    *
    * @param ret [type:HwmErrCode][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    */
    virtual void OnModifySubCycleConfResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en]
    * [cn]取消周期会议的结果回调
    *
    * @param ret [type:HwmErrCode][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    */
    virtual void OnCancelCycleConfResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en]
    * [cn]取消周期子会议的结果回调
    *
    * @param ret [type:HwmErrCode][cs_allowNull:N]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*][cs_allowNull:N]
    * [en] Reason
    * [cn] 错误描述
    */
    virtual void OnCancelSubCycleConfResult(SDKERR ret, const HWM_CHAR* reason) {};

};


/**
* @brief 
* [en] This class is about confMgr notify callback.
* [cn] 会议管理通知
*/
class HWM_API IHwmConfMgrNotifyCallback
{
public:

    /**
    * @brief 
    * [en] Notify of the conf incoming
    * [cn] 会议来电通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示收到会议来电。
    *
    * @param confIncomingInfo [type:HwmConfIncomingInfo*]
    * [en] Conf incoming info
    * [cn] 会议来电信息
    **/
    virtual HWM_VOID OnConfIncomingNotify(HwmConfIncomingInfo* confIncomingInfo) {};

    /**
    * @brief 
    * [en] Conf list notify. when login/bookconf/cancelconf, will receive this notify
    * [cn] 会议列表通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示会议列表信息发生变更。
    *
    * @param confListInfo [type:HwmConfListInfo*]
    * [en] Conf list info
    * [cn] 会议列表信息
    */
    virtual HWM_VOID OnConfListChanged(HwmConfListInfo* confListInfo) {};

    /**
    * @brief
    * [en] Create Conf Finished Notify. when create conf, will receive this notify
    * [cn] 完成创建会议通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示完成创建会议操作。
    *
    * @param ret [type:hwmsdk::HwmErrCode]
    * [en] Return code
    * [cn] 返回值
    *
    * @param reason [type:const char*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param createConfResult [type:HwmCreateConfResult*]
    * [en] Create Conf Result
    * [cn] 创建会议结果
    *
    * @mac:disable
    */
    virtual HWM_VOID OnCreateConfFinishedNotify(SDKERR ret, const HWM_CHAR* reason, const HwmCreateConfResult* createConfResult) {};
};

/**
* @brief [en] This class is about conference management object.
*        [cn] 会议管理对象
*
* @detail
* [en]
* [cn] 获取该对象的方法HwmUISDK.getConfMgrApi()
*
* @GetInstance
*        HWM_UISDK_NAMESPACE::HwmUISDK::GetConfMgrApi()
*
*/
class IHwmConfMgr
{
public:
    /**
    * @brief
    * [en] This class is about the global callback for the operating results of conference management.
    * [cn] 设置会议管理操作类全局回调
    * 
    * @param callback [type:IHwmConfMgrResultCallback*][cs_allowNull:N]
    * [en]
    * [cn] 会议管理操作类全局回调
    */
    virtual HWM_VOID SetConfMgrResultCallback(IHwmConfMgrResultCallback* callback) = 0;

    /**
    * @brief
    * [en] This class is about the global callback for the notifications of conference management.
    * [cn] 设置会议管理通知类全局回调
    * 
    * @param callback [type:IHwmConfMgrNotifyCallback*][cs_allowNull:N]
    * [en]
    * [cn] 会议管理通知类全局回调
    */
    virtual HWM_VOID SetConfMgrNotifyCallback(IHwmConfMgrNotifyCallback* callback) = 0;

    /**
    * @brief 
    * [en] This interface is book a conference.
    * [cn] 预定会议
    *
    * @detail
    * [en]
    * [cn] 该接口用于预约会议。注意事项：1、该接口在已登录状态下才能调用；2、调用该接口，与会者列表中需要添加本会场，并且将本会场设置为主持人身份；3、回调函数中会返回预约成功后的会议详情，详见HwmConfDetail结构体表。如果参数内只有会议id有值则获取会议详情失败。
    *
    * @param bookConfParam [type:BookConfParam*][cs_allowNull:N]
    * [en] Book conf param
    * [cn] 预约会议的参数体
    **/
    virtual SDKERR BookConf(const HwmBookConfParam* bookConfParam) = 0;

    /**
    * @brief 
    * [en] This interface is used to cancel a conference booked before.
    * [cn] 取消会议
    *
    * @detail
    * [en]
    * [cn] 该接口用于取消已预约的会议。注意事项：1、该接口在登录状态下才能调用；2、调用该接口时，仅能取消尚未开始的会议，若会议已召开，则接口返回失败。
    *
    * @param cancelConfParam [type:HwmCancelConfParam*][cs_allowNull:N]
    * [en] Cancel conf param
    * [cn] 取消会议的参数体
    **/
    virtual SDKERR CancelConf(const HwmCancelConfParam* cancelConfParam) = 0;

    /**
    * @brief 
    * [en] This interface is used to modify a conf booked before.
    * [cn] 编辑会议
    *
    * @detail
    * [en]
    * [cn] 该接口用于编辑已预约的会议。注意事项：1、该接口在已登录状态下才能调用；2、调用该接口时，仅能编辑尚未开始的会议，若会议已召开，接口将返回失败；3、所有字段必填，编辑会议前需要查询会议详情，不需要变更的数据请使用查询会议详情返回的数据赋值。
    *
    * @param editConfParam [type:HwmEditConfParam*][cs_allowNull:N]
    * [en] Edit conf param
    * [cn] 编辑会议的参数体
    **/
    virtual SDKERR EditConf(const HwmEditConfParam* editConfParam) = 0;

    /**
    * @brief 
    * [en] This interface is used to get vmr list.
    * [cn] 获取VMR列表
    *
    * @detail
    * [en]
    * [cn] 该接口用于获取VMR列表。注意事项：1、该接口在已登录状态下才能调用。
    *
    * @param NULL [cs_allowNull:Y]
    **/
    virtual SDKERR GetVmrList() = 0;

    /**
    * @brief 
    * [en] This interface is used to get conf detail.
    * [cn] 获取会议详情
    *
    * @detail
    * [en]
    * [cn] 该接口用于获取会议的详细信息。注意事项：1、该接口在已登录状态下才能调用。
    *
    * @param getConfDetailParam [type:HwmGetConfDetailParam*][cs_allowNull:N]
    * [en] Get conf detail param
    * [cn] 获取会议详情的参数体
    **/
    virtual SDKERR GetConfDetail(const HwmGetConfDetailParam* getConfDetailParam) = 0;

    /**
    * @brief 
    * [en] This interface is used to change virtual meeting room information.
    * [cn] 修改云会议室信息
    *
    * @detail
    * [en]
    * [cn] 该接口用于修改云会议室信息。注意事项：1、该接口在已登录状态下才能调用；2、所有字段必填，调用该接口前需要获取VMR列表，不需要变更的数据请使用获取VMR列表接口返回的数据赋值。
    *
    * @param vmrInfoParam [type:HwmChangeVmrInfoParam][cs_allowNull:N]
    * [en] The new value of the vmr info
    * [cn] 虚拟会议室的修改信息
    **/
    virtual SDKERR ChangeVmrInfo(const HwmChangeVmrInfoParam* vmrInfoParam) = 0;

    /**
    * @brief 
    * [en]This interface is used to get conf list
    * [cn]获取会议列表
    *
    * @detail
    * [en] (only for mac)
    * [cn] 该接口用于获取会议列表(仅对mac平台生效)。注意事项：1、已登录状态，首次收到会议列表通知之后，调用才有效，否则返回为空的列表
    *
    * @param NULL [cs_allowNull:Y]
    *
    * @win:todo
    * @mac:transToSync
    */
    virtual SDKERR GetConfList() = 0;

    /**
    * @brief
    * [en]
    * [cn]预约周期会议
    *
    * @detail
    * [en] 
    * [cn] 该接口用于预约周期会议。注意事项：1、该接口在已登录状态下才能调用。2、该接口为异步接口，返回值只代表接口是否调用成功，实际业务处理结果在对应的回调函数中返回。
    3、会议开始时间必须晚于当前时间，会议时长应大于15分钟。4、携带与会者入会的时候需要传入attendees。5、周期会议最大跨度一年，会议数量不超过50，超过则只预约前50个会议。
    6、回调函数中会返回预约成功后的会议详情，详见HwmConfDetail结构体表。如果参数内只有会议id有值则获取会议详情失败。
    *
    * @param bookCycleConfParam [type:HwmBookCycleConfParam][cs_allowNull:N]
    * [en]
    * [cn]预约周期会议的参数体
    */
    virtual SDKERR BookCycleConf(const HwmBookCycleConfParam* bookCycleConfParam) = 0;

    /**
    * @brief
    * [en]
    * [cn]编辑周期会议
    *
    * @detail
    * [en] 
    * [cn] 该接口用于编辑周期会议。注意事项：1、该接口在已登录状态下才能调用。2、该接口为异步接口，返回值只代表接口是否调用成功，实际业务处理结果在对应的回调函数中返回。
    3、所有字段必填，编辑修改的字段可来源于用户输入，其他不想修改的属性一定要取会议详情对应字段来赋值。
    *
    * @param modifyParam [type:HwmModifyCycleConfParam][cs_allowNull:N]
    * [en]
    * [cn]编辑周期会议的参数体
    */
    virtual SDKERR ModifyCycleConf(const HwmModifyCycleConfParam* modifyParam) = 0;

    /**
    * @brief
    * [en]
    * [cn]编辑周期子会议
    *
    * @detail
    * [en]
    * [cn] 该接口用于编辑周期子会议。注意事项：1、该接口在已登录状态下才能调用。2、该接口为异步接口，返回值只代表接口是否调用成功，实际业务处理结果在对应的回调函数中返回。
    *
    * @param modifySubCycleConfParam [type:HwmModifySubCycleConfParam][cs_allowNull:N]
    * [en]
    * [cn]编辑周期会议的参数体
    */
    virtual SDKERR ModifySubCycleConf(const HwmModifySubCycleConfParam* modifySubCycleConfParam) = 0;

    /**
    * @brief
    * [en]
    * [cn]取消周期会议
    *
    * @detail
    * [en]
    * [cn] 该接口用于取消至少还有一场子会议未开始的周期会议。注意事项：1、该接口在登录状态下才能调用。2、最后一场子会议未开始前可以调用接口，开始后调用则无效。
    3、该接口为异步接口，返回值只代表接口是否调用成功，实际业务处理结果在对应的回调函数中返回。
    *
    * @param cancelCycleConfParam [type:HwmCancelCycleConfParam][cs_allowNull:N]
    * [en]
    * [cn]取消周期会议参数
    */
    virtual SDKERR CancelCycleConf(const HwmCancelCycleConfParam* cancelCycleConfParam) = 0;

    /**
    * @brief
    * [en]
    * [cn]取消周期子会议
    *
    * @detail
    * [en]
    * [cn]该接口用于取消周期子会议。注意事项：1、该接口在已登录状态下才能调用。2、子会议未开始前可以调用接口，开始后调用则无效
    3、该接口为异步接口，返回值只代表接口是否调用成功，实际业务处理结果在对应的回调函数中返回。
    *
    * @param cancelSubCycleConfParam [type:HwmCancelSubCycleConfParam][cs_allowNull:N]
    * [en]
    * [cn]取消周期子会议参数
    */
    virtual SDKERR CancelSubCycleConf(const HwmCancelSubCycleConfParam* cancelSubCycleConfParam) = 0;
};

END_HWM_UISDK_NAMESPACE
#endif /* __HWM_UISDK_CONF_SERVICE_H__ */