"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ILogin = void 0;
var enum_1 = require("./enum");
var util_1 = require("./util");
var electron = (typeof window !== 'undefined') ? window.require('electron') : require('electron');
var ILogin = /** @class */ (function () {
    function ILogin(addOn) {
        this.callWithIPC = false;
        this.addOn = addOn;
        this._inProcessCallbackSet = new Set();
        this.callWithIPC = util_1.needCallWithIPC();
    }
    ILogin.prototype.init = function () {
        this._inProcessCallbackSet.clear();
    };
    /**
    * [en] Notify of be kicked out. [cn] 登录被踢通知
    */
    ILogin.prototype.setOnKickedOutNotifyCB = function (onKickedOutNotify) {
        if (this.callWithIPC) {
            electron.ipcRenderer.sendSync('setOnKickedOutNotifyCB', {});
            electron.ipcRenderer.on('onKickedOutNotify', function (event, arg) {
                if (onKickedOutNotify) {
                    onKickedOutNotify();
                }
            });
            return;
        }
        this._onKickedOutNotifyCallback = function () {
            if (onKickedOutNotify) {
                onKickedOutNotify();
            }
        };
        this.addOn.setOnKickedOutNotifyCB(this._onKickedOutNotifyCallback);
    };
    /**
    * [en] Notify of the config of corporation. [cn] 企业配置通知
    */
    ILogin.prototype.setOnCorpConfigNotifyCB = function (onCorpConfigNotify) {
        if (this.callWithIPC) {
            electron.ipcRenderer.sendSync('setOnCorpConfigNotifyCB', {});
            electron.ipcRenderer.on('onCorpConfigNotify', function (event, arg) {
                if (onCorpConfigNotify) {
                    onCorpConfigNotify(arg["configInfo"]);
                }
            });
            return;
        }
        this._onCorpConfigNotifyCallback = function (configInfo) {
            if (onCorpConfigNotify) {
                onCorpConfigNotify(configInfo);
            }
        };
        this.addOn.setOnCorpConfigNotifyCB(this._onCorpConfigNotifyCallback);
    };
    /**
    * [en] Notify of the info of corporation. [cn] 客户企业信息变化通知
    */
    ILogin.prototype.setOnUserCorpInfoChangedCB = function (onUserCorpInfoChanged) {
        if (this.callWithIPC) {
            electron.ipcRenderer.sendSync('setOnUserCorpInfoChangedCB', {});
            electron.ipcRenderer.on('onUserCorpInfoChanged', function (event, arg) {
                if (onUserCorpInfoChanged) {
                    onUserCorpInfoChanged(arg["type"]);
                }
            });
            return;
        }
        this._onUserCorpInfoChangedCallback = function (type) {
            if (onUserCorpInfoChanged) {
                onUserCorpInfoChanged(type);
            }
        };
        this.addOn.setOnUserCorpInfoChangedCB(this._onUserCorpInfoChangedCallback);
    };
    /**
    * [en]This is about login error notify. [cn]登录错误通知
    */
    ILogin.prototype.setOnLoginErrorNotifyCB = function (onLoginErrorNotify) {
        if (this.callWithIPC) {
            electron.ipcRenderer.sendSync('setOnLoginErrorNotifyCB', {});
            electron.ipcRenderer.on('onLoginErrorNotify', function (event, arg) {
                if (onLoginErrorNotify) {
                    onLoginErrorNotify(arg["ret"], arg["reason"]);
                }
            });
            return;
        }
        this._onLoginErrorNotifyCallback = function (ret, reason) {
            if (onLoginErrorNotify) {
                onLoginErrorNotify(ret, reason);
            }
        };
        this.addOn.setOnLoginErrorNotifyCB(this._onLoginErrorNotifyCallback);
    };
    /**
    * [en]This is about share vmr info notify. [cn]共享vmr信息通知
    */
    ILogin.prototype.setOnShareVmrInfoNotifyCB = function (onShareVmrInfoNotify) {
        if (this.callWithIPC) {
            electron.ipcRenderer.sendSync('setOnShareVmrInfoNotifyCB', {});
            electron.ipcRenderer.on('onShareVmrInfoNotify', function (event, arg) {
                if (onShareVmrInfoNotify) {
                    onShareVmrInfoNotify(arg["shareVmrInfo"]);
                }
            });
            return;
        }
        this._onShareVmrInfoNotifyCallback = function (shareVmrInfo) {
            if (onShareVmrInfoNotify) {
                onShareVmrInfoNotify(shareVmrInfo);
            }
        };
        this.addOn.setOnShareVmrInfoNotifyCB(this._onShareVmrInfoNotifyCallback);
    };
    /**
    * [en]This is about login state change notify. [cn]登录状态变化通知
    */
    ILogin.prototype.setOnLoginStateChangeNotifyCB = function (onLoginStateChangeNotify) {
        if (this.callWithIPC) {
            electron.ipcRenderer.sendSync('setOnLoginStateChangeNotifyCB', {});
            electron.ipcRenderer.on('onLoginStateChangeNotify', function (event, arg) {
                if (onLoginStateChangeNotify) {
                    onLoginStateChangeNotify(arg["loginStateInfo"]);
                }
            });
            return;
        }
        this._onLoginStateChangeNotifyCallback = function (loginStateInfo) {
            if (onLoginStateChangeNotify) {
                onLoginStateChangeNotify(loginStateInfo);
            }
        };
        this.addOn.setOnLoginStateChangeNotifyCB(this._onLoginStateChangeNotifyCallback);
    };
    /**
    * [en] This interface is used to login by account info. [cn] 账号密码登录
    */
    ILogin.prototype.loginByAccount = function (accountAuthInfo, onLoginResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("loginByAccount")) {
            if (onLoginResult) {
                var nullResult = {};
                onLoginResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("loginByAccount");
        this._loginByAccountCallback = function (ret, reason, loginResult) {
            _this._inProcessCallbackSet.delete("loginByAccount");
            if (onLoginResult) {
                onLoginResult(ret, reason, loginResult);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('loginByAccount', { "accountAuthInfo": accountAuthInfo }).then(function (ret) {
                _this._inProcessCallbackSet.delete('loginByAccount');
                if (onLoginResult) {
                    onLoginResult(ret["ret"], ret["reason"], ret["loginResult"]);
                }
            });
        }
        this.addOn.loginByAccount(accountAuthInfo, this._loginByAccountCallback);
    };
    /**
    * [en] This interface is used to login by appId info. [cn] AppId登录
    */
    ILogin.prototype.loginByAppId = function (appIdAuthInfo, onLoginResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("loginByAppId")) {
            if (onLoginResult) {
                var nullResult = {};
                onLoginResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("loginByAppId");
        this._loginByAppIdCallback = function (ret, reason, loginResult) {
            _this._inProcessCallbackSet.delete("loginByAppId");
            if (onLoginResult) {
                onLoginResult(ret, reason, loginResult);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('loginByAppId', { "appIdAuthInfo": appIdAuthInfo }).then(function (ret) {
                _this._inProcessCallbackSet.delete('loginByAppId');
                if (onLoginResult) {
                    onLoginResult(ret["ret"], ret["reason"], ret["loginResult"]);
                }
            });
        }
        this.addOn.loginByAppId(appIdAuthInfo, this._loginByAppIdCallback);
    };
    /**
    * [en] This method is used to login by SSO. [cn] SSO登录
    */
    ILogin.prototype.loginBySSO = function (ssoAuthInfo, onLoginResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("loginBySSO")) {
            if (onLoginResult) {
                var nullResult = {};
                onLoginResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("loginBySSO");
        this._loginBySSOCallback = function (ret, reason, loginResult) {
            _this._inProcessCallbackSet.delete("loginBySSO");
            if (onLoginResult) {
                onLoginResult(ret, reason, loginResult);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('loginBySSO', { "ssoAuthInfo": ssoAuthInfo }).then(function (ret) {
                _this._inProcessCallbackSet.delete('loginBySSO');
                if (onLoginResult) {
                    onLoginResult(ret["ret"], ret["reason"], ret["loginResult"]);
                }
            });
        }
        this.addOn.loginBySSO(ssoAuthInfo, this._loginBySSOCallback);
    };
    /**
    * [en] This interface is used to logout sdk. [cn] 退出登录
    */
    ILogin.prototype.logout = function (onLogoutResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("logout")) {
            if (onLogoutResult) {
                var nullResult = {};
                onLogoutResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("logout");
        this._logoutCallback = function (ret, reason, logoutResult) {
            _this._inProcessCallbackSet.delete("logout");
            if (onLogoutResult) {
                onLogoutResult(ret, reason, logoutResult);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('logout', {}).then(function (ret) {
                _this._inProcessCallbackSet.delete('logout');
                if (onLogoutResult) {
                    onLogoutResult(ret["ret"], ret["reason"], ret["logoutResult"]);
                }
            });
        }
        this.addOn.logout(this._logoutCallback);
    };
    /**
    * [en] This interface is used to set network mode. [cn] 设置网络模式
    */
    ILogin.prototype.setNetworkMode = function (networkInfo, onSetNetworkModeResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("setNetworkMode")) {
            if (onSetNetworkModeResult) {
                onSetNetworkModeResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("setNetworkMode");
        this._setNetworkModeCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("setNetworkMode");
            if (onSetNetworkModeResult) {
                onSetNetworkModeResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('setNetworkMode', { "networkInfo": networkInfo }).then(function (ret) {
                _this._inProcessCallbackSet.delete('setNetworkMode');
                if (onSetNetworkModeResult) {
                    onSetNetworkModeResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.setNetworkMode(networkInfo, this._setNetworkModeCallback);
    };
    /**
    * [en] This interface is used to upload self avatar. [cn] 上传头像
    */
    ILogin.prototype.uploadSelfAvatar = function (selfAvatarInfo, onUploadSelfAvatarResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("uploadSelfAvatar")) {
            if (onUploadSelfAvatarResult) {
                onUploadSelfAvatarResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("uploadSelfAvatar");
        this._uploadSelfAvatarCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("uploadSelfAvatar");
            if (onUploadSelfAvatarResult) {
                onUploadSelfAvatarResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('uploadSelfAvatar', { "selfAvatarInfo": selfAvatarInfo }).then(function (ret) {
                _this._inProcessCallbackSet.delete('uploadSelfAvatar');
                if (onUploadSelfAvatarResult) {
                    onUploadSelfAvatarResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.uploadSelfAvatar(selfAvatarInfo, this._uploadSelfAvatarCallback);
    };
    /**
    * [en] This interface is used to delete self avatar. [cn] 删除头像。
    */
    ILogin.prototype.deleteSelfAvatar = function (onDeleteSelfAvatarResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("deleteSelfAvatar")) {
            if (onDeleteSelfAvatarResult) {
                onDeleteSelfAvatarResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("deleteSelfAvatar");
        this._deleteSelfAvatarCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("deleteSelfAvatar");
            if (onDeleteSelfAvatarResult) {
                onDeleteSelfAvatarResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('deleteSelfAvatar', {}).then(function (ret) {
                _this._inProcessCallbackSet.delete('deleteSelfAvatar');
                if (onDeleteSelfAvatarResult) {
                    onDeleteSelfAvatarResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.deleteSelfAvatar(this._deleteSelfAvatarCallback);
    };
    /**
    * [en] This interface is used to set guaranteed ip addresses. [cn] 设置(会控)保底ip地址
    */
    ILogin.prototype.setGuaranteedIpAddresses = function (guaranteedIpAddresses) {
        if (this.callWithIPC) {
            var ret = electron.ipcRenderer.sendSync('setGuaranteedIpAddresses', { "guaranteedIpAddresses": guaranteedIpAddresses });
            return ret;
        }
        return this.addOn.setGuaranteedIpAddresses(guaranteedIpAddresses);
    };
    /**
    * [en] This interface is used to set guaranteed ip addresses of global domain. [cn] 设置管理面域名对应的保底ip地址
    */
    ILogin.prototype.setGlobalGuaranteedIpAddresses = function (guaranteedIpAddressesParam, onSetGlobalGuaranteedIpAddressesResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("setGlobalGuaranteedIpAddresses")) {
            if (onSetGlobalGuaranteedIpAddressesResult) {
                onSetGlobalGuaranteedIpAddressesResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("setGlobalGuaranteedIpAddresses");
        this._setGlobalGuaranteedIpAddressesCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("setGlobalGuaranteedIpAddresses");
            if (onSetGlobalGuaranteedIpAddressesResult) {
                onSetGlobalGuaranteedIpAddressesResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('setGlobalGuaranteedIpAddresses', { "guaranteedIpAddressesParam": guaranteedIpAddressesParam }).then(function (ret) {
                _this._inProcessCallbackSet.delete('setGlobalGuaranteedIpAddresses');
                if (onSetGlobalGuaranteedIpAddressesResult) {
                    onSetGlobalGuaranteedIpAddressesResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.setGlobalGuaranteedIpAddresses(guaranteedIpAddressesParam, this._setGlobalGuaranteedIpAddressesCallback);
    };
    /**
    * [en] This interface is used to set login server info. [cn] 设置登录服务器信息
    */
    ILogin.prototype.setServerAddress = function (serverInfo, onSetServerAddressResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("setServerAddress")) {
            if (onSetServerAddressResult) {
                onSetServerAddressResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("setServerAddress");
        this._setServerAddressCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("setServerAddress");
            if (onSetServerAddressResult) {
                onSetServerAddressResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('setServerAddress', { "serverInfo": serverInfo }).then(function (ret) {
                _this._inProcessCallbackSet.delete('setServerAddress');
                if (onSetServerAddressResult) {
                    onSetServerAddressResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.setServerAddress(serverInfo, this._setServerAddressCallback);
    };
    /**
    * [en] This interface is used to set proxy info. [cn] 设置代理信息
    */
    ILogin.prototype.setProxy = function (proxyInfo, onSetProxyResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("setProxy")) {
            if (onSetProxyResult) {
                onSetProxyResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("setProxy");
        this._setProxyCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("setProxy");
            if (onSetProxyResult) {
                onSetProxyResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('setProxy', { "proxyInfo": proxyInfo }).then(function (ret) {
                _this._inProcessCallbackSet.delete('setProxy');
                if (onSetProxyResult) {
                    onSetProxyResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.setProxy(proxyInfo, this._setProxyCallback);
    };
    /**
    * [en] This interface is used to set CA verify info. [cn] 设置证书校验信息
    */
    ILogin.prototype.setCAVerifyInfo = function (CAVerifyInfo, onSetCAVerifyInfoResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("setCAVerifyInfo")) {
            if (onSetCAVerifyInfoResult) {
                onSetCAVerifyInfoResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("setCAVerifyInfo");
        this._setCAVerifyInfoCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("setCAVerifyInfo");
            if (onSetCAVerifyInfoResult) {
                onSetCAVerifyInfoResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('setCAVerifyInfo', { "CAVerifyInfo": CAVerifyInfo }).then(function (ret) {
                _this._inProcessCallbackSet.delete('setCAVerifyInfo');
                if (onSetCAVerifyInfoResult) {
                    onSetCAVerifyInfoResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.setCAVerifyInfo(CAVerifyInfo, this._setCAVerifyInfoCallback);
    };
    /**
    * [en] This interface is used to get login state. [cn] 获取登录状态
    */
    ILogin.prototype.getLoginState = function (onGetLoginStateResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("getLoginState")) {
            if (onGetLoginStateResult) {
                var nullResult = {};
                onGetLoginStateResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("getLoginState");
        this._getLoginStateCallback = function (ret, reason, loginStateInfo) {
            _this._inProcessCallbackSet.delete("getLoginState");
            if (onGetLoginStateResult) {
                onGetLoginStateResult(ret, reason, loginStateInfo);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('getLoginState', {}).then(function (ret) {
                _this._inProcessCallbackSet.delete('getLoginState');
                if (onGetLoginStateResult) {
                    onGetLoginStateResult(ret["ret"], ret["reason"], ret["loginStateInfo"]);
                }
            });
        }
        this.addOn.getLoginState(this._getLoginStateCallback);
    };
    /**
    * [en] This interface is used to get login token. [cn] 获取登录token
    */
    ILogin.prototype.getLoginToken = function (onGetLoginTokenResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("getLoginToken")) {
            if (onGetLoginTokenResult) {
                var nullResult = {};
                onGetLoginTokenResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("getLoginToken");
        this._getLoginTokenCallback = function (ret, reason, loginTokenInfo) {
            _this._inProcessCallbackSet.delete("getLoginToken");
            if (onGetLoginTokenResult) {
                onGetLoginTokenResult(ret, reason, loginTokenInfo);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('getLoginToken', {}).then(function (ret) {
                _this._inProcessCallbackSet.delete('getLoginToken');
                if (onGetLoginTokenResult) {
                    onGetLoginTokenResult(ret["ret"], ret["reason"], ret["loginTokenInfo"]);
                }
            });
        }
        this.addOn.getLoginToken(this._getLoginTokenCallback);
    };
    return ILogin;
}());
exports.ILogin = ILogin;
