import { SavePathType, SDKERR} from './enum';
import { AudioFrameData, AudioRawDataConfigInfo, CapabilityInfo, CustomSceneImageInfoList, CustomSceneTextInfoList, DeviceConfig, DeviceCtrlInfo, DeviceStateInfo, ErrorMessageInfo, InComingAnswerTypeInfo, LanguageInfo, LogPathInfo, PstnAccessNumberInfo, SharePermission, ShareStrategyInfo, ShowVoiceprintStatementInfo, TransparentWndInfo, UserSavePathInfo, VirtualBackgroundInfo, VoicePromptsInfo, VoiceprintRecordWindowConfig} from './model';
import { needCallWithIPC } from './util';
const electron = (typeof window !== 'undefined') ? (window as any).require('electron') : require('electron');

export class IConfig {
	addOn!: IConfig;
	_onAVDeviceCtrlStateChangedCallback!: (info: DeviceStateInfo) => void;
	_onAudioFrameNotifyCallback!: (audioFrameData: AudioFrameData) => void;
	_onShowVoiceprintStatementCallback!: (info: ShowVoiceprintStatementInfo) => void;
	_onUserSavePathChangedCallback!: (userSavePathInfo: UserSavePathInfo) => void;
	_enableCapabilityCallback!: (ret: SDKERR, reason: string) => void;
	_setLanguageCallback!: (ret: SDKERR, reason: string) => void;
	_setUserSavePathCallback!: (ret: SDKERR, reason: string) => void;
	_getUserSavePathCallback!: (ret: SDKERR, reason: string, pathInfo: UserSavePathInfo) => void;
	_setShareStrategyCallback!: (ret: SDKERR, reason: string) => void;
	_setAudioRawDataOutputConfigCallback!: (ret: SDKERR, reason: string) => void;
	_setAnswerTypeCallback!: (ret: SDKERR, reason: string) => void;
	_setDeviceConfigForConfIncomingCallback!: (ret: SDKERR, reason: string) => void;
	_getConfPstnAccessNumberCallback!: (ret: SDKERR, reason: string, pstnAccessNumberInfo: PstnAccessNumberInfo) => void;
	_aVDeviceCtrlCallback!: (ret: SDKERR, reason: string, deviceCtrlInfo: DeviceCtrlInfo) => void;
	_disableVoicePromptsCallback!: (ret: SDKERR, reason: string) => void;
	_setShareScreenPermissionCallback!: (ret: SDKERR, reason: string) => void;
	_agreeVoiceprintStatementCallback!: (ret: SDKERR, reason: string) => void;
	_showVoiceprintRecordWindowCallback!: (ret: SDKERR, reason: string) => void;
	_setLogPathCallback!: (ret: SDKERR, reason: string) => void;
	_setTransparentWndCallback!: (ret: SDKERR, reason: string) => void;
	_setCustomUITextCallback!: (ret: SDKERR, reason: string) => void;
	_setCustomUIImageCallback!: (ret: SDKERR, reason: string) => void;
	_setVirtualBackgroundDefaultImagesCallback!: (ret: SDKERR, reason: string) => void;
	_inProcessCallbackSet!: Set<string>;
	callWithIPC: boolean = false;
	constructor(addOn: IConfig) {
		this.addOn = addOn;
		this._inProcessCallbackSet = new Set<string>();
		this.callWithIPC = needCallWithIPC();
	}

	init(): void {
		this._inProcessCallbackSet.clear();
	}

	/**
	* [en] Notify of AVDevice control state [cn] 音视频设备控制状态变化通知
	*/
	setOnAVDeviceCtrlStateChangedCB(onAVDeviceCtrlStateChanged: (info: DeviceStateInfo) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnAVDeviceCtrlStateChangedCB', {});
			electron.ipcRenderer.on('onAVDeviceCtrlStateChanged', (event: any, arg: any) => {
				if (onAVDeviceCtrlStateChanged) {
					onAVDeviceCtrlStateChanged(arg["info"])
				}
			});
			return;
		}

		this._onAVDeviceCtrlStateChangedCallback = (info: DeviceStateInfo) => {
			if (onAVDeviceCtrlStateChanged) {
				onAVDeviceCtrlStateChanged(info);
			}
		}
		this.addOn.setOnAVDeviceCtrlStateChangedCB(this._onAVDeviceCtrlStateChangedCallback);
	}

	/**
	* [en] Notify of audio famme [cn] 上报音频码流帧数据通知
	*/
	setOnAudioFrameNotifyCB(onAudioFrameNotify: (audioFrameData: AudioFrameData) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnAudioFrameNotifyCB', {});
			electron.ipcRenderer.on('onAudioFrameNotify', (event: any, arg: any) => {
				if (onAudioFrameNotify) {
					onAudioFrameNotify(arg["audioFrameData"])
				}
			});
			return;
		}

		this._onAudioFrameNotifyCallback = (audioFrameData: AudioFrameData) => {
			if (onAudioFrameNotify) {
				onAudioFrameNotify(audioFrameData);
			}
		}
		this.addOn.setOnAudioFrameNotifyCB(this._onAudioFrameNotifyCallback);
	}

	/**
	* [en] Notify of show voiceprint statement [cn] 声纹降噪功能声明显示通知
	*/
	setOnShowVoiceprintStatementCB(onShowVoiceprintStatement: (info: ShowVoiceprintStatementInfo) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnShowVoiceprintStatementCB', {});
			electron.ipcRenderer.on('onShowVoiceprintStatement', (event: any, arg: any) => {
				if (onShowVoiceprintStatement) {
					onShowVoiceprintStatement(arg["info"])
				}
			});
			return;
		}

		this._onShowVoiceprintStatementCallback = (info: ShowVoiceprintStatementInfo) => {
			if (onShowVoiceprintStatement) {
				onShowVoiceprintStatement(info);
			}
		}
		this.addOn.setOnShowVoiceprintStatementCB(this._onShowVoiceprintStatementCallback);
	}

	/**
	* [en] User data path change notification [cn] 用户数据路径变化通知
	*/
	setOnUserSavePathChangedCB(onUserSavePathChanged: (userSavePathInfo: UserSavePathInfo) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnUserSavePathChangedCB', {});
			electron.ipcRenderer.on('onUserSavePathChanged', (event: any, arg: any) => {
				if (onUserSavePathChanged) {
					onUserSavePathChanged(arg["userSavePathInfo"])
				}
			});
			return;
		}

		this._onUserSavePathChangedCallback = (userSavePathInfo: UserSavePathInfo) => {
			if (onUserSavePathChanged) {
				onUserSavePathChanged(userSavePathInfo);
			}
		}
		this.addOn.setOnUserSavePathChangedCB(this._onUserSavePathChangedCallback);
	}

	/**
	* [en] This interface is used to enable capability [cn] 功能配置开关
	*/
	enableCapability(capabilityInfo: CapabilityInfo, onEnableCapabilityResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("enableCapability")) {
			if (onEnableCapabilityResult) {
				onEnableCapabilityResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("enableCapability");
		this._enableCapabilityCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("enableCapability");
			if (onEnableCapabilityResult) {
				onEnableCapabilityResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('enableCapability', {"capabilityInfo":capabilityInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('enableCapability');
				if (onEnableCapabilityResult) {
					onEnableCapabilityResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.enableCapability(capabilityInfo, this._enableCapabilityCallback);
	}

	/**
	* [en] This interface is used to set language. [cn] 设置语言
	*/
	setLanguage(languageInfo: LanguageInfo, onSetLanguageResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setLanguage")) {
			if (onSetLanguageResult) {
				onSetLanguageResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setLanguage");
		this._setLanguageCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setLanguage");
			if (onSetLanguageResult) {
				onSetLanguageResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setLanguage', {"languageInfo":languageInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setLanguage');
				if (onSetLanguageResult) {
					onSetLanguageResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setLanguage(languageInfo, this._setLanguageCallback);
	}

	/**
	* [en] This interface is used to set file save path. [cn] 设置用户保存路径
	*/
	setUserSavePath(userSavePathInfo: UserSavePathInfo, onSetUserSavePathResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setUserSavePath")) {
			if (onSetUserSavePathResult) {
				onSetUserSavePathResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setUserSavePath");
		this._setUserSavePathCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setUserSavePath");
			if (onSetUserSavePathResult) {
				onSetUserSavePathResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setUserSavePath', {"userSavePathInfo":userSavePathInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setUserSavePath');
				if (onSetUserSavePathResult) {
					onSetUserSavePathResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setUserSavePath(userSavePathInfo, this._setUserSavePathCallback);
	}

	/**
	* [en] This interface is used to get file save path. [cn] 获取用户保存路径
	*/
	getUserSavePath(pathType: SavePathType, onGetUserSavePathResult?: (ret: SDKERR, reason: string, pathInfo: UserSavePathInfo) => void): void {
		if (this._inProcessCallbackSet.has("getUserSavePath")) {
			if (onGetUserSavePathResult) {
				let nullResult = <UserSavePathInfo>{};
				onGetUserSavePathResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("getUserSavePath");
		this._getUserSavePathCallback = (ret: SDKERR, reason: string, pathInfo: UserSavePathInfo) => {
			this._inProcessCallbackSet.delete("getUserSavePath");
			if (onGetUserSavePathResult) {
				onGetUserSavePathResult(ret, reason, pathInfo);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('getUserSavePath', {"pathType":pathType}).then((ret: any) => {
				this._inProcessCallbackSet.delete('getUserSavePath');
				if (onGetUserSavePathResult) {
					onGetUserSavePathResult(ret["ret"], ret["reason"], ret["pathInfo"]);
				}
			});
		}

		this.addOn.getUserSavePath(pathType, this._getUserSavePathCallback);
	}

	/**
	* [en] This interface is used to set share strategy. [cn] 设置共享策略
	*/
	setShareStrategy(shareStrategyInfo: ShareStrategyInfo, onSetShareStrategyResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setShareStrategy")) {
			if (onSetShareStrategyResult) {
				onSetShareStrategyResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setShareStrategy");
		this._setShareStrategyCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setShareStrategy");
			if (onSetShareStrategyResult) {
				onSetShareStrategyResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setShareStrategy', {"shareStrategyInfo":shareStrategyInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setShareStrategy');
				if (onSetShareStrategyResult) {
					onSetShareStrategyResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setShareStrategy(shareStrategyInfo, this._setShareStrategyCallback);
	}

	/**
	* [en] This interface is used to set audio raw data output config. [cn] 设置开启或关闭音频码流
	*/
	setAudioRawDataOutputConfig(audioRawDataConfigInfo: AudioRawDataConfigInfo, onSetAudioRawDataOutputConfigResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setAudioRawDataOutputConfig")) {
			if (onSetAudioRawDataOutputConfigResult) {
				onSetAudioRawDataOutputConfigResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setAudioRawDataOutputConfig");
		this._setAudioRawDataOutputConfigCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setAudioRawDataOutputConfig");
			if (onSetAudioRawDataOutputConfigResult) {
				onSetAudioRawDataOutputConfigResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setAudioRawDataOutputConfig', {"audioRawDataConfigInfo":audioRawDataConfigInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setAudioRawDataOutputConfig');
				if (onSetAudioRawDataOutputConfigResult) {
					onSetAudioRawDataOutputConfigResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setAudioRawDataOutputConfig(audioRawDataConfigInfo, this._setAudioRawDataOutputConfigCallback);
	}

	/**
	* [en] This interface is used to set call incoming answer type. [cn] 设置来电应答方式
	*/
	setAnswerType(inComingAnswerTypeInfo: InComingAnswerTypeInfo, onSetAnswerTypeResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setAnswerType")) {
			if (onSetAnswerTypeResult) {
				onSetAnswerTypeResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setAnswerType");
		this._setAnswerTypeCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setAnswerType");
			if (onSetAnswerTypeResult) {
				onSetAnswerTypeResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setAnswerType', {"inComingAnswerTypeInfo":inComingAnswerTypeInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setAnswerType');
				if (onSetAnswerTypeResult) {
					onSetAnswerTypeResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setAnswerType(inComingAnswerTypeInfo, this._setAnswerTypeCallback);
	}

	/**
	* [en] This interface is used to set device config for conf incoming. [cn] 会议来电时设备开关配置
	*/
	setDeviceConfigForConfIncoming(deviceConfig: DeviceConfig, onSetDeviceConfigForConfIncomingResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setDeviceConfigForConfIncoming")) {
			if (onSetDeviceConfigForConfIncomingResult) {
				onSetDeviceConfigForConfIncomingResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setDeviceConfigForConfIncoming");
		this._setDeviceConfigForConfIncomingCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setDeviceConfigForConfIncoming");
			if (onSetDeviceConfigForConfIncomingResult) {
				onSetDeviceConfigForConfIncomingResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setDeviceConfigForConfIncoming', {"deviceConfig":deviceConfig}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setDeviceConfigForConfIncoming');
				if (onSetDeviceConfigForConfIncomingResult) {
					onSetDeviceConfigForConfIncomingResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setDeviceConfigForConfIncoming(deviceConfig, this._setDeviceConfigForConfIncomingCallback);
	}

	/**
	* [en] This interface is used to get conf pstn access number. [cn] 获取会议pstn接入号码
	*/
	getConfPstnAccessNumber(onGetConfPstnAccessNumberResult?: (ret: SDKERR, reason: string, pstnAccessNumberInfo: PstnAccessNumberInfo) => void): void {
		if (this._inProcessCallbackSet.has("getConfPstnAccessNumber")) {
			if (onGetConfPstnAccessNumberResult) {
				let nullResult = <PstnAccessNumberInfo>{};
				onGetConfPstnAccessNumberResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("getConfPstnAccessNumber");
		this._getConfPstnAccessNumberCallback = (ret: SDKERR, reason: string, pstnAccessNumberInfo: PstnAccessNumberInfo) => {
			this._inProcessCallbackSet.delete("getConfPstnAccessNumber");
			if (onGetConfPstnAccessNumberResult) {
				onGetConfPstnAccessNumberResult(ret, reason, pstnAccessNumberInfo);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('getConfPstnAccessNumber', {}).then((ret: any) => {
				this._inProcessCallbackSet.delete('getConfPstnAccessNumber');
				if (onGetConfPstnAccessNumberResult) {
					onGetConfPstnAccessNumberResult(ret["ret"], ret["reason"], ret["pstnAccessNumberInfo"]);
				}
			});
		}

		this.addOn.getConfPstnAccessNumber(this._getConfPstnAccessNumberCallback);
	}

	/**
	* [en] This interface is used to AVDeviceCtrl. [cn] 音视频设备控制功能
	*/
	aVDeviceCtrl(deviceStateInfo: DeviceStateInfo, onAVDeviceCtrlResult?: (ret: SDKERR, reason: string, deviceCtrlInfo: DeviceCtrlInfo) => void): void {
		if (this._inProcessCallbackSet.has("aVDeviceCtrl")) {
			if (onAVDeviceCtrlResult) {
				let nullResult = <DeviceCtrlInfo>{};
				onAVDeviceCtrlResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("aVDeviceCtrl");
		this._aVDeviceCtrlCallback = (ret: SDKERR, reason: string, deviceCtrlInfo: DeviceCtrlInfo) => {
			this._inProcessCallbackSet.delete("aVDeviceCtrl");
			if (onAVDeviceCtrlResult) {
				onAVDeviceCtrlResult(ret, reason, deviceCtrlInfo);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('aVDeviceCtrl', {"deviceStateInfo":deviceStateInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('aVDeviceCtrl');
				if (onAVDeviceCtrlResult) {
					onAVDeviceCtrlResult(ret["ret"], ret["reason"], ret["deviceCtrlInfo"]);
				}
			});
		}

		this.addOn.aVDeviceCtrl(deviceStateInfo, this._aVDeviceCtrlCallback);
	}

	/**
	* [en] This interface is used to disable voice prompts. [cn] 禁用提示音
	*/
	disableVoicePrompts(voicePromptsInfo: VoicePromptsInfo, onDisableVoicePromptsResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("disableVoicePrompts")) {
			if (onDisableVoicePromptsResult) {
				onDisableVoicePromptsResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("disableVoicePrompts");
		this._disableVoicePromptsCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("disableVoicePrompts");
			if (onDisableVoicePromptsResult) {
				onDisableVoicePromptsResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('disableVoicePrompts', {"voicePromptsInfo":voicePromptsInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('disableVoicePrompts');
				if (onDisableVoicePromptsResult) {
					onDisableVoicePromptsResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.disableVoicePrompts(voicePromptsInfo, this._disableVoicePromptsCallback);
	}

	/**
	* [en] This interface is used to set share permission. [cn] 设置共享权限
	*/
	setShareScreenPermission(permission: SharePermission, onSetShareScreenPermissionResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setShareScreenPermission")) {
			if (onSetShareScreenPermissionResult) {
				onSetShareScreenPermissionResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setShareScreenPermission");
		this._setShareScreenPermissionCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setShareScreenPermission");
			if (onSetShareScreenPermissionResult) {
				onSetShareScreenPermissionResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setShareScreenPermission', {"permission":permission}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setShareScreenPermission');
				if (onSetShareScreenPermissionResult) {
					onSetShareScreenPermissionResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setShareScreenPermission(permission, this._setShareScreenPermissionCallback);
	}

	/**
	* [en] This interface is used to set voiceprint statement agreement. [cn] 签署声纹功能声明
	*/
	agreeVoiceprintStatement(isAgreed: boolean, onAgreeVoiceprintStatementResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("agreeVoiceprintStatement")) {
			if (onAgreeVoiceprintStatementResult) {
				onAgreeVoiceprintStatementResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("agreeVoiceprintStatement");
		this._agreeVoiceprintStatementCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("agreeVoiceprintStatement");
			if (onAgreeVoiceprintStatementResult) {
				onAgreeVoiceprintStatementResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('agreeVoiceprintStatement', {"isAgreed":isAgreed}).then((ret: any) => {
				this._inProcessCallbackSet.delete('agreeVoiceprintStatement');
				if (onAgreeVoiceprintStatementResult) {
					onAgreeVoiceprintStatementResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.agreeVoiceprintStatement(isAgreed, this._agreeVoiceprintStatementCallback);
	}

	/**
	* [en] This interface is used to show voiceprint record window. [cn] 显示声纹录入窗口接口
	*/
	showVoiceprintRecordWindow(isShow: boolean, onShowVoiceprintRecordWindowResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("showVoiceprintRecordWindow")) {
			if (onShowVoiceprintRecordWindowResult) {
				onShowVoiceprintRecordWindowResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("showVoiceprintRecordWindow");
		this._showVoiceprintRecordWindowCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("showVoiceprintRecordWindow");
			if (onShowVoiceprintRecordWindowResult) {
				onShowVoiceprintRecordWindowResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('showVoiceprintRecordWindow', {"isShow":isShow}).then((ret: any) => {
				this._inProcessCallbackSet.delete('showVoiceprintRecordWindow');
				if (onShowVoiceprintRecordWindowResult) {
					onShowVoiceprintRecordWindowResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.showVoiceprintRecordWindow(isShow, this._showVoiceprintRecordWindowCallback);
	}

	/**
	* [en] This interface is used to set log save path. [cn] 设置日志保存路径
	*/
	setLogPath(logPathInfo: LogPathInfo, onSetLogPathResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setLogPath")) {
			if (onSetLogPathResult) {
				onSetLogPathResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setLogPath");
		this._setLogPathCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setLogPath");
			if (onSetLogPathResult) {
				onSetLogPathResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setLogPath', {"logPathInfo":logPathInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setLogPath');
				if (onSetLogPathResult) {
					onSetLogPathResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setLogPath(logPathInfo, this._setLogPathCallback);
	}

	/**
	* [en] This interface is used to get error message. [cn] 获取错误提示
	*/
	getErrorMessage(errorCode: number): ErrorMessageInfo {
		if (this.callWithIPC) {
			const ret = electron.ipcRenderer.sendSync('getErrorMessage', {"errorCode":errorCode});
			return ret;
		}

		return this.addOn.getErrorMessage(errorCode);
	}

	/**
	* [en] This interface is used to set need transparent windows in share [cn] 设置屏幕共享中需透明的窗口
	*/
	setTransparentWnd(wndInfo: TransparentWndInfo, onSetTransparentWndResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setTransparentWnd")) {
			if (onSetTransparentWndResult) {
				onSetTransparentWndResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setTransparentWnd");
		this._setTransparentWndCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setTransparentWnd");
			if (onSetTransparentWndResult) {
				onSetTransparentWndResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setTransparentWnd', {"wndInfo":wndInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setTransparentWnd');
				if (onSetTransparentWndResult) {
					onSetTransparentWndResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setTransparentWnd(wndInfo, this._setTransparentWndCallback);
	}

	/**
	* [en] This interface is used to customize the text of the specified scene [cn] 自定义指定场景文案
	*/
	setCustomUIText(customSceneTextInfoList: CustomSceneTextInfoList, onSetCustomUITextResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setCustomUIText")) {
			if (onSetCustomUITextResult) {
				onSetCustomUITextResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setCustomUIText");
		this._setCustomUITextCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setCustomUIText");
			if (onSetCustomUITextResult) {
				onSetCustomUITextResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setCustomUIText', {"customSceneTextInfoList":customSceneTextInfoList}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setCustomUIText');
				if (onSetCustomUITextResult) {
					onSetCustomUITextResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setCustomUIText(customSceneTextInfoList, this._setCustomUITextCallback);
	}

	/**
	* [en] This interface is used to customize the image of the specified scene [cn] 自定义指定场景图标
	*/
	setCustomUIImage(customSceneImageInfoList: CustomSceneImageInfoList, onSetCustomUIImageResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setCustomUIImage")) {
			if (onSetCustomUIImageResult) {
				onSetCustomUIImageResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setCustomUIImage");
		this._setCustomUIImageCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setCustomUIImage");
			if (onSetCustomUIImageResult) {
				onSetCustomUIImageResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setCustomUIImage', {"customSceneImageInfoList":customSceneImageInfoList}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setCustomUIImage');
				if (onSetCustomUIImageResult) {
					onSetCustomUIImageResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setCustomUIImage(customSceneImageInfoList, this._setCustomUIImageCallback);
	}

	/**
	* [en]set virtual background default images [cn]设置虚拟背景默认图片
	*/
	setVirtualBackgroundDefaultImages(defaultImagesInfo: VirtualBackgroundInfo, onSetVirtualBackgroundDefaultImagesResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setVirtualBackgroundDefaultImages")) {
			if (onSetVirtualBackgroundDefaultImagesResult) {
				onSetVirtualBackgroundDefaultImagesResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setVirtualBackgroundDefaultImages");
		this._setVirtualBackgroundDefaultImagesCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setVirtualBackgroundDefaultImages");
			if (onSetVirtualBackgroundDefaultImagesResult) {
				onSetVirtualBackgroundDefaultImagesResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setVirtualBackgroundDefaultImages', {"defaultImagesInfo":defaultImagesInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setVirtualBackgroundDefaultImages');
				if (onSetVirtualBackgroundDefaultImagesResult) {
					onSetVirtualBackgroundDefaultImagesResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setVirtualBackgroundDefaultImages(defaultImagesInfo, this._setVirtualBackgroundDefaultImagesCallback);
	}

}

