import { SDKERR} from './enum';
import { InitInfo} from './model';
import { ILogin } from './ILogin';
import { IConfCtrl } from './IConfCtrl';
import { IConfMgr } from './IConfMgr';
import { IConfig } from './IConfig';
import { IUIConfig } from './IUIConfig';
import { ICall } from './ICall';
import { needCallWithIPC } from './util';
let instance:any = null;
const electron = (typeof window !== 'undefined') ? (window as any).require('electron') : require('electron');

class UISDK {
	loginImpl!: ILogin;
	confCtrlImpl!: IConfCtrl;
	confMgrImpl!: IConfMgr;
	configImpl!: IConfig;
	uIConfigImpl!: IUIConfig;
	callImpl!: ICall;
	addOn!: UISDK;
	_onSdkDisconnectedNotifyCallback!: () => void;
	callWithIPC: boolean = false;
	constructor(nodePath: string) {
		if (!instance) {
			instance = this;
			let uisdkNode = (typeof window !== 'undefined') ? (window as any).require(nodePath) : require(nodePath);
			this.addOn = uisdkNode;
			this.loginImpl = new ILogin(uisdkNode);
			this.confCtrlImpl = new IConfCtrl(uisdkNode);
			this.confMgrImpl = new IConfMgr(uisdkNode);
			this.configImpl = new IConfig(uisdkNode);
			this.uIConfigImpl = new IUIConfig(uisdkNode);
			this.callImpl = new ICall(uisdkNode);
			this.callWithIPC = needCallWithIPC();
		}
		return instance;
	}

	/**
	* [en] Notify of sdk disconnected. [cn] 连接断开通知
	*/
	setOnSdkDisconnectedNotifyCB(onSdkDisconnectedNotify: () => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnSdkDisconnectedNotifyCB', {});
			electron.ipcRenderer.on('onSdkDisconnectedNotify', (event: any, arg: any) => {
				if (onSdkDisconnectedNotify) {
					onSdkDisconnectedNotify()
				}
			});
			return;
		}

		this._onSdkDisconnectedNotifyCallback = () => {
			if (onSdkDisconnectedNotify) {
				onSdkDisconnectedNotify();
			}
		}
		this.addOn.setOnSdkDisconnectedNotifyCB(this._onSdkDisconnectedNotifyCallback);
	}

	/**
	* [en] This interface is used to init sdk. [cn] 初始化
	*/
	init(initInfo: InitInfo): SDKERR {
		this.loginImpl.init();
		this.confCtrlImpl.init();
		this.confMgrImpl.init();
		this.configImpl.init();
		this.uIConfigImpl.init();
		this.callImpl.init();
		if (this.callWithIPC) {
			const ret = electron.ipcRenderer.sendSync('init', {"initInfo":initInfo});
			return ret;
		}

		return this.addOn.init(initInfo);
	}

	/**
	* [en] This interface is used to exit sdk. [cn] 退出
	*/
	exit(): SDKERR {
		if (this.callWithIPC) {
			const ret = electron.ipcRenderer.sendSync('exit', {});
			return ret;
		}

		return this.addOn.exit();
	}

	/**
	* [en] This interface is used to get the login service object. [cn] 获取登录服务对象
	*/
	getLoginApi(): ILogin {
		return this.loginImpl;
	}

	/**
	* [en] This interface is used to get the conf control service object. [cn] 获取会议控制服务对象
	*/
	getConfCtrlApi(): IConfCtrl {
		return this.confCtrlImpl;
	}

	/**
	* [en] This interface is used to get the conf management service object. [cn] 获取会议管理服务对象
	*/
	getConfMgrApi(): IConfMgr {
		return this.confMgrImpl;
	}

	/**
	* [en] This interface is used to get the config service object. [cn] 获取配置服务对象
	*/
	getConfigApi(): IConfig {
		return this.configImpl;
	}

	/**
	* [en] This interface is used to get the UI config service object. [cn] 获取界面配置服务对象
	*/
	getUIConfigApi(): IUIConfig {
		return this.uIConfigImpl;
	}

	/**
	* [en] This interface is used to get the call service object. [cn] 获取呼叫服务对象
	*/
	getCallApi(): ICall {
		return this.callImpl;
	}

}


export default UISDK;