﻿/**
 * @file HwmUISdkCallService.h
 *
 * Copyright(C), 2019-2020, Huawei Tech. Co., Ltd. ALL RIGHTS RESERVED.
 *
 * @brief 呼叫模块接口文件.
 */

#ifndef __HWM_UISDK_CALL_SERVICE_H__
#define __HWM_UISDK_CALL_SERVICE_H__

#include "HwmUISdkCallServiceDef.h"
#include "HwmGlobalErrorCodeDef.h"

BEGIN_HWM_UISDK_NAMESPACE

/**
* @brief
* [en] This class is about call result callback.
* [cn] 呼叫操作类结果回调
*
* @doc:disable
*/
class HWM_API IHwmCallResultCallback
{
public:
    /**
    * @brief 
    * [en] Callback of StartCall interface.
    * [cn] 发起呼叫回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnStartCallResult(SDKERR ret, const HWM_CHAR* reason) {};
};

/**
* @brief
* [en] This class is about call notify callback.
* [cn] 呼叫通知
*
* @doc:disable
*/
class HWM_API IHwmCallNotifyCallback
{
public:
    /**
    * @brief 
    * [en] Notify of call state.
    * [cn] 通话状态通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示通话状态发生变更。
    *
    * @param callStateInfo [type:HwmCallStateInfo*] 
    * [en] Call state info
    * [cn] 通话状态信息
    */
    virtual HWM_VOID OnCallStateChanged(const HwmCallStateInfo* callStateInfo) {};

    /**
    * @brief 
    * [en] Notify of call info
    * [cn] 通话信息通知
    *
    * @detail
    * [en] (only for win)
    * [cn] 收到该通知时，表示通话中的信息发生变更(仅对win平台生效)
    *
    * @param callInfo [type:HwmCallInfo*] 
    * [en] Call info
    * [cn] 通话信息
    *
    * @mac:disable
    * @js:disable
    */
    virtual HWM_VOID OnCallInfoNotify(const HwmCallInfo* callInfo) {};

    /**
    * @brief 
    * [en] Notify of the call incoming
    * [cn] 呼叫来电通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示收到呼叫来电。
    *
    * @param callIncomingInfo [type:HwmCallIncomingInfo*]
    * [en] Call incoming info
    * [cn] 呼叫来电信息
    *
    * @mac:disable
    * @js:disable
    */
    virtual HWM_VOID OnCallIncomingNotify(const HwmCallIncomingInfo* callIncomingInfo) {};

    /**
    * @brief 
    * [en] Notify of the call record when the call ended
    * [cn] 通话结束通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示呼叫已结束，同时返回通话记录信息。
    *
    * @param callRecordInfo [type:HwmCallRecordInfo*]
    * [en] Call record info
    * [cn] 通话记录信息
    */
    virtual HWM_VOID OnCallEndedNotify(const HwmCallRecordInfo* callRecordInfo) {};
};

/**
* @brief 
* [en] This class is about call management object.
* [cn] 呼叫管理对象
*
* @detail
* [en]
* [cn] 获取呼叫管理对象的方法HwmUISDK.getCallApi()
*
* @doc:disable
*
* @GetInstance
*        HWM_UISDK_NAMESPACE::HwmUISDK::GetCallApi()
*/
class IHwmCall
{
public:
    /**
    * @brief
    * [en] This class is about the global callback for the operating results of call management.
    * [cn] 设置呼叫操作类全局回调
    *
    * @param callback [cs_allowNull:N]
    * [en]
    * [cn] 通用配置操作类全局回调
    */
    virtual HWM_VOID SetCallResultCallback(IHwmCallResultCallback* callback) = 0;

    /**
    * @brief
    * [en] This class is about the global callback for the notifications of call management.
    * [cn] 设置呼叫通知类全局回调
    *
    * @param callback [cs_allowNull:N]
    * [en]
    * [cn] 通用配置通知类全局回调
    */
    virtual HWM_VOID SetCallNotifyCallback(IHwmCallNotifyCallback* callback) = 0;

    /**
    * @deprecated
    * [en] Deprecated API.
    * [cn] 废弃接口
    *
    * @brief
    * [en] This interface is used to start call.
    * [cn] 发起呼叫
    *
    * @detail
    * [en]
    * [cn] 该接口用于通过呼叫号码或者呼叫第三方账号发起呼叫。注意事项：1、在登录状态下调用该接口。
    *
    * @param startCallInfo [type:HwmStartCallInfo*]
    * [en] Start call info
    * [cn] 发起呼叫参数信息
    * @doc:disable
    **/
    virtual SDKERR StartCall(const HwmStartCallInfo* startCallInfo) = 0;
};

END_HWM_UISDK_NAMESPACE
#endif /* __HWM_UISDK_CALL_SERVICE_H__ */