﻿/**
* @file HwmUISdkLoginService.h
*
* Copyright(C), 2019-2020, Huawei Tech. Co., Ltd. ALL RIGHTS RESERVED.
*
* @brief 登录模块接口文件.
*/


#ifndef __HWM_UISDK_LOGIN_SERVICE_H__
#define __HWM_UISDK_LOGIN_SERVICE_H__

#include "HwmUISdkLoginServiceDef.h"
#include "HwmGlobalErrorCodeDef.h"

BEGIN_HWM_UISDK_NAMESPACE

/**
* @brief 
* [en] This class is about login result callback.
* [cn] 登录操作类结果回调
*/
class HWM_API IHwmLoginResultCallback
{
public:
    /**
    * @brief 
    * [en] Callback of Login interface.
    * [cn] 登录回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param loginResult [type:HwmLoginResult*]
    * [en] Login result
    * [cn] 登录结果信息
    * 
    * @Callback ["LoginByAccount","LoginByAppId","LoginBySSO"]
    **/
    virtual HWM_VOID OnLoginResult(SDKERR ret, const HWM_CHAR* reason, const HwmLoginResult* loginResult) {};

    /**
    * @brief 
    * [en] Callback of Logout interface.
    * [cn] 登出回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param logoutResult [type:HwmLogoutResult*]
    * [en] Logout result
    * [cn] 登出结果信息
    **/
    virtual HWM_VOID OnLogoutResult(SDKERR ret, const HWM_CHAR* reason, const HwmLogoutResult* logoutResult) {};

    /**
    * @brief 
    * [en] Callback of SetNetworkMode interface.
    * [cn] 设置网络模式结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual void OnSetNetworkModeResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of UploadSelfAvatar interface.
    * [cn] 上传个人头像结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnUploadSelfAvatarResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of DeleteSelfAvatar interface.
    * [cn] 删除个人头像结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * @mac:disable
    * @doc:disable
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnDeleteSelfAvatarResult(SDKERR ret, const HWM_CHAR* reason) {};


    /**
    * @brief [en] Callback of SetServerAddress interface.
    *        [cn] 设置登录地址信息回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetServerAddressResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief [en] Callback of SetProxy interface.
    *        [cn] 设置代理信息回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    **/
    virtual HWM_VOID OnSetProxyResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief [en] Callback of SetCAVerifyInfo interface.
    *        [cn] 设置证书校验信息回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    **/
    virtual HWM_VOID OnSetCAVerifyInfoResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief [en] Callback of SetGlobalGuaranteedIpAddresses interface.
    *        [cn] 设置管理面域名对应的保底ip地址回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    **/
    virtual HWM_VOID OnSetGlobalGuaranteedIpAddressesResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief [en] Callback of GetLoginState interface.
    *        [cn] 获取登录状态接口回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param loginStateInfo [type:HwmLoginStateInfo*]
    * [en] login state info
    * [cn] 登录状态信息
    **/
    virtual HWM_VOID OnGetLoginStateResult(SDKERR ret, const HWM_CHAR* reason, const HwmLoginStateInfo* loginStateInfo) {};

    /**
    * @brief [en] Callback of GetLoginToken interface.
    *        [cn] 获取登录token接口回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param loginTokenInfo [type:HwmLoginTokenInfo*]
    * [en] login token info
    * [cn] 登录token信息
    **/
    virtual HWM_VOID OnGetLoginTokenResult(SDKERR ret, const HWM_CHAR* reason, const HwmLoginTokenInfo* loginTokenInfo) = 0;
};

/**
* @brief 
* [en] This class is about login notify callback.
* [cn] 登录通知
*/
class HWM_API IHwmLoginNotifyCallback
{
public:
    /**
    * @brief 
    * [en] Notify of be kicked out.
    * [cn] 登录被踢通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示当前使用的账号被其他终端踢出，退出登录状态，需要重新登录。
    *
    * @param NULL [cs_allowNull:Y]
    *
    **/
    virtual HWM_VOID OnKickedOutNotify() {};

    /**
    * @brief 
    * [en] Notify of the config of corporation.
    * [cn] 企业配置通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示企业配置信息变更。
    *
    * @param configInfo[type:HwmCorpConfigInfo*]
    * [en] Config data
    * [cn] 配置信息
    **/
    virtual HWM_VOID OnCorpConfigNotify(HwmCorpConfigInfo* configInfo) {};

    /**
    * @brief 
    * [en] Notify of the info of corporation.
    * [cn] 客户企业信息变化通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示客户企业信息变更。
    *
    * @param type[type:HwmCorpInfoNotifyType]                
    * [en] User corp info type
    * [cn] 客户企业信息变化通知类型
    *
    **/
    virtual HWM_VOID OnUserCorpInfoChanged(HwmUserCorpInfoNotifyType type) {};

    /**
    * @brief 
    * [en]This is about login error notify.
    * [cn]登录错误通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示登录失败。
    *
    * @param ret [type:SDKERR]
    * [en]Error code
    * [cn]错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en]Reason
    * [cn]错误描述
    **/
    virtual HWM_VOID OnLoginErrorNotify(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en]This is about share vmr info notify.
    * [cn]共享vmr信息通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示共享vmr信息变更
    *
    * @param shareVmrInfo[type:HwmShareVmrInfo*]
    * [en]Share vmr info
    * [cn]共享vmr信息
    **/
    virtual HWM_VOID OnShareVmrInfoNotify(const HwmShareVmrInfo* shareVmrInfo) {};

    /**
    * @brief
    * [en]This is about login state change notify.
    * [cn]登录状态变化通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示登录状态变更
    *
    * @param loginStateInfo[type:HwmLoginStateInfo*]
    * [en]Login state info
    * [cn]登录状态信息
    **/
    virtual HWM_VOID OnLoginStateChangeNotify(const HwmLoginStateInfo* loginStateInfo) {};
};

/**
* @brief 
* [en] This class is about login management object.
* [cn] 登录管理对象
*
* @detail
* [en] This class is about login management object.
* [cn] 获取登录管理对象的方法HwmUISDK.getLoginApi()
*
* @GetInstance
*        HWM_UISDK_NAMESPACE::HwmUISDK::GetLoginApi()
*/
class IHwmLogin
{
public:
    /**
    * @brief
    * [en] This class is about the global callbacks of setting login operations class.
    * [cn] 设置登录操作类全局回调
    *
    * @param callback [type:IHwmLoginResultCallback*][cs_allowNull:N]
    * [en]
    * [cn] 登录操作类全局回调
    */
    virtual HWM_VOID SetLoginResultCallback(IHwmLoginResultCallback* callback) = 0;

    /**
    * @brief
    * [en] This class is about the global callbacks of login notifications class.
    * [cn] 设置登录通知类全局回调
    *
    * @param callback [type:IHwmLoginNotifyCallback*][cs_allowNull:N]
    * [en]
    * [cn] 登录通知类全局回调
    */
    virtual HWM_VOID SetLoginNotifyCallback(IHwmLoginNotifyCallback* callback) = 0;

    /**
    * @brief 
    * [en] This interface is used to login by account info.
    * [cn] 账号密码登录
    *
    * @detail
    * [en]
    * [cn] 该接口用于账号密码登录。注意事项：1、创建会议，被邀入会等功能只有在登录后才能使用，若不登录，将无法体验完整会议功能。
    *
    * @param accountAuthInfo [type:HwmAccountAuthInfo*][cs_allowNull:N]
    * [en] Account auth info
    * [cn] 账号密码鉴权信息
    *
    * @mac:rename:Login
    **/
    virtual SDKERR LoginByAccount(const HwmAccountAuthInfo* accountAuthInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to login by appId info.
    * [cn] AppId登录
    *
    * @detail
    * [en]
    * [cn] 该接口用于AppId登录。注意事项：1、创建会议，被邀入会等功能只有在登录后才能使用，若不登录，将无法体验完整会议功能。
    *
    * @param appIdAuthInfo [type:HwmAppIdAuthInfo*][cs_allowNull:N]
    * [en] AppId auth info
    * [cn] AppId鉴权信息
    *
    * @mac:rename:Login
    **/
    virtual SDKERR LoginByAppId(const HwmAppIdAuthInfo* appIdAuthInfo) = 0;

    /**
    * @brief 
    * [en] This method is used to login by SSO.
    * [cn] SSO登录
    *
    * @detail
    * [en]
    * [cn] 该接口用于SSO登录。注意事项：1、创建会议，被邀入会等功能只有在登录后才能使用，若不登录，将无法体验完整会议功能。
    *
    * @param ssoAuthInfo [type:HwmSSOAuthInfo*][cs_allowNull:N]
    * [en] SSO auth info
    * [cn] SSO鉴权信息
    *
    * @mac:rename:Login
    **/
    virtual SDKERR LoginBySSO(const HwmSSOAuthInfo* ssoAuthInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to logout sdk.
    * [cn] 退出登录
    *
    * @detail
    * [en]
    * [cn] 该接口用于退出已登录的华为云会议账号。注意事项：1、请在已登录的场景下调用。
    *
    * @param NULL [cs_allowNull:Y]
    **/
    virtual SDKERR Logout() = 0;

    /**
    * @brief
    * [en] This interface is used to set network mode.
    * [cn] 设置网络模式
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置网络模式。注意事项：1、企业内的App无法直接访问公网（即媒体节点下沉和管理节点代理下沉场景）的情况下才需要调用该接口。如华为云会议的媒体节点下沉部署到企业，并且管理节点也是通过企业内的代理访问，需要将网络模式设置成“企业内外接入”；2、该接口非必需调用，若不设置，则使用默认模式。
    *
    * @param networkInfo [type:HwmNetWorkModeInfo*] 
    * [en] Network mode info
    * [cn] 网络模式信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR SetNetworkMode(const HwmNetWorkModeInfo* networkInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to upload self avatar.
    * [cn] 上传头像
    *
    * @detail
    * [en] This interface is used to upload self avatar.
    * [cn] 该接口用于上传当前自己的头像到会议服务器的通讯录。注意事项：1、请在已登录状态下调用；2、上传的图片文件格式仅支持jpg/png格式，大小需要限制在200KB以下；3、为了头像展示效果，建议图片宽高比为1:1
    *
    * @param selfAvatarInfo [type:HwmSelfAvatarInfo*]
    * [en] Self avatar info
    * [cn] 自己的头像信息
    *
    * @norepeat
    **/
    virtual SDKERR UploadSelfAvatar(const HwmSelfAvatarInfo* selfAvatarInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to delete self avatar.
    * [cn] 删除头像。
    *
    * @detail
    * [en] This interface is used to delete self avatar (only for win).
    * [cn] 该接口用于删除自定义的通讯录头像(仅对win平台生效)。注意事项：1、请在已登录状态下调用
    *
    * @norepeat
    * @mac:disable
    * @doc:disable
    **/
    virtual SDKERR DeleteSelfAvatar() = 0;

    /**
    * @brief
    * [en] This interface is used to set guaranteed ip addresses.
    * [cn] 设置(会控)保底ip地址
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置(会控)保底ip地址。注意事项：1、多个地址中间用分号隔开，最多支持4个地址
    *
    * @param guaranteedIpAddresses [type:HWM_CHAR*]
    * [en] Guarantee ip Addresses
    * [cn] 保底ip地址信息
    *
    * @synchronized
    **/
    virtual SDKERR SetGuaranteedIpAddresses(const HWM_CHAR* guaranteedIpAddresses) = 0;
    
    /**
    * @brief
    * [en] This interface is used to set guaranteed ip addresses of global domain.
    * [cn] 设置管理面域名对应的保底ip地址
    *
    * @detail
    * [en] This interface is used to set guaranteed ip addresses of global domain.
    * [cn] 该接口用于设置管理面域名对应的保底ip地址。
    *
    * @param guaranteedIpAddressesParam [type:HwmGuaranteedIpAddressesParam*]
    * [en] Guarantee ip Addresses
    * [cn] 保底ip地址信息
    **/
    virtual SDKERR SetGlobalGuaranteedIpAddresses(const HwmGuaranteedIpAddressesParam* guaranteedIpAddressesParam) = 0;

    /**
    * @brief [en] This interface is used to set login server info.
    *        [cn] 设置登录服务器信息
    *
    * @param serverInfo [type:HwmServerInfo*]
    * [en] Server info
    * [cn] 服务器信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR SetServerAddress(const HwmServerInfo* serverInfo) = 0;

    /**
    * @brief [en] This interface is used to set proxy info.
    *        [cn] 设置代理信息
    *
    * @param proxyInfo [type:HwmProxyInfo*]
    * [en] Proxy info
    * [cn] 代理信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR SetProxy(const HwmProxyInfo* proxyInfo) = 0;

    /**
    * @brief [en] This interface is used to set CA verify info.
    *        [cn] 设置证书校验信息
    *
    * @param CAVerifyInfo [type:HwmCAVerifyInfo*]
    * [en] CA Verify info
    * [cn] 证书校验信息
    **/
    virtual SDKERR SetCAVerifyInfo(const HwmCAVerifyInfo* CAVerifyInfo) = 0;

    /**
    * @brief [en] This interface is used to get login state.
    *        [cn] 获取登录状态
    *
    * @mac:transToSync
    **/
    virtual SDKERR GetLoginState() = 0;

    /**
    * @brief [en] This interface is used to get login token.
    *        [cn] 获取登录token
    *
    * @mac:transToSync
    **/
    virtual SDKERR GetLoginToken() = 0;
};

END_HWM_UISDK_NAMESPACE
#endif /* __HWM_UISDK_LOGIN_SERVICE_H__ */


