import { createSlice } from '@reduxjs/toolkit'
import type { PayloadAction } from '@reduxjs/toolkit'

import { ProxyConfigPolicy, DeployType } from 'hwmsdk-electron/ts/api/Enum'
import { isMac } from 'hwmsdk-electron-demo/src/common/platform'
import { DEMOCONFIG } from 'hwmsdk-electron-demo/src/demoConfig.js'

interface IServer {
  serverAddr: string;
  serverPort: number;
}

interface IProxy {
  proxyConfigPolicy: ProxyConfigPolicy,
  server: string,
  port: number,
  account: string,
  password: string,
}

interface IDeploy {
  serverAddr: string,
  serverPort: number,
  deployType: number,
}

interface rootState {
  server: IServer;
  proxy: IProxy;
  deploy: IDeploy;
}

const initialState: rootState = {
  server: {
    serverAddr: DEMOCONFIG.DEFAULT_MEETING_SERVER,
    serverPort: 443,
  },
  proxy: {
    proxyConfigPolicy: isMac() ? ProxyConfigPolicy.HWM_PROXY_CONFIG_POLICY_CUSTOM : ProxyConfigPolicy.HWM_PROXY_CONFIG_POLICY_OFF,
    server: DEMOCONFIG.DEFAULT_PROXY_SERVER,
    port: 8080,
    account: '',
    password: '',
  },
  deploy: {
    serverAddr: DEMOCONFIG.DEFAULT_MEETING_SERVER,
    serverPort: 443,
    deployType: DeployType.CLOUD_MODE,
  },
}

const settingSlice = createSlice({
  name: 'setting',
  initialState,
  reducers: {
    setServer (state, action: PayloadAction<Partial<IServer>>) {
      state.server = {
        ...state.server,
        ...action.payload,
      }
    },
    setProxy (state, action: PayloadAction<Partial<IProxy>>) {
      state.proxy = {
        ...state.proxy,
        ...action.payload,
      }
    },
    setDeploy (state, action: PayloadAction<Partial<IDeploy>>) {
      state.deploy = {
        ...state.deploy,
        ...action.payload,
      }
    },
  },
})

export default settingSlice

const {
  setServer,
  setProxy,
  setDeploy,
} = settingSlice.actions
let store
let globalDispatch

export function initSettingRedux (_store, dispatch) {
  store = _store
  globalDispatch = dispatch
}

export function updateServer (server: Partial<IServer>) {
  globalDispatch(setServer(server))
}

export function updateProxy (proxy: Partial<IProxy>) {
  globalDispatch(setProxy(proxy))
}

export function updateDeploy (deploy: Partial<IDeploy>) {
  globalDispatch(setDeploy(deploy))
}
