import { UserCorpInfoNotifyType, SDKERR} from './enum';
import { AccountAuthInfo, AppIdAuthInfo, CAVerifyInfo, CorpConfigInfo, GuaranteedIpAddressesParam, LoginResult, LoginStateInfo, LoginTokenInfo, LogoutResult, NetWorkModeInfo, ProxyInfo, SSOAuthInfo, SelfAvatarInfo, ServerInfo, ShareVmrInfo} from './model';
import { needCallWithIPC } from './util';
const electron = (typeof window !== 'undefined') ? (window as any).require('electron') : require('electron');

export class ILogin {
	addOn!: ILogin;
	_onKickedOutNotifyCallback!: () => void;
	_onCorpConfigNotifyCallback!: (configInfo: CorpConfigInfo) => void;
	_onUserCorpInfoChangedCallback!: (type: UserCorpInfoNotifyType) => void;
	_onLoginErrorNotifyCallback!: (ret: SDKERR, reason: string) => void;
	_onShareVmrInfoNotifyCallback!: (shareVmrInfo: ShareVmrInfo) => void;
	_onLoginStateChangeNotifyCallback!: (loginStateInfo: LoginStateInfo) => void;
	_loginByAccountCallback!: (ret: SDKERR, reason: string, loginResult: LoginResult) => void;
	_loginByAppIdCallback!: (ret: SDKERR, reason: string, loginResult: LoginResult) => void;
	_loginBySSOCallback!: (ret: SDKERR, reason: string, loginResult: LoginResult) => void;
	_logoutCallback!: (ret: SDKERR, reason: string, logoutResult: LogoutResult) => void;
	_setNetworkModeCallback!: (ret: SDKERR, reason: string) => void;
	_uploadSelfAvatarCallback!: (ret: SDKERR, reason: string) => void;
	_deleteSelfAvatarCallback!: (ret: SDKERR, reason: string) => void;
	_setGlobalGuaranteedIpAddressesCallback!: (ret: SDKERR, reason: string) => void;
	_setServerAddressCallback!: (ret: SDKERR, reason: string) => void;
	_setProxyCallback!: (ret: SDKERR, reason: string) => void;
	_setCAVerifyInfoCallback!: (ret: SDKERR, reason: string) => void;
	_getLoginStateCallback!: (ret: SDKERR, reason: string, loginStateInfo: LoginStateInfo) => void;
	_getLoginTokenCallback!: (ret: SDKERR, reason: string, loginTokenInfo: LoginTokenInfo) => void;
	_inProcessCallbackSet!: Set<string>;
	callWithIPC: boolean = false;
	constructor(addOn: ILogin) {
		this.addOn = addOn;
		this._inProcessCallbackSet = new Set<string>();
		this.callWithIPC = needCallWithIPC();
	}

	init(): void {
		this._inProcessCallbackSet.clear();
	}

	/**
	* [en] Notify of be kicked out. [cn] 登录被踢通知
	*/
	setOnKickedOutNotifyCB(onKickedOutNotify: () => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnKickedOutNotifyCB', {});
			electron.ipcRenderer.on('onKickedOutNotify', (event: any, arg: any) => {
				if (onKickedOutNotify) {
					onKickedOutNotify()
				}
			});
			return;
		}

		this._onKickedOutNotifyCallback = () => {
			if (onKickedOutNotify) {
				onKickedOutNotify();
			}
		}
		this.addOn.setOnKickedOutNotifyCB(this._onKickedOutNotifyCallback);
	}

	/**
	* [en] Notify of the config of corporation. [cn] 企业配置通知
	*/
	setOnCorpConfigNotifyCB(onCorpConfigNotify: (configInfo: CorpConfigInfo) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnCorpConfigNotifyCB', {});
			electron.ipcRenderer.on('onCorpConfigNotify', (event: any, arg: any) => {
				if (onCorpConfigNotify) {
					onCorpConfigNotify(arg["configInfo"])
				}
			});
			return;
		}

		this._onCorpConfigNotifyCallback = (configInfo: CorpConfigInfo) => {
			if (onCorpConfigNotify) {
				onCorpConfigNotify(configInfo);
			}
		}
		this.addOn.setOnCorpConfigNotifyCB(this._onCorpConfigNotifyCallback);
	}

	/**
	* [en] Notify of the info of corporation. [cn] 客户企业信息变化通知
	*/
	setOnUserCorpInfoChangedCB(onUserCorpInfoChanged: (type: UserCorpInfoNotifyType) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnUserCorpInfoChangedCB', {});
			electron.ipcRenderer.on('onUserCorpInfoChanged', (event: any, arg: any) => {
				if (onUserCorpInfoChanged) {
					onUserCorpInfoChanged(arg["type"])
				}
			});
			return;
		}

		this._onUserCorpInfoChangedCallback = (type: UserCorpInfoNotifyType) => {
			if (onUserCorpInfoChanged) {
				onUserCorpInfoChanged(type);
			}
		}
		this.addOn.setOnUserCorpInfoChangedCB(this._onUserCorpInfoChangedCallback);
	}

	/**
	* [en]This is about login error notify. [cn]登录错误通知
	*/
	setOnLoginErrorNotifyCB(onLoginErrorNotify: (ret: SDKERR, reason: string) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnLoginErrorNotifyCB', {});
			electron.ipcRenderer.on('onLoginErrorNotify', (event: any, arg: any) => {
				if (onLoginErrorNotify) {
					onLoginErrorNotify(arg["ret"], arg["reason"])
				}
			});
			return;
		}

		this._onLoginErrorNotifyCallback = (ret: SDKERR, reason: string) => {
			if (onLoginErrorNotify) {
				onLoginErrorNotify(ret, reason);
			}
		}
		this.addOn.setOnLoginErrorNotifyCB(this._onLoginErrorNotifyCallback);
	}

	/**
	* [en]This is about share vmr info notify. [cn]共享vmr信息通知
	*/
	setOnShareVmrInfoNotifyCB(onShareVmrInfoNotify: (shareVmrInfo: ShareVmrInfo) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnShareVmrInfoNotifyCB', {});
			electron.ipcRenderer.on('onShareVmrInfoNotify', (event: any, arg: any) => {
				if (onShareVmrInfoNotify) {
					onShareVmrInfoNotify(arg["shareVmrInfo"])
				}
			});
			return;
		}

		this._onShareVmrInfoNotifyCallback = (shareVmrInfo: ShareVmrInfo) => {
			if (onShareVmrInfoNotify) {
				onShareVmrInfoNotify(shareVmrInfo);
			}
		}
		this.addOn.setOnShareVmrInfoNotifyCB(this._onShareVmrInfoNotifyCallback);
	}

	/**
	* [en]This is about login state change notify. [cn]登录状态变化通知
	*/
	setOnLoginStateChangeNotifyCB(onLoginStateChangeNotify: (loginStateInfo: LoginStateInfo) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnLoginStateChangeNotifyCB', {});
			electron.ipcRenderer.on('onLoginStateChangeNotify', (event: any, arg: any) => {
				if (onLoginStateChangeNotify) {
					onLoginStateChangeNotify(arg["loginStateInfo"])
				}
			});
			return;
		}

		this._onLoginStateChangeNotifyCallback = (loginStateInfo: LoginStateInfo) => {
			if (onLoginStateChangeNotify) {
				onLoginStateChangeNotify(loginStateInfo);
			}
		}
		this.addOn.setOnLoginStateChangeNotifyCB(this._onLoginStateChangeNotifyCallback);
	}

	/**
	* [en] This interface is used to login by account info. [cn] 账号密码登录
	*/
	loginByAccount(accountAuthInfo: AccountAuthInfo, onLoginResult?: (ret: SDKERR, reason: string, loginResult: LoginResult) => void): void {
		if (this._inProcessCallbackSet.has("loginByAccount")) {
			if (onLoginResult) {
				let nullResult = <LoginResult>{};
				onLoginResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("loginByAccount");
		this._loginByAccountCallback = (ret: SDKERR, reason: string, loginResult: LoginResult) => {
			this._inProcessCallbackSet.delete("loginByAccount");
			if (onLoginResult) {
				onLoginResult(ret, reason, loginResult);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('loginByAccount', {"accountAuthInfo":accountAuthInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('loginByAccount');
				if (onLoginResult) {
					onLoginResult(ret["ret"], ret["reason"], ret["loginResult"]);
				}
			});
		}

		this.addOn.loginByAccount(accountAuthInfo, this._loginByAccountCallback);
	}

	/**
	* [en] This interface is used to login by appId info. [cn] AppId登录
	*/
	loginByAppId(appIdAuthInfo: AppIdAuthInfo, onLoginResult?: (ret: SDKERR, reason: string, loginResult: LoginResult) => void): void {
		if (this._inProcessCallbackSet.has("loginByAppId")) {
			if (onLoginResult) {
				let nullResult = <LoginResult>{};
				onLoginResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("loginByAppId");
		this._loginByAppIdCallback = (ret: SDKERR, reason: string, loginResult: LoginResult) => {
			this._inProcessCallbackSet.delete("loginByAppId");
			if (onLoginResult) {
				onLoginResult(ret, reason, loginResult);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('loginByAppId', {"appIdAuthInfo":appIdAuthInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('loginByAppId');
				if (onLoginResult) {
					onLoginResult(ret["ret"], ret["reason"], ret["loginResult"]);
				}
			});
		}

		this.addOn.loginByAppId(appIdAuthInfo, this._loginByAppIdCallback);
	}

	/**
	* [en] This method is used to login by SSO. [cn] SSO登录
	*/
	loginBySSO(ssoAuthInfo: SSOAuthInfo, onLoginResult?: (ret: SDKERR, reason: string, loginResult: LoginResult) => void): void {
		if (this._inProcessCallbackSet.has("loginBySSO")) {
			if (onLoginResult) {
				let nullResult = <LoginResult>{};
				onLoginResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("loginBySSO");
		this._loginBySSOCallback = (ret: SDKERR, reason: string, loginResult: LoginResult) => {
			this._inProcessCallbackSet.delete("loginBySSO");
			if (onLoginResult) {
				onLoginResult(ret, reason, loginResult);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('loginBySSO', {"ssoAuthInfo":ssoAuthInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('loginBySSO');
				if (onLoginResult) {
					onLoginResult(ret["ret"], ret["reason"], ret["loginResult"]);
				}
			});
		}

		this.addOn.loginBySSO(ssoAuthInfo, this._loginBySSOCallback);
	}

	/**
	* [en] This interface is used to logout sdk. [cn] 退出登录
	*/
	logout(onLogoutResult?: (ret: SDKERR, reason: string, logoutResult: LogoutResult) => void): void {
		if (this._inProcessCallbackSet.has("logout")) {
			if (onLogoutResult) {
				let nullResult = <LogoutResult>{};
				onLogoutResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("logout");
		this._logoutCallback = (ret: SDKERR, reason: string, logoutResult: LogoutResult) => {
			this._inProcessCallbackSet.delete("logout");
			if (onLogoutResult) {
				onLogoutResult(ret, reason, logoutResult);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('logout', {}).then((ret: any) => {
				this._inProcessCallbackSet.delete('logout');
				if (onLogoutResult) {
					onLogoutResult(ret["ret"], ret["reason"], ret["logoutResult"]);
				}
			});
		}

		this.addOn.logout(this._logoutCallback);
	}

	/**
	* [en] This interface is used to set network mode. [cn] 设置网络模式
	*/
	setNetworkMode(networkInfo: NetWorkModeInfo, onSetNetworkModeResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setNetworkMode")) {
			if (onSetNetworkModeResult) {
				onSetNetworkModeResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setNetworkMode");
		this._setNetworkModeCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setNetworkMode");
			if (onSetNetworkModeResult) {
				onSetNetworkModeResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setNetworkMode', {"networkInfo":networkInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setNetworkMode');
				if (onSetNetworkModeResult) {
					onSetNetworkModeResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setNetworkMode(networkInfo, this._setNetworkModeCallback);
	}

	/**
	* [en] This interface is used to upload self avatar. [cn] 上传头像
	*/
	uploadSelfAvatar(selfAvatarInfo: SelfAvatarInfo, onUploadSelfAvatarResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("uploadSelfAvatar")) {
			if (onUploadSelfAvatarResult) {
				onUploadSelfAvatarResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("uploadSelfAvatar");
		this._uploadSelfAvatarCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("uploadSelfAvatar");
			if (onUploadSelfAvatarResult) {
				onUploadSelfAvatarResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('uploadSelfAvatar', {"selfAvatarInfo":selfAvatarInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('uploadSelfAvatar');
				if (onUploadSelfAvatarResult) {
					onUploadSelfAvatarResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.uploadSelfAvatar(selfAvatarInfo, this._uploadSelfAvatarCallback);
	}

	/**
	* [en] This interface is used to delete self avatar. [cn] 删除头像。
	*/
	deleteSelfAvatar(onDeleteSelfAvatarResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("deleteSelfAvatar")) {
			if (onDeleteSelfAvatarResult) {
				onDeleteSelfAvatarResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("deleteSelfAvatar");
		this._deleteSelfAvatarCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("deleteSelfAvatar");
			if (onDeleteSelfAvatarResult) {
				onDeleteSelfAvatarResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('deleteSelfAvatar', {}).then((ret: any) => {
				this._inProcessCallbackSet.delete('deleteSelfAvatar');
				if (onDeleteSelfAvatarResult) {
					onDeleteSelfAvatarResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.deleteSelfAvatar(this._deleteSelfAvatarCallback);
	}

	/**
	* [en] This interface is used to set guaranteed ip addresses. [cn] 设置(会控)保底ip地址
	*/
	setGuaranteedIpAddresses(guaranteedIpAddresses: string): SDKERR {
		if (this.callWithIPC) {
			const ret = electron.ipcRenderer.sendSync('setGuaranteedIpAddresses', {"guaranteedIpAddresses":guaranteedIpAddresses});
			return ret;
		}

		return this.addOn.setGuaranteedIpAddresses(guaranteedIpAddresses);
	}

	/**
	* [en] This interface is used to set guaranteed ip addresses of global domain. [cn] 设置管理面域名对应的保底ip地址
	*/
	setGlobalGuaranteedIpAddresses(guaranteedIpAddressesParam: GuaranteedIpAddressesParam, onSetGlobalGuaranteedIpAddressesResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setGlobalGuaranteedIpAddresses")) {
			if (onSetGlobalGuaranteedIpAddressesResult) {
				onSetGlobalGuaranteedIpAddressesResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setGlobalGuaranteedIpAddresses");
		this._setGlobalGuaranteedIpAddressesCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setGlobalGuaranteedIpAddresses");
			if (onSetGlobalGuaranteedIpAddressesResult) {
				onSetGlobalGuaranteedIpAddressesResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setGlobalGuaranteedIpAddresses', {"guaranteedIpAddressesParam":guaranteedIpAddressesParam}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setGlobalGuaranteedIpAddresses');
				if (onSetGlobalGuaranteedIpAddressesResult) {
					onSetGlobalGuaranteedIpAddressesResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setGlobalGuaranteedIpAddresses(guaranteedIpAddressesParam, this._setGlobalGuaranteedIpAddressesCallback);
	}

	/**
	* [en] This interface is used to set login server info. [cn] 设置登录服务器信息
	*/
	setServerAddress(serverInfo: ServerInfo, onSetServerAddressResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setServerAddress")) {
			if (onSetServerAddressResult) {
				onSetServerAddressResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setServerAddress");
		this._setServerAddressCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setServerAddress");
			if (onSetServerAddressResult) {
				onSetServerAddressResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setServerAddress', {"serverInfo":serverInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setServerAddress');
				if (onSetServerAddressResult) {
					onSetServerAddressResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setServerAddress(serverInfo, this._setServerAddressCallback);
	}

	/**
	* [en] This interface is used to set proxy info. [cn] 设置代理信息
	*/
	setProxy(proxyInfo: ProxyInfo, onSetProxyResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setProxy")) {
			if (onSetProxyResult) {
				onSetProxyResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setProxy");
		this._setProxyCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setProxy");
			if (onSetProxyResult) {
				onSetProxyResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setProxy', {"proxyInfo":proxyInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setProxy');
				if (onSetProxyResult) {
					onSetProxyResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setProxy(proxyInfo, this._setProxyCallback);
	}

	/**
	* [en] This interface is used to set CA verify info. [cn] 设置证书校验信息
	*/
	setCAVerifyInfo(CAVerifyInfo: CAVerifyInfo, onSetCAVerifyInfoResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("setCAVerifyInfo")) {
			if (onSetCAVerifyInfoResult) {
				onSetCAVerifyInfoResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("setCAVerifyInfo");
		this._setCAVerifyInfoCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("setCAVerifyInfo");
			if (onSetCAVerifyInfoResult) {
				onSetCAVerifyInfoResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('setCAVerifyInfo', {"CAVerifyInfo":CAVerifyInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('setCAVerifyInfo');
				if (onSetCAVerifyInfoResult) {
					onSetCAVerifyInfoResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.setCAVerifyInfo(CAVerifyInfo, this._setCAVerifyInfoCallback);
	}

	/**
	* [en] This interface is used to get login state. [cn] 获取登录状态
	*/
	getLoginState(onGetLoginStateResult?: (ret: SDKERR, reason: string, loginStateInfo: LoginStateInfo) => void): void {
		if (this._inProcessCallbackSet.has("getLoginState")) {
			if (onGetLoginStateResult) {
				let nullResult = <LoginStateInfo>{};
				onGetLoginStateResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("getLoginState");
		this._getLoginStateCallback = (ret: SDKERR, reason: string, loginStateInfo: LoginStateInfo) => {
			this._inProcessCallbackSet.delete("getLoginState");
			if (onGetLoginStateResult) {
				onGetLoginStateResult(ret, reason, loginStateInfo);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('getLoginState', {}).then((ret: any) => {
				this._inProcessCallbackSet.delete('getLoginState');
				if (onGetLoginStateResult) {
					onGetLoginStateResult(ret["ret"], ret["reason"], ret["loginStateInfo"]);
				}
			});
		}

		this.addOn.getLoginState(this._getLoginStateCallback);
	}

	/**
	* [en] This interface is used to get login token. [cn] 获取登录token
	*/
	getLoginToken(onGetLoginTokenResult?: (ret: SDKERR, reason: string, loginTokenInfo: LoginTokenInfo) => void): void {
		if (this._inProcessCallbackSet.has("getLoginToken")) {
			if (onGetLoginTokenResult) {
				let nullResult = <LoginTokenInfo>{};
				onGetLoginTokenResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
			};
			return;
		}
		this._inProcessCallbackSet.add("getLoginToken");
		this._getLoginTokenCallback = (ret: SDKERR, reason: string, loginTokenInfo: LoginTokenInfo) => {
			this._inProcessCallbackSet.delete("getLoginToken");
			if (onGetLoginTokenResult) {
				onGetLoginTokenResult(ret, reason, loginTokenInfo);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('getLoginToken', {}).then((ret: any) => {
				this._inProcessCallbackSet.delete('getLoginToken');
				if (onGetLoginTokenResult) {
					onGetLoginTokenResult(ret["ret"], ret["reason"], ret["loginTokenInfo"]);
				}
			});
		}

		this.addOn.getLoginToken(this._getLoginTokenCallback);
	}

}

