//
//  HWMSDKDeviceInformation.h
//  HWMNativeSDK
//
//  Created by guomeng on 2021/5/20.
//

#import <Foundation/Foundation.h>
#import "HWMSDKDeviceCpuUsage.h"

NS_ASSUME_NONNULL_BEGIN
extern NSString *const HWMSDKDisplayConfigurationChangedNotify;

//字节序
typedef NS_ENUM (NSInteger, HWMSDKByteOrder) {
    HWMSDKByteOrderLittleEndian,
    HWMSDKByteOrderBigEndian,
};

//系统版本
typedef struct {
    /**
     Major OS version number. For e.g. 10.8.2 => 10
     */
    NSUInteger major;

    /**
     Minor OS version number. For e.g. 10.8.2 => 8
     */
    NSUInteger minor;

    /**
     Patch OS version number. For e.g. 10.8.2 => 2
     */
    NSUInteger patch;
} HWMSDKOSVersion;

/**
 Makes a HWMSDKOSVersion struct.
 */
inline static HWMSDKOSVersion HWMSDKOSVersionMake(NSUInteger major, NSUInteger minor,  NSUInteger patch)
{
    return (HWMSDKOSVersion) { major, minor, patch };
}

/**
 Makes a HWMSDKOSVersion struct by parsing a NSString.

 e.g.   @"8.2.3"    -> HWMSDKOSVersionMake(8,2,3)
        @"9.1"      -> HWMSDKOSVersionMake(9,1,0)
 */
inline static HWMSDKOSVersion HWMSDKOSVersionFromString(NSString *versionString)
{
    NSArray *components = [versionString componentsSeparatedByString:@"."];
    NSInteger major = components.count >= 1 ? [components[0] integerValue] : 0;
    NSInteger minor = components.count >= 2 ? [components[1] integerValue] : 0;
    NSInteger patch = components.count >= 3 ? [components[2] integerValue] : 0;

    return HWMSDKOSVersionMake(major, minor, patch);
}

//设备版本
typedef struct {
    /**
     Major device model. e.g. 13 for iMac13,2
     */
    NSUInteger major;

    /**
     Minor device model. e.g. 2 for iMac13,2
     */
    NSUInteger minor;
} HWMSDKDeviceVersion;

/**
 Makes a HWMSDKDeviceVersion struct.
 */
inline static HWMSDKDeviceVersion HWMSDKDeviceVersionMake(NSUInteger major, NSUInteger minor)
{
    return (HWMSDKDeviceVersion) { major, minor };
}

//CPU信息
typedef struct {
    /**
     CPU frequency, in GHz.

     Warning: Might not be (=probably won't be) available on all iOS devices.
     */
    CGFloat frequency;                  // GHz (giga)

    /**
    物理核数
    */
    NSUInteger numberOfPhysicalCores;
    /**
     Number of logical cores the CPU has.
     */
    NSUInteger numberOfLogicalCores;

    /**
     CPU's l2 cache size, in KB.
     */
    CGFloat l2CacheSize;                // KB (kibi)
} HWMSDKCPUInfo;

/**
 Makes a HWMSDKCPUInfo struct.
 */
inline static HWMSDKCPUInfo HWMSDKCPUInfoMake(CGFloat frequency, NSUInteger numberOfPhysicalCores, NSUInteger numberOfLogicalCores, CGFloat l2CacheSize)
{
    return (HWMSDKCPUInfo) { frequency, numberOfPhysicalCores, numberOfLogicalCores, l2CacheSize };
}

//产品类型
typedef NS_ENUM (NSInteger, HWMSDKDeviceFamily) {
    HWMSDKDeviceFamilyUnknown = 0,
    HWMSDKDeviceFamilyiMac,
    HWMSDKDeviceFamilyiMacPro,
    HWMSDKDeviceFamilyMacMini,
    HWMSDKDeviceFamilyMacPro,
    HWMSDKDeviceFamilyMacBook,
    HWMSDKDeviceFamilyMacBookAir,
    HWMSDKDeviceFamilyMacBookPro,
    HWMSDKDeviceFamilyXserve,
};

//显示信息
typedef struct {
    /**
     The main display's resolution.
     */
    CGSize resolution;

    /**
     The display's pixel density in ppi (pixels per inch).
     */
    CGFloat pixelsPerInch;
} HWMSDKDisplayInfo;

//cpu等级划分
typedef NS_ENUM (NSUInteger, HWMSDKDeviceCpuLevel) {
    HWMSDKDeviceCpuLevelLow = 0,
    HWMSDKDeviceCpuLevelMiddle = 1,
    HWMSDKDeviceCpuLevelHigh = 2
};

//应用转写状态
typedef NS_ENUM(NSInteger, HWMSDKDeviceProcessTranslateState) {
    HWMSDKDeviceProcessTranslateStateError = -1, //获取时候发生错误
    HWMSDKDeviceProcessTranslateStateNative = 0, //未转译
    HWMSDKDeviceProcessTranslateStateTranslated = 1 //转译
};

/**
 Makes a GBDisplayInfo struct.
 */
inline static HWMSDKDisplayInfo HWMSDKDisplayInfoMake(CGSize resolution, CGFloat pixelsPerInch)
{
    return (HWMSDKDisplayInfo) { resolution, pixelsPerInch };
}

@interface HWMSDKDeviceInformation : NSObject
/**
 The identify model string, e.g. "MacBookPro16,1".
 */
@property (nonatomic, readonly, copy) NSString *identifyModelString;

/// 设备名字  MacBook Pro (16-inch, 2019)
@property (nonatomic, readonly, copy) NSString *productName;

/// 电脑的名字 比如 xxx的MacBook-Pro.local
@property (nonatomic, readonly, copy) NSString *nodeName;
/**
 The device family. e.g. HWMSDKDeviceFamily
 */
@property (assign, nonatomic, readonly) HWMSDKDeviceFamily family;

/**
 Information about the CPU.
 */
@property (assign, nonatomic, readonly) HWMSDKCPUInfo cpuInfo;

/// 处理器名称
@property (nonatomic, readonly, copy) NSString *cpuBrandInfo;

/// CPU架构 比如 x86_64
@property (nonatomic, readonly, copy) NSString *cpuArch;

/// App是否经过转译
@property (nonatomic, readonly, assign) HWMSDKDeviceProcessTranslateState processTranslateState;

/**
 Amount of physical memory (RAM) available to the system, in HWMSDK.  GB
 */
@property (assign, nonatomic, readonly) CGFloat physicalMemory;

/**
 Information about the system's OS. e.g. {10, 8, 2}.
 */
@property (assign, nonatomic, readonly) HWMSDKOSVersion osVersion;

/**
 main monitor's display information
*/
@property (assign, nonatomic, readonly) HWMSDKDisplayInfo mainDisplayInfo;

/**
all monitors's resolution information
*/
@property (assign, nonatomic, readonly) CGRect *allDisplayInfo;

@property (nonatomic, readonly, copy) NSArray *graphicsInfo;

//cpu使用率
@property (nonatomic, strong)HWMSDKDeviceCpuUsage *cpuUsage;

/**
 Returns information about the device the current app is running on.
 */
+ (HWMSDKDeviceInformation *)currentDeviceInfo;

/**
 Check if the OS version is equal to or higher than version.
 */
- (BOOL)isOperatingSystemAtLeastOSVersion:(HWMSDKOSVersion)version;

/**
 Check if the OS version is equal to or higher than versionString, where versionString gets parsed into a GBOSVersion.

 e.g.   @"8.2.3"    -> GBOSVersionMake(8,2,3)
        @"9.1"      -> GBOSVersionMake(9,1,0)
 */
- (BOOL)isOperatingSystemAtLeastVersion:(NSString *)versionString;

//获取所有ip地址
- (nullable NSDictionary *)getAllIPAddresses;

//获取当前使用的ip地址，优先级 网卡->蜂窝->VPN
- (NSString *)getIPAddress:(BOOL)preferIPv4;

+ (NSDictionary *)getSystemProxySettings;

+ (nullable NSString *)getSystemProxyAddress;

+ (int)getSystemProxyPort;

+ (NSString *)getDNSDomainName;

- (nullable NSString *) getHardwareSerialNumber;

//cpu整体的使用率
- (CGFloat)cpuUsgPercent;
//本app的cpu占用率
- (CGFloat)appCpuUsage;
//CPU等级
- (HWMSDKDeviceCpuLevel)getCpuPerformanceLevel;

/// 当前电量百分比
+ (double)currentChargePercent;
/// 当前网络访问类型，是wifi 还是网线
+ (NSString *)currentNetAccessType;
/// 设置设备信息:SDK判断设备性能
+ (void)updateDeviceInfo;

@end

NS_ASSUME_NONNULL_END
