/**
* @file HwmUISdkConfigService.h
*
* Copyright(C), 2019-2020, Huawei Tech. Co., Ltd. ALL RIGHTS RESERVED.
*
* @brief 登录模块接口文件.
*/


#ifndef __HWM_UISDK_CONFIG_SERVICE_H__
#define __HWM_UISDK_CONFIG_SERVICE_H__

#include "HwmUISdkConfigServiceDef.h"
#include "HwmGlobalErrorCodeDef.h"

BEGIN_HWM_UISDK_NAMESPACE

/**
* @brief 
* [en] This class is about config result callback.
* [cn] 通用配置操作类结果回调
*/
class HWM_API IHwmConfigResultCallback
{
public:
    /**
    * @brief 
    * [en] Callback of EnableCapability interface.
    * [cn] 功能开关配置接口回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnEnableCapabilityResult(SDKERR ret, const HWM_CHAR* reason) {};
 
    /**
    * @brief 
    * [en] Callback of SetLanguage interface.
    * [cn] 设置语言接口回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetLanguageResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of SetShareScreenPermission interface.
    * [cn] 设置共享权限接口回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetShareScreenPermissionResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of set file save path.
    * [cn] 设置用户保存路径结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetUserSavePathResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of set file save path.
    * [cn] 设置用户保存路径结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param pathInfo [type:HwmUserSavePathInfo*]
    * [en] Save path info
    * [cn] 保存路径信息
    *
    * @mac:disable
    **/
    virtual HWM_VOID OnGetUserSavePathResult(SDKERR ret, const HWM_CHAR* reason, const HwmUserSavePathInfo* pathInfo) {};

    /**
    * @brief 
    * [en] Callback of SetShareStrategy interface.
    * [cn] 设置屏幕共享策略接口回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetShareStrategyResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of SetAudioRawDataOutputConfig interface.
    * [cn] 开启或关闭接收音频流接口回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetAudioRawDataOutputConfigResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of SetAnswerType interface.
    * [cn] 设置来电应答结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetAnswerTypeResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of SetDeviceConfigForConfIncoming interface.
    * [cn] 设置接听会议来电时麦克风/摄像头开关状态结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en]Reason
    * [cn]错误描述
    **/
    virtual HWM_VOID OnSetDeviceConfigForConfIncomingResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of GetConfPstnAccessNumber interface.
    * [cn] 获取会议pstn接入号码结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param pstnAccessNumberInfo [type:HwmPstnAccessNumberInfo*]
    * [en] Pstn access number info.
    * [cn] 会议pstn接入号码信息
    **/
    virtual HWM_VOID OnGetConfPstnAccessNumberResult(SDKERR ret, const HWM_CHAR* reason, const HwmPstnAccessNumberInfo* pstnAccessNumberInfo) {};

    /**
    * @brief 
    * [en] Callback of AVDeviceCtrl interface.
    * [cn] 音视频开关控制接口回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param deviceCtrlInfo [type:HwmDeviceCtrlInfo*]                 
    * [en] Device ctrl info.
    * [cn] 设备操作信息
    **/
    virtual HWM_VOID OnAVDeviceCtrlResult(SDKERR ret, const HWM_CHAR* reason, const HwmDeviceCtrlInfo* deviceCtrlInfo) {};

    /**
    * @brief 
    * [en] Callback of DisableVoicePrompts interface.
    * [cn] 禁用语音提示接口结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnDisableVoicePromptsResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief [en] Callback of AgreeVoiceprintStatement interface.
    *        [cn] 签署声纹功能声明回调
    *
    * @detail
    * [en] (only for win)
    * [cn] 签署声纹功能声明结果回调。(仅对win平台生效)
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @mac:disable
    * @doc:disable
    **/
    virtual HWM_VOID OnAgreeVoiceprintStatementResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief [en] Callback of ConfigVoiceprintRecordWindow interface.
    *        [cn] 定制声纹录入界面回调
    *
    * @detail
    * [en] (only for win)
    * [cn] 定制声纹录入界面回调。(仅对win平台生效)
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @mac:disable
    * @js:disable
    **/
    virtual HWM_VOID OnConfigVoiceprintRecordWindowResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief [en] Callback of ShowVoiceprintRecordWindow interface.
    *        [cn] 显示声纹录入窗口回调
    *
    * @detail
    * [en] (only for win)
    * [cn] 显示声纹录入窗口回调。(仅对win平台生效)
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @mac:disable
    * @doc:disable
    **/
    virtual HWM_VOID OnShowVoiceprintRecordWindowResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en] Callback of set log save path.
    * [cn] 设置日志路径结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetLogPathResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief [en] Callback of SetTransparentWnd interface.
    *        [cn] 设置透明窗口接口回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetTransparentWndResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief [en] Callback of SetCustomUIText interface.
    *        [cn] 设置用户自定义场景文案回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetCustomUITextResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief [en] Callback of SetCustomUIImage interface.
    *        [cn] 设置用户自定义场景图标回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @mac:disable
    **/
    virtual HWM_VOID OnSetCustomUIImageResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief [en] Callback of SetVirtualBackgroundDefaultImages interface.
    *        [cn] 设置虚拟背景默认图片接口回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetVirtualBackgroundDefaultImagesResult(SDKERR ret, const HWM_CHAR* reason) {};
};

/**
* @brief 
* [en] This class is about config notify callback.
* [cn] 通用配置通知
*/
class HWM_API IHwmConfigNotifyCallback
{
public:
    /**
    * @brief 
    * [en] Notify of AVDevice control state
    * [cn] 音视频设备控制状态变化通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示会议中麦克风、扬声器或摄像头的状态信息发生变更。
    *
    * @param info [type:HwmDeviceStateInfo*]
    * [en] Device state info.
    * [cn] 设备状态信息
    *
    * @doc:disable
    */
    virtual HWM_VOID OnAVDeviceCtrlStateChanged(const HwmDeviceStateInfo* info) {};

    /**
    * @brief 
    * [en] Notify of audio famme
    * [cn] 上报音频码流帧数据通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，表示开始上报音频码流帧数据。(win平台只支持32位)
    *
    * @param audioFrameData [type:HwmAudioFrameData*]
    * [en] Audio stream frame data
    * [cn] 音频码流帧数据
    *
    * @synchronized
    */
    virtual HWM_VOID OnAudioFrameNotify(const HwmAudioFrameData* audioFrameData) {};

    /**
    * @brief [en] Notify of show voiceprint statement
    *        [cn] 声纹降噪功能声明显示通知
    *
    * @detail
    * [en] (only for win)
    * [cn] 收到该通知时，表示需要打开功能声明。(仅对win平台生效)
    *
    * @param info [type:HwmShowVoiceprintStatementInfo*]        
    * [en] info of show voiceprint statement
    * [cn] 打开声纹降噪功能声明信息
    *
    * @mac:disable
    * @doc:disable
    */
    virtual HWM_VOID OnShowVoiceprintStatement(const HwmShowVoiceprintStatementInfo* info) {};

    /**
    * @brief[en] User data path change notification
    *       [cn] 用户数据路径变化通知
    *
    * @detail
    * [en] Notify of the user data path changed. (only for windows)
    * [cn] 收到该通知时，表示用户数据路径发生变化。（仅对Windows平台生效）
    *
    * @param userSavePathInfo [type:HwmUserSavePathInfo*]
    * [en] Path Info
    * [cn] 路径信息
    *
    * @mac:disable
    */
    virtual HWM_VOID OnUserSavePathChanged(const HwmUserSavePathInfo* userSavePathInfo) {};
};

/**
* @brief 
* [en] This class is about config management object.
* [cn] 通用配置管理对象
*
* @detail
* [en] This class is about login management object.
* [cn] 获取配置管理对象的方法HwmUISDK.getConfigApi()
*
* @GetInstance
*        HWM_UISDK_NAMESPACE::HwmUISDK::GetConfigApi()
*/
class IHwmConfig
{
public:
    /**
    * @brief
    * [en] This class is about the global callback for the operating results of config management.
    * [cn] 设置通用配置操作类全局回调
    *
    * @param callback [type:IHwmConfigResultCallback*][cs_allowNull:N]
    * [en]
    * [cn] 通用配置操作类全局回调
    */
    virtual HWM_VOID SetConfigResultCallback(IHwmConfigResultCallback* callback) = 0;

    /**
    * @brief
    * [en] This class is about the global callback for the notifications of config management.
    * [cn] 设置通用配置通知类全局回调
    *
    * @param callback [type:IHwmConfigNotifyCallback*][cs_allowNull:N]
    * [en]
    * [cn] 通用配置通知类全局回调
    */
    virtual HWM_VOID SetConfigNotifyCallback(IHwmConfigNotifyCallback* callback) = 0;

    /**
    * @brief 
    * [en] This interface is used to enable capability
    * [cn] 功能配置开关
    *
    * @detail
    * [en]
    * [cn] 该接口用于修改会议相关功能配置开关。注意事项：1、枚举中的功能项均有默认值，若对开关状态不做要求，可以不必调用该接口。
    *
    * @param capabilityInfo [type:HwmCapabilityInfo*]
    * [en] Capability config info
    * [cn] 功能配置信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR EnableCapability(const HwmCapabilityInfo* capabilityInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to set language.
    * [cn] 设置语言
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置语言，默认是中文。注意事项：1、实现自定义语言，windows平台要求在 \HwmSdk\sdkResources\language\ 目录下创建对应的国际化文件，如新增阿拉伯语则可以创建 \ar\temp.js；mac平台要求在 /HWMUISDK.framework/Versions/A/Frameworks/WKLanguage.framework/Resources/ 目录下创建对应的国际化文件，如新增阿拉伯语则可以创建 ar.lproj/Localizable.strings。通过该接口设置的语言应与系统当前语言保持一致。
    *
    * @param languageInfo [type:HwmLanguageInfo*]
    * [en] HwmLanguage type info
    * [cn] 语言类型信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR SetLanguage(const HwmLanguageInfo* languageInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to set file save path.
    * [cn] 设置用户保存路径
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置用户文件（包括本地录制文件）的保存路径。注意事项：1、该接口非必需调用，若不设置，则使用默认路径。
    *
    * @param userSavePathInfo [type:HwmUserSavePathInfo*]
    * [en] Save path info
    * [cn] 保存路径信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR SetUserSavePath(const HwmUserSavePathInfo* userSavePathInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to get file save path.
    * [cn] 获取用户保存路径
    *
    * @detail
    * [en]
    * [cn] 该接口用于获取用户文件（包括本地录制文件）的保存路径。
    *
    * @param pathType [type:HwmSavePathType*]
    * [en] Save path type
    * [cn] 保存路径类型
    *
    * @mac:disable
    **/
    virtual SDKERR GetUserSavePath(HwmSavePathType pathType) = 0;

    /**
    * @brief 
    * [en] This interface is used to set share strategy.
    * [cn] 设置共享策略
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置屏幕共享的策略是清晰度优先还是流畅度优先。
    *
    * @param shareStrategyInfo [type:HwmShareStrategyInfo*]
    * [en] Share strategy info
    * [cn] 屏幕共享策略信息
    *
    * @mac:disable
    * @doc:disable
    **/
    virtual SDKERR SetShareStrategy(const HwmShareStrategyInfo* shareStrategyInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to set audio raw data output config.
    * [cn] 设置开启或关闭音频码流
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置开启或关闭音频码流。
    *
    * @param audioRawDataConfigInfo [type:HwmAudioRawDataConfigInfo*]
    * [en] Audio rawData config info
    * [cn] 音频码流配置信息
    * @mac:transToSync
    * 
    **/
    virtual SDKERR SetAudioRawDataOutputConfig(const HwmAudioRawDataConfigInfo* audioRawDataConfigInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to set call incoming answer type.
    * [cn] 设置来电应答方式
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置来电应答: 原始弹框、自动拒绝或自动接听。
    *
    * @param inComingAnswerTypeInfo [type:HwmInComingAnswerTypeInfo*]
    * [en] Incoming answer info
    * [cn] 来电应答信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR SetAnswerType(const HwmInComingAnswerTypeInfo* inComingAnswerTypeInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to set device config for conf incoming.
    * [cn] 会议来电时设备开关配置
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置接听会议来电时设备开关状态，包括麦克风和摄像头。
    *
    * @param deviceConfig [type:HwmDeviceConfig*]
    * [en] Device config info
    * [cn] 设备配置信息
    *
	* @mac:transToSync
    **/
    virtual SDKERR SetDeviceConfigForConfIncoming(const HwmDeviceConfig* deviceConfig) = 0;

    /**
    * @brief 
    * [en] This interface is used to get conf pstn access number.
    * [cn] 获取会议pstn接入号码
    *
    * @detail
    * [en]
    * [cn] 该接口用于获取会议pstn接入号码。
    *
    * @param [cs_allowNull:Y]
    *
    * @mac:transToSync
    **/
    virtual SDKERR GetConfPstnAccessNumber() = 0;

    /**
    * @brief 
    * [en] This interface is used to AVDeviceCtrl.
    * [cn] 音视频设备控制功能
    *
    * @detail
    * [en]
    * [cn] 该接口用于音视频设备控制功能。
    *
    * @param deviceStateInfo [type:HwmDeviceStateInfo*]
    * [en] Device config info
    * [cn] 设备配置信息
    * 
    * @doc:disable
    * @mac:transToSync
    **/
    virtual SDKERR AVDeviceCtrl(const HwmDeviceStateInfo* deviceStateInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to disable voice prompts.
    * [cn] 禁用提示音
    *
    * @detail
    * [en] This interface is used to disable voice prompts.
    * [cn] 该接口用于禁用提示音。
    *
    * @param voicePromptsInfo [type:HwmVoicePromptsInfo*]      
    * [en] Voice prompts info
    * [cn] 语音提示信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR DisableVoicePrompts(const HwmVoicePromptsInfo* voicePromptsInfo) = 0;

    /**
    * @brief
    * [en] This interface is used to set share permission.
    * [cn] 设置共享权限
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置共享权限，默认是允许共享。
    *
    * @param permission [type:HwmSharePermission*]
    * [en] share permission type info
    * [cn] 权限类型信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR SetShareScreenPermission(const HwmSharePermission * permission) = 0;

    /**
    * @brief [en] This interface is used to set voiceprint statement agreement.
    *        [cn] 签署声纹功能声明
    *
    * @detail
    * [en] This interface is used to set voiceprint statement.(only for win)
    * [cn] 该接口用于签署声纹功能声明。(仅对win平台生效)
    *
    * @param [in] HWM_BOOL isAgreed                  [en] Is agreed
    *                                                [cn] 是否同意
    *
    * @mac:disable
    * @doc:disable
    **/
    virtual SDKERR AgreeVoiceprintStatement(HWM_BOOL isAgreed) = 0;

    /**
    * @brief
    * [en] This interface is config voiceprint record window.
    * [cn] 定制声纹录入窗口
    *
    * @detail
    * [en] config voiceprint record window(only for win)
    * [cn] 该接口用于定制声纹录入窗口。(仅对win平台生效)
    *
    * @param configInfo [type:HwmVoiceprintRecordWindowConfig*]
    * [en] Voiceprint window config info
    * [cn] 声纹录入窗口定制信息
    *
    * @mac:disable
    * @js:disable
    **/
    virtual SDKERR ConfigVoiceprintRecordWindow(const HwmVoiceprintRecordWindowConfig* configInfo) = 0;

    /**
    * @brief
    * [en] This interface is used to show voiceprint record window.
    * [cn] 显示声纹录入窗口接口
    *
    * @detail
    * [en] show voiceprint record window(only for win)
    * [cn] 该接口用于显示声纹录入窗口。(仅对win平台生效)
    *
    * @param isShow [type:HWM_BOOL]
    * [en] is show
    * [cn] 是否显示
    *
    * @mac:disable
    * @doc:disable
    **/
    virtual SDKERR ShowVoiceprintRecordWindow(HWM_BOOL isShow) = 0;

    /**
    * @brief
    * [en] This interface is used to set log save path.
    * [cn] 设置日志保存路径
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置日志的保存路径。注意事项：1、该接口非必需调用，若不设置，则使用默认路径。
    *
    * @param logPathInfo [type:HwmLogPathInfo*]
    * [en] Log path info
    * [cn] 日志路径信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR SetLogPath(const HwmLogPathInfo* logPathInfo) = 0;

    /**
    * @brief
    * [en] This interface is used to get error message.
    * [cn] 获取错误提示
    *
    * @param errorCode [type:HWM_INT32]
    * [en] Error code
    * [cn] 错误码
    *
    * @param errorMessageInfo [type:HwmErrorMessageInfo*]
    * [en] error message
    * [cn] 错误提示
    *
    * @synchronized
    * @paramOut
    **/
    virtual SDKERR GetErrorMessage(HWM_INT32 errorCode, OUT HwmErrorMessageInfo* errorMessageInfo) = 0;

    /**
    * @brief [en] This interface is used to set need transparent windows in share
    *        [cn] 设置屏幕共享中需透明的窗口
    *
    * @detail
    * [en] the count of windows must less than 4
    * [cn] 设置窗口句柄个数不超过4个
    *
    * @param wndInfo [type:HwmTransparentWndInfo*]
    * [en] Window handle info
    * [cn] 窗口句柄信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR SetTransparentWnd(HwmTransparentWndInfo* wndInfo) = 0;

    /**
    * @brief [en] This interface is used to customize the text of the specified scene
    *        [cn] 自定义指定场景文案
    *
    * @param customSceneTextInfoList [type:HwmCustomSceneTextInfoList*]
    * [en] Customize scene text
    * [cn] 自定义场景文案信息列表
    *
    * @mac:transToSync
    **/
    virtual SDKERR SetCustomUIText(const HwmCustomSceneTextInfoList* customSceneTextInfoList) = 0;

    /**
    * @brief [en] This interface is used to customize the image of the specified scene
    *        [cn] 自定义指定场景图标
    *
    * @param customSceneImageInfoList [type:HwmCustomSceneImageInfoList*]
    * [en] Customize scene image
    * [cn] 自定义场景图标信息列表
    *
    * @mac:disable
    **/
    virtual SDKERR SetCustomUIImage(const HwmCustomSceneImageInfoList* customSceneImageInfoList) = 0;

    /**
    * @brief
    * [en]set virtual background default images
    * [cn]设置虚拟背景默认图片
    *
    * @detail
    * [en]This interface is used to set virtual background default images
    * [cn]该接口用于设置虚拟背景默认图片
    *
    * @param info [type:HwmVirtualBackgroundInfo*][cs_allowNull:N]
    * [en]Virtual background info.
    * [cn]虚拟背景信息
    */
    virtual SDKERR SetVirtualBackgroundDefaultImages(const HwmVirtualBackgroundInfo* defaultImagesInfo) { return SDKERR::UISDK_NOT_SUPPORT_END; }
};

/**
* @brief 
* [en] This class is about UIConfig result callback.
* [cn] 界面配置操作类结果回调
*/
class HWM_API IHwmUIConfigResultCallback
{
public:
    /**
    * @brief 
    * [en] Callback of Config interface.
    * [cn] 配置定制化参数回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnConfigResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of SetWaterMarkInjectInfo interface.
    * [cn] 设置水印注入信息回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetWaterMarkResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] window handle list callback.when call GetHwndInfoList, will receive this calback
    * [cn] 窗口句柄列表回调，GetRenderWindowInfoList 请求返回
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param hwndInfo [type:HwmHwndListInfo*]
    * [en] Window handle info
    * [cn] 窗口句柄信息
    */
    virtual HWM_VOID OnGetRenderWindowInfoListResult(SDKERR ret, const HWM_CHAR* reason, const HwmHwndListInfo* hwndInfo) {};

    /**
    * @brief 
    * [en] Callback of ShowMeetingWindow interface.
    * [cn] 显示会议窗口
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnShowMeetingWindowResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of SetAvatar interface.
    * [cn] 设置头像回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetAvatarResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief 
    * [en] Callback of SetMeetingWindowIcon interface.
    * [cn] 设置会议窗口图标回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetMeetingWindowIconResult(SDKERR ret, const HWM_CHAR* reason) {};
    
    /**
    * @brief
    * [en] Callback of ShowSettingWindow interface.
    * [cn] 显示设置界面接口结果回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnShowSettingWindowResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en] Callback of ShowToast interface.
    * [cn] 显示提示信息回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnShowToastResult(SDKERR ret, const HWM_CHAR* reason) {};

    /**
    * @brief
    * [en] Callback of GetMainWindowInfo interface
    * [cn] 主窗口信息通知，获取主窗口信息请求返回
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    *
    * @param mainWindowInfo [type:HwmMainWindowInfo*]
    * [en] Main window info
    * [cn] 主窗口信息
    */
    virtual HWM_VOID OnGetMainWindowInfoResult(SDKERR ret, const HWM_CHAR* reason, const HwmMainWindowInfo* mainWindowInfo) {};

    /**
    * @brief
    * [en] Callback of SetAvatarStyle interface.
    * [cn] 设置头像风格回调
    *
    * @param ret [type:SDKERR]
    * [en] Error code
    * [cn] 错误码
    *
    * @param reason [type:HWM_CHAR*]
    * [en] Reason
    * [cn] 错误描述
    **/
    virtual HWM_VOID OnSetAvatarStyleResult(SDKERR ret, const HWM_CHAR* reason) {};
};

/**
* @brief 
* [en] This class is about UIConfig notify callback.
* [cn] 界面配置通知
*/
class HWM_API IHwmUIConfigNotifyCallback
{
public:
    /**
    * @brief 
    * [en] Notify of inject btn.
    * [cn] 注入点击通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，用户点击了配置过点击注入的按钮。
    *
    * @param injectBtnInfo [type:HwmInjectBtnInfo*]
    * [en] Inject btn info
    * [cn] 注入按钮点击通知信息
    **/
    virtual HWM_VOID OnClickInjectBtnNotify(const HwmInjectBtnInfo* injectBtnInfo) {};

    /**
    * @brief 
    * [en] Notify of inject participant item btn.
    * [cn] 与会者列表子项注入点击通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，用户点击了与会者列表子项中配置过点击注入的按钮。
    *
    * @param injectBtnInfo [type:HwmInjectParticipantItemBtnInfo*]
    * [en] Inject btn info
    * [cn] 注入按钮点击通知信息
    **/
    virtual HWM_VOID OnClickInjectParticipantItemBtnNotify(const HwmInjectParticipantItemBtnInfo* injectBtnInfo) {};

    /**
    * @brief 
    * [en] Notify of inject Call all btn.
    * [cn] 全部呼叫注入点击通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，用户点击了与会者列表未入会列表中的全部呼叫点击注入的按钮。
    *
    * @param injectBtnInfo [type:HwmCallAllBtnInfo*]
    * [en] Inject btn info
    * [cn] 注入按钮点击通知信息
    **/
    virtual HWM_VOID OnClickInjectCallAllBtnNotify(const HwmCallAllBtnInfo* injectBtnInfo) {};

    /**
    * @brief 
    * [en] window handle list notify. when window handle changed, will receive this notify
    * [cn] 窗口句柄通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，窗口句柄信息发生变更。
    *
    * @param hwndInfo [type:HwmHwndListInfo*]
    * [en] Window handle info
    * [cn] 窗口句柄信息
    */
    virtual HWM_VOID OnRenderWindowInfoListChanged(const HwmHwndListInfo* hwndInfo) {};

    /**
    * @brief 
    * [en] Notify of query external contact avatar
    * [cn] 查询外部通讯录头像通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，用户打开了界面配置中的外部通讯录头像信息的开关，入会且未打开摄像头。
    *
    * @param contactInfo [type:HwmContactInfo*]
    * [en] Query external avatar contact info
    * [cn] 查询外部头像的联系人信息
    */
    virtual HWM_VOID OnQueryContactAvatarNotify(const HwmContactInfo* contactInfo) {};

    /**
    * @brief
    * [en] Notify of double click participant list item
    * [cn] 双击与会者项通知
    *
    * @detail
    * [en]
    * [cn] 收到该通知时，用户双击了与会者列表中的与会者项。
    *
    * @param participantInfo [type:HwmParticipantItemInfo*]
    * [en] Participant info
    * [cn] 与会者信息
    */
    virtual HWM_VOID OnDoubleClickParticipantListItemNotify(const HwmParticipantItemInfo* participantInfo) {};
};

/**
* @brief 
* [en]This class is about UIConfig management object.
* [cn]界面配置管理对象
*
* @detail
* [en]This class is about UIConfig management object.
* [cn]获取界面配置管理对象的方法HwmUISDK.getUIConfigApi()
*
* @GetInstance
*        HWM_UISDK_NAMESPACE::HwmUISDK::GetUIConfigApi()
*/
class IHwmUIConfig
{
public:
    /**
    * @brief
    * [en] This class is about the global callback for the operating results of UIConfig management.
    * [cn] 设置界面配置操作类全局回调
    *
    * @param callback [type:IHwmUIConfigResultCallback*][cs_allowNull:N]
    * [en]
    * [cn] 通用配置操作类全局回调
    */
    virtual HWM_VOID SetUIConfigResultCallback(IHwmUIConfigResultCallback* callback) = 0;

    /**
    * @brief
    * [en] This class is about the global callback for the notifications of UIConfig management.
    * [cn] 设置界面配置通知类全局回调
    *
    * @param callback [type:IHwmUIConfigNotifyCallback*][cs_allowNull:N]
    * [en]
    * [cn] 通用配置通知类全局回调
    */
    virtual HWM_VOID SetUIConfigNotifyCallback(IHwmUIConfigNotifyCallback* callback) = 0;

    /**
    * @brief 
    * [en] This interface is used to config sdk.
    * [cn] 界面配置
    *
    * @detail
    * [en]
    * [cn] 该接口用于配置定制化参数，自定义会议中界面的部分元素。
    *
    * @param uiConfig [type:HWM_CHAR*]
    * [en] Config info
    * [cn] 配置信息，为Json格式字符串
    *
    * @mac:transToSync
    **/
    virtual SDKERR Config(const HWM_CHAR uiConfig[HWM_MAX_CONFIG_LEN]) = 0;

    /**
    * @brief 
    * [en] This interface is used to set watermark inject information.
    * [cn] 自定义会中水印
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置水印自定义内容注入，若未进行设置，将默认使用内部优先级来展示水印内容。注意事项：1、该接口必须在会前调用，会中调用不会即时生效；2、企业级水印开关未打开的场景下，水印自定义内容注入也不会生效。
    *
    * @param waterMarkInfo [type:HwmWaterMarkInjectInfo*]
    * [en] WaterMark Inject Info
    * [cn] 水印注入信息
    *
	* @mac:transToSync
    **/
    virtual SDKERR SetWaterMark(const HwmWaterMarkInfo* waterMarkInfo) = 0;

    /**
    * @brief 
    * [en] This method is used to get window handle list.
    * [cn] 获取窗口句柄列表
    *
    * @detail
    * [en]
    * [cn] 该接口用于获取窗口句柄列表。
    *
    * @param [cs_allowNull:Y]
    *
	* @mac:transToSync
    **/
    virtual SDKERR GetRenderWindowInfoList() = 0;

    /**
    * @brief 
    * [en] This interface is used to ShowMeetingWindow.
    * [cn] 显示会议窗口
    *
    * @detail
    * [en]
    * [cn] 当会议主窗口处于最小化或者被其他窗口覆盖时，使用该接口将会议主窗口拉到最上层。一般用于从会议列表中拉起正在进行的会议窗口。
    *
    * @param [cs_allowNull:Y]
    *
	* @mac:transToSync
    **/
    virtual SDKERR ShowMeetingWindow() = 0;

    /**
    * @brief 
    * [en] This interface is used to set avatar.
    * [cn] 设置头像
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置头像信息。
    *
    * @param avatarInfo [type:HwmAvatarInfo*]
    * [en] Avatar info
    * [cn] 头像信息
    **/
    virtual SDKERR SetAvatar(const HwmAvatarInfo* avatarInfo) = 0;

    /**
    * @brief 
    * [en] This interface is used to set meeting window icon info.
    * [cn] 自定义会议窗口任务栏图标
    *
    * @detail
    * [en] (only for mac)
    * [cn] 该接口用于设置会议窗口任务栏图标信息(仅对mac平台生效)，win平台可以通过替换HwmSdk\sdkResources\nativeimg目录下的video.ico图标文件实现。
    *
    * @param windowIconInfo [type:HwmWindowIconInfo*]
    * [en] Window icon info
    * [cn] 窗口图标信息
    *
	* @mac:transToSync
    * @win:todo
    **/
    virtual SDKERR SetMeetingWindowIcon(const HwmWindowIconInfo* windowIconInfo) = 0;

    /**
    * @brief
    * [en] This interface is used to show setting window.
    * [cn] 显示设置窗口接口
    *
    * @detail
    * [en]
    * [cn] 该接口用于显示设置窗口接口。
    *
    * @param showInfo [type:HwmShowSettingWindowInfo*]
    * [en] Show setting window info
    * [cn] 显示设置窗口信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR ShowSettingWindow(const HwmShowSettingWindowInfo* showInfo) = 0;

    /**
    * @brief
    * [en] This interface is used to show toast info.
    * [cn] 显示提示信息
    *
    * @detail
    * [en]
    * [cn] 该接口用于显示提示信息
    *
    * @param toastInfo [type:HwmToastInfo*]
    * [en] Toast info
    * [cn] 提示信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR ShowToast(const HwmToastInfo* toastInfo) = 0;

    /**
    * @brief
    * [en] This interface is used to get main window info.
    * [cn] 获取主窗口信息
    *
    * @detail
    * [en]
    * [cn] 该接口用于获取主窗口信息
    *
    * @mac:transToSync
    **/
    virtual SDKERR GetMainWindowInfo() = 0;

    /**
    * @brief 
    * [en] This interface is used to set avatar style.
    * [cn] 设置头像风格
    *
    * @detail
    * [en] This interface is used to set avatar style.
    * [cn] 该接口用于设置头像风格。
    *
    * @param avatarStyle [type:HwmAvatarStyle*]
    * [en] Avatar style
    * [cn] 头像风格
    **/
    virtual SDKERR SetAvatarStyle(const HwmAvatarStyle* avatarStyle) = 0;
};

END_HWM_UISDK_NAMESPACE
#endif /* __HWM_UISDK_CONFIG_SERVICE_H__ */


