import { SDKERR} from './enum';
import { StartCallInfo, CallInfo, CallIncomingInfo, CallStateInfo, CallRecordInfo} from './model';
import { needCallWithIPC } from './util';
const electron = window.require('electron');

export class ICall {
	addOn!: ICall;
	_onCallStateChangedCallback!: (callStateInfo: CallStateInfo) => void;
	_onCallEndedNotifyCallback!: (callRecordInfo: CallRecordInfo) => void;
	_startCallCallback!: (ret: SDKERR, reason: string) => void;
	_inProcessCallbackSet!: Set<string>;
	callWithIPC: boolean = false;
	constructor(addOn: ICall) {
		this.addOn = addOn;
		this._inProcessCallbackSet = new Set<string>();
		this.callWithIPC = needCallWithIPC();
	}

	init(): void {
		this._inProcessCallbackSet.clear();
	}

	/**
	* [en] Notify of call state. [cn] 通话状态通知
	*/
	setOnCallStateChangedCB(onCallStateChanged: (callStateInfo: CallStateInfo) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnCallStateChangedCB', {});
			electron.ipcRenderer.on('onCallStateChanged', (event: any, arg: any) => {
				if (onCallStateChanged) {
					onCallStateChanged(arg["callStateInfo"])
				}
			});
			return;
		}

		this._onCallStateChangedCallback = (callStateInfo: CallStateInfo) => {
			if (onCallStateChanged) {
				onCallStateChanged(callStateInfo);
			}
		}
		this.addOn.setOnCallStateChangedCB(this._onCallStateChangedCallback);
	}

	/**
	* [en] Notify of the call record when the call ended [cn] 通话结束通知
	*/
	setOnCallEndedNotifyCB(onCallEndedNotify: (callRecordInfo: CallRecordInfo) => void): void {
		if (this.callWithIPC) {
			electron.ipcRenderer.sendSync('setOnCallEndedNotifyCB', {});
			electron.ipcRenderer.on('onCallEndedNotify', (event: any, arg: any) => {
				if (onCallEndedNotify) {
					onCallEndedNotify(arg["callRecordInfo"])
				}
			});
			return;
		}

		this._onCallEndedNotifyCallback = (callRecordInfo: CallRecordInfo) => {
			if (onCallEndedNotify) {
				onCallEndedNotify(callRecordInfo);
			}
		}
		this.addOn.setOnCallEndedNotifyCB(this._onCallEndedNotifyCallback);
	}

	/**
	* [en] This interface is used to start call. [cn] 发起呼叫
	*/
	startCall(startCallInfo: StartCallInfo, onStartCallResult?: (ret: SDKERR, reason: string) => void): void {
		if (this._inProcessCallbackSet.has("startCall")) {
			if (onStartCallResult) {
				onStartCallResult(SDKERR.UISDK_IN_PROCESS, "uisdk in process");
			};
			return;
		}
		this._inProcessCallbackSet.add("startCall");
		this._startCallCallback = (ret: SDKERR, reason: string) => {
			this._inProcessCallbackSet.delete("startCall");
			if (onStartCallResult) {
				onStartCallResult(ret, reason);
			}
		}

		if (this.callWithIPC) {
			electron.ipcRenderer.invoke('startCall', {"startCallInfo":startCallInfo}).then((ret: any) => {
				this._inProcessCallbackSet.delete('startCall');
				if (onStartCallResult) {
					onStartCallResult(ret["ret"], ret["reason"]);
				}
			});
		}

		this.addOn.startCall(startCallInfo, this._startCallCallback);
	}

}

