﻿/**
 * @file HwmUISdkConfServiceDef.h
 *
 * Copyright(C), 2019-2020, Huawei Tech. Co., Ltd. ALL RIGHTS RESERVED.
 *
 * @brief [en] define file.
 *        [cn] 结构体、枚举定义文件
 */

#ifndef _HWM_UISDK_CONF_SERVICE_DEF_H_
#define _HWM_UISDK_CONF_SERVICE_DEF_H_

#include "HwmUISdkCommonDef.h"
#include "HwmBaseMacroDef.h"
#include "HwmGlobalErrorCodeDef.h"

BEGIN_HWM_UISDK_NAMESPACE

/**
* [en]allow join conference user type.
* [cn]允许加入会议的用户类型
*/
typedef enum tagHwmConfAllowJoinUserType
{
    CONF_ALLOW_JOIN_ANYONE = 0,                 /**< [en]Anyone
                                                     [cn]所有用户。
                                                     [ios:rename:Anyone]*/
    CONF_ALLOW_JOIN_LOGINED_USER,               /**< [en]logined user
                                                     [cn]已登录用户。
                                                     [ios:rename:LoginedUser]*/
    CONF_ALLOW_JOIN_IN_COMPANY_USER,            /**< [en]Company user
                                                     [cn]企业内用户。
                                                     [ios:rename:InCompanyUser]*/
    CONF_ALLOW_JOIN_INVITED_USER,               /**< [en]Invited user
                                                     [cn]被邀请用户。
                                                     [ios:rename:InvitedUser]*/
}HwmConfAllowJoinUserType;

/**
* [en]VMR conf Id type.
* [cn]云会议室的会议ID类型
*/
typedef enum tagHwmVmrConfIdType
{
    HWM_VMR_CONF_ID_TYPE_FIXED = 0,                    /**< [en]VMR conf Id type is fixed
                                                            [cn]云会议室的会议ID类型是固定类型。 */

    HWM_VMR_CONF_ID_TYPE_RANDOM                        /**< [en]VMR conf Id type is random
                                                            [cn]云会议室的会议ID类型是随机类型。 */
}HwmVmrConfIdType;

/**
* [en]Conf role.
* [cn]会议角色
*/
typedef enum tagHwmConfRole
{
    HWM_CONF_ROLE_ATTENDEE = 0,           /**< [en]Indicates attendee
                                               [cn]普通与会者 */
    HWM_CONF_ROLE_HOST,                   /**< [en]Indicates host
                                               [cn]主持人 */
    HWM_CONF_ROLE_AUDIENCE,               /**< [en]Indicates audience
                                               [cn]观众 */
    HWM_CONF_ROLE_COHOST,                 /**< [en]Indicates cohost
                                               [cn]联席主持人 */
    HWM_CONF_ROLE_WAITING,                /**< [en]Indicates waiting
                                               [cn]等候室成员 */
    HWM_CONF_ROLE_BUTT
} HwmConfRole;

/**
 * [en]conference resource type
 * [cn]会议资源类型
 */
typedef enum tagHwmConfResType
{
    HWM_CONF_RESTYPE_DEFAULT = 0,                 /**< [en]default
                                                       [cn]默认*/
    HWM_CONF_RESTYPE_SHARE_VMR = 3,               /**< [en]share vmr
                                                       [cn]共享vmr*/
}HwmConfResType;

/**
 * [en]This structure is summary state.
 * [cn]会议纪要状态
 */
typedef enum tagHwmSummaryState
{
    SUMMARY_STATE_CLOSE = 0,                 /**< [en]Indicates summary close
                                                  [cn]会议纪要关闭 */
    SUMMARY_STATE_OPEN,                      /**< [en]Indicates summary open
                                                  [cn]会议纪要开启 */
}HwmSummaryState;

/**
* [en]This enum is used to describe multi-share type<br>
* [cn] 多人共享类型
*/
typedef enum tagHwmMultiShareType
{
    MULTI_SHARE_TYPE_SINGAL = 1, /**< [en]Indicates only one user can share.[cn]只允许单人共享*/
    MULTI_SHARE_TYPE_MULTI,      /**< [en]Indicates not only one user can share.[cn]允许多人共享*/
}HwmMultiShareType;

/**
 * [en]This enum is used to describe auto mute type<br>
 * [cn]入会自动静音类型
 */
typedef enum tagHWMAutoMuteType
{
    AUTO_MUTE_TYPE_DEFAULT, /**< [en]Indicates auto mute type of default. [cn]默认 */
    AUTO_MUTE_TYPE_MUTE,    /**< [en]Indicates auto mute type of mute. [cn]静音 */
    AUTO_MUTE_TYPE_UNMUTE,  /**< [en]Indicates auto mute type of unmute. [cn]非静音 */
}HwmAutoMuteType;

/**
* [en]This struct is used to describe auto invite type.
* [cn]自动邀请状态
*/
typedef enum tagHwmAutoInviteType
{
    AUTO_INVITE_NONE,           /**<[en]Indicates not set (default open auto-invite)
                                    [cn]不设置（默认自动邀请）*/
    AUTO_INVITE_OPEN,           /**<[en]Indicates auto-invite open.
                                    [cn]自动邀请开启 */
    AUTO_INVITE_CLOSE,          /**<[en]Indicates auto-invite close.
                                    [cn]自动邀请关闭 */
} HwmAutoInviteType;

/**
* [en]This enum is used to describe attendee type
* [cn]成员类型
*/
typedef enum tagHwmAttendeeType
{
    ATTENDEE_TYPE_NORMAL = 0,                               /**< [en]Indicates normal: Voice, HD, SD participants address
                                                                 [cn]语音、高清、标清与会者地址 */
    ATTENDEE_TYPE_TELEPRESENCE,                             /**< [en]Indicates telepresence participants address type, single screen, three screen telepresence all belong to this
                                                                 [cn]智真与会者地址类型，单屏、三屏智真均属此类 */
    ATTENDEE_TYPE_TERMINAL,                                 /**< [en]Indicates conference room or hard terminal
                                                                 [cn]会议室或硬终端 */
    ATTENDEE_TYPE_OUTSIDE,                                  /**< [en]Indicates outside attendee
                                                                 [cn]外部与会人 */
    ATTENDEE_TYPE_CUSTOMNUMBER,                             /**< [en]Indicates custom number
                                                                 [cn]custom number*/
    ATTENDEE_TYPE_MOBILE,                                   /**< [en]Indicates mobile user
                                                                 [cn]软终端用户手机 */
    ATTENDEE_TYPE_ANONYMOUS,                                /**< [en]Indicates anonymous user
                                                                 [cn]匿名用户 */
    ATTENDEE_TYPE_TELEPHONE,                                /**< [en]Indicates telephone user
                                                                 [cn]电话用户 */
    ATTENDEE_TYPE_BOARD,                                    /**< [en]Indicates board user
                                                                 [cn]白板用户 */
    ATTENDEE_TYPE_IDEAHUB,                                  /**< [en]Indicates ideahub user
                                                                 [cn]海思芯片的大屏设备 */
    ATTENDEE_TYPE_HWVISION,                                 /**< [en]Indicates hwvision user
                                                                 [cn]智慧屏 */
    ATTENDEE_TYPE_WELINKC                                   /**< [en]Indicates welinkc user
                                                                 [cn]welink大众版 */
} HwmAttendeeType;

/**
* [en]This struct is used to describe attendee base info.
* [cn]与会者基础信息
*/
typedef struct tagHwmAttendeeBaseInfo
{
    HWM_CHAR nickName[HWM_MAX_USER_NAME_LEN];               /**< [en]Indicates nickName
                                                                 [cn]与会者名称。
                                                                 [cs_allowNull:N] */
    HWM_CHAR aliasCN[HWM_MAX_DISPLAY_NAME_LEN];             /**< [en]Indicates chinese alias.
                                                                 [cn]中文别名（获取会中与会者信息使用）
                                                                 [cs_allowNull:Y][doc:disable] */
    HWM_CHAR aliasEN[HWM_MAX_DISPLAY_NAME_LEN];             /**< [en]Indicates english alias.
                                                                 [cn]英文别名（获取会中与会者信息使用）
                                                                 [cs_allowNull:Y][doc:disable] */
    HWM_CHAR extendedField[HWM_MAX_EXTENDED_FIELD_LEN];     /**< [en]Indicates extended field.
                                                                 [cn]扩展字段
                                                                 [doc:disable] */
    HWM_CHAR number[HWM_MAX_NUMBER_LEN];                    /**< [en]Indicates number
                                                                 [cn]号码。如果填分配给账号的sip号码（如+99111244216210249）则呼叫该账号的App；
                                                                     如果填PSTN号码（如18700000000），则通过VoIP网关呼叫该号码，前提是该企业已开通PSTN呼叫权限。
                                                                     与thirdUserId二选一，账号密码鉴权时使用。
                                                                 [cs_allowNull:N]*/
    HWM_CHAR thirdUserId[HWM_MAX_USER_ID_LEN];              /**< [en]The attendee's third account
                                                                 [cn]第三方用户ID。与number二选一，App ID鉴权时使用。
                                                                 [cs_allowNull:N] */
    HWM_CHAR accountId[MAX_ACCOUNT_ID_LEN];                 /**< [en]Indicates account id
                                                                 [cn]华为云会议账号
                                                                 [cs_allowNull:Y] */
    HWM_CHAR email[HWM_MAX_EMAIL_LEN];                      /**< [en]Indicates email
                                                                 [cn]电子邮箱地址，用于发送邮件通知。
                                                                 [cs_allowNull:Y] */
    HWM_CHAR sms[HWM_MAX_PHONE_NUM_LEN];                    /**< [en]Indicates sms
                                                                 [cn]短信通知手机号码，用于发送短信通知。
                                                                 [cs_allowNull:Y] */
    HWM_BOOL isAutoInvite;                                  /**< [en]Indicates whether is auto invite
                                                                 [cn]是否自动外邀。
                                                                 [cs_allowNull:Y] */
    HWM_BOOL isMute;                                        /**< [en]Indicates whether is mute
                                                                 [cn]是否闭音。
                                                                 [cs_allowNull:Y] */
    HwmConfRole role;                                       /**< [en]Indicates role
                                                                 [cn]会议角色。
                                                                 [cs_allowNull:Y] */
    HwmAttendeeType type;                                   /**< [en]This enum is used to describe attendee type
                                                                 [cn]成员类型
                                                                 [cs_allowNull:Y] */
    HWM_CHAR userUuid[HWM_MAX_ACCOUNT_ID_LEN];              /**< [en]Indicates the uuid.
                                                                 [cn]与会者uuid
                                                                 [cs_allowNull:Y] */
    HWM_CHAR corpId[HWM_MAX_ORG_ID_LENGTH];                 /**< [en]Indicates corp id.
                                                                 [cn]企业id */
    HwmAutoInviteType autoInviteType;                       /**< [en]Indicates the auto-invite type.
                                                                 [cn]自动邀请类型（仅创建会议生效）
                                                                 [cs_allowNull:Y] */
} HwmAttendeeBaseInfo;

/**
* [en]Creat conf info.
* [cn]会议信息。
*/
typedef struct tagHwmCreateConfInfo
{
    HWM_CHAR subject[HWM_MAX_SUBJECT_LEN];                  /**< [en]The conf suject
                                                                 [cn]会议主题。
                                                                 [cs_allowNull:N] */
    HwmMediaType mediaType;                                 /**< [en]The media type
                                                                 [cn]会议类型。
                                                                 [cs_allowNull:N] */
    HWM_BOOL needPassword;                                  /**< [en]Is need password
                                                                 [cn]是否需要来宾密码（仅对随机会议ID生效）。
                                                                 [cs_allowNull:N] */
    HWM_BOOL isAutoRecord;                                  /**< [en]Is autorecord
                                                                 [cn]是否开启自动录制会议。开启自动录制会议开关，内部会默认支持录制会议，录制会议开关的打开与关闭并不影响。
                                                                     说明：仅对云录制生效，客户端本地录制不生效。
                                                                 [cs_allowNull:Y] */
    HWM_BOOL allowRecord;                                   /**< [en]Is allow record(only for win)
                                                                 [cn]是否支持录制会议。(仅对win平台生效)
                                                                     说明：仅对云录制生效，客户端本地录制不生效。
                                                                 [cs_allowNull:Y] */
    HwmConfAllowJoinUserType confAllowJoinUser;             /**< [en]Indicates the user type allowed to join the conference.
                                                                 [cn]允许入会范围限制。默认所有用户。
                                                                 [cs_allowNull:Y] */
    HWM_CHAR vmrId[HWM_MAX_VMR_ID_LEN];                     /**< [en]The vmr Id
                                                                 [cn]云会议室唯一ID。预约随机会议时，设置为空字符串。
                                                                 [cs_allowNull:Y] */
    HwmVmrConfIdType vmrConfIdType;                         /**< [en]VMR conf Id type.
                                                                 [cn]云会议室的会议ID类型
                                                                 [cs_allowNull:Y] */
    HWM_CHAR guestPwd[HWM_MAX_CONF_PASSWORD_LEN];           /**< [en]Indicates the password for common participants.If the password is null, the server will generate a password randomly.
                                                                 [cn]来宾密码，如果空则服务器随机生成（仅对随机会议ID生效）。
                                                                 [cs_allowNull:Y] */
    HWM_BOOL isOpenWaitingRoom;                             /**< [en]Indicates whether is open waiting room
                                                                 [cn]等候室开启开关。
                                                                     说明：需要等候室功能开通后才能生效
                                                                 [cs_allowNull:Y] */
    HWM_INT32 duration;                                     /**< [en]Indicates the duration of the conference. (minute)
                                                                 [cn]会议时长，单位分钟，最小值15分钟，最大值1440分钟。
                                                                     注：不填时由服务端取默认会议时长
                                                                 [cs_allowNull:Y][mac:disable] */
    HWM_BOOL isMicOn;                                       /**< [en]Is mic on
                                                                 [cn]是否开启麦克风，true开启麦克风；false关闭麦克风。
                                                                 [cs_allowNull:Y] */
    HWM_BOOL isCameraOn;                                    /**< [en]Is camera on
                                                                 [cn]是否开启摄像头，true开启摄像头；false关闭摄像头。
                                                                 [cs_allowNull:Y] */
    HWM_UINT32 concurrentParticipants;                      /**< [en]concurrent participants.
                                                                  [cn]会议方数，会议最大与会人数限制. 0: 无限制, 大于0: 会议最大与会人数
                                                                 [cs_allowNull:Y] */
    HWM_UINT32 numOfAttendee;                               /**< [en]length for attendees. 
                                                                 [cn]与会者个数。若不需要带与会者，参数值可以为0。
                                                                 [cs_allowNull:N][length for attendees] */
    HwmAttendeeBaseInfo* attendees;                         /**< [en]Attendees info 
                                                                 [cn]与会者信息。若不需要带与会者，参数值可以为空。
                                                                 [cs_allowNull:N] */
    HWM_CHAR customInfo[HWM_MAX_CUSTOM_INFO_LEN];           /**< [en]User-defined information
                                                                 [cn]用户自定义信息。
                                                                 [cs_allowNull:Y] */
    HwmConfResType confResType;                             /**< [en]Indicates conf resource type
                                                                 [cn]会议资源类型
                                                                 [cs_allowNull:Y] */
    HwmSummaryState defaultSummaryState;                    /**< [en]Indicates record include summary by default
                                                                 [cn]会议云录制是否包含纪要，初始值
                                                                 [cs_allowNull:Y] */
    HwmAutoMuteType autoMuteMode;                           /**< [en]Indicates soft terminal auto mute
                                                                 [cn]来宾入会，软终端是否自动静音 
                                                                 [cs_allowNull:Y] */
    HwmAutoMuteType hardTerminalAutoMuteMode;               /**< [en]Indicates hard terminal auto mute
                                                                 [cn]来宾入会，硬终端是否自动静音 
                                                                 [cs_allowNull:Y] */
    HWM_BOOL supportMultiShare;                             /**< [en]Indicates support multi-share.
                                                                 [cn]是否支持多人共享
                                                                 [cs_allowNull:Y] [doc:disable] */
    HWM_BOOL supportWatermark;                              /**< [en]Indicates is whether support watermark
                                                                 [cn]是否支持水印
                                                                 [cs_allowNull:Y] */
    HWM_BOOL forbiddenScreenShots;                          /**< [en]Indicates is whether disable screenshots
                                                                 [cn]是否禁止截屏
                                                                 [cs_allowNull:Y] */
    HWM_BOOL autoPublishSummary;                            /**< [en]Indicates Whether auto publish summary. 
                                                                 [cn]是否自动发布纪要（免人工审核）
                                                                 [cs_allowNull:Y] */
    HWM_BOOL supportDataGuardAudio;                         /**< [en]Indicates whether is support data guard audio.
                                                                 [cn]是否支持音频暗水印
                                                                 [cs_allowNull:Y][doc:disable] */
    HWM_BOOL supportDataGuardVideo;                         /**< [en]Indicates whether is support data guard video.
                                                                 [cn]是否支持视频暗水印
                                                                 [cs_allowNull:Y][doc:disable] */
}HwmCreateConfInfo;

/**
* [en]Join conf info.
* [cn]会议id入会信息
*/
typedef struct tagHwmJoinConfByIdInfo
{
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];                    /**< [en]Conf id
                                                                  [cn]会议ID。
                                                                  [cs_allowNull:N] */
    HWM_CHAR password[HWM_MAX_CONF_PASSWORD_LEN];            /**< [en]Conf password
                                                                  [cn]会议密码。若希望以主持人身份入会，必须传入主持人密码。
                                                                      若希望以来宾身份入会，会议要求来宾密码时，必须传入来宾密码，
                                                                      会议不要求来宾密码时，可以为空字符串。
                                                                  [cs_allowNull:Y] */
    HWM_CHAR nickName[HWM_MAX_DISPLAY_NAME_LEN];             /**< [en]Indicates show nickName in conf.
                                                                  [cn]会议昵称。在已登录状态下可为空，已登录时为空将使用登录的用户名作为会场名。
                                                                      匿名入会必须传该名字
                                                                  [cs_allowNull:Y][cs_length:0-256][cs_sensitive:name] */
    HWM_CHAR aliasCN[HWM_MAX_DISPLAY_NAME_LEN];              /**< [en]Indicates chinese alias.
                                                                  [cn]中文别名
                                                                  [cs_allowNull:Y][doc:disable] */
    HWM_CHAR aliasEN[HWM_MAX_DISPLAY_NAME_LEN];              /**< [en]Indicates english alias.
                                                                  [cn]英文别名
                                                                  [cs_allowNull:Y][doc:disable] */
    HWM_CHAR extendedField[HWM_MAX_EXTENDED_FIELD_LEN];      /**< [en]Indicates extended field.
                                                                  [cn]扩展字段
                                                                  [doc:disable] */
    HWM_CHAR corpTrustAppId[HWM_MAX_CORP_TRUST_APP_ID_LEN];  /**< [en]Indicates corp trust app id.
                                                                  [cn]企业互信应用ID */
    HWM_BOOL isSpeakerOff;                                   /**< [en]is speaker off
                                                                  [cn]是否关闭扬声器，true关闭扬声器；false不关闭扬声器。
                                                                  [cs_allowNull:Y] */
    HWM_BOOL isMicOn;                                        /**< [en]Is mic on
                                                                  [cn]是否开启麦克风，true开启麦克风；false关闭麦克风。
                                                                  [cs_allowNull:Y] */
    HWM_BOOL isCameraOn;                                     /**< [en]Is camera on
                                                                  [cn]是否开启摄像头，true开启摄像头；false关闭摄像头。
                                                                  [cs_allowNull:Y] */
    HWM_BOOL isStopConflictConf;                             /**< [en]Indicates whether stop confilct conf.
                                                                  [cn]强制结束和当前VMR会议资源冲突的会议（必须是vmr资源的拥有者才有权限，否则无效）
                                                                  [cs_allowNull:Y]*/
    HWM_BOOL isCascadingMeeting;                             /**< [en]Indicates On-premises Meeting. (Anonymous is not supported; Only for Windows).
                                                                  [cn]是否硬件会议（硬件会议不支持匿名入会，仅对Windows平台生效） 
                                                                  [doc:disable]*/
    HWM_CHAR confPrefix[HWM_MAX_CORP_TRUST_APP_ID_LEN];      /**< [en]Indicates On-premises prefix id (Only for Windows).
                                                                  [cn]硬件会议企业前缀（仅对Windows平台生效）
                                                                  [doc:disable] */
	HWM_BOOL isHidePasswordAlert;                            /**< [en]Is hide password alert. (Only for Mac).
                                                                  [cn]是否隐藏密码弹窗，true隐藏；false不隐藏，展示sdk内部的默认弹窗（仅对Mac平台生效） 
                                                                  [cs_allowNull:Y]
                                                                  [win:disable]*/
}HwmJoinConfByIdInfo;

/**
* [en]Join conf by random info.
* [cn]会议random入会信息
*/
typedef struct tagHwmJoinConfByRandomInfo
{
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];                    /**< [en]Conf id
                                                                  [cn]会议ID。
                                                                  [cs_allowNull:N] */
    HWM_CHAR random[HWM_MAX_RANDOM_LEN];                     /**< [en]random
                                                                  [cn]48位会议密码映射值。来源于分享会议信息中的会议链接尾部。
                                                                      会议不要求来宾密码时，可以为空字符串。
                                                                  [cs_allowNull:Y] */
    HWM_CHAR nickName[HWM_MAX_DISPLAY_NAME_LEN];             /**< [en]Indicates show name in conf.
                                                                  [cn]会议昵称。在已登录状态下可为空，已登录时为空将使用登录的用户名作为会场名。
                                                                      匿名入会必须传该名字
                                                                  [cs_allowNull:Y][cs_length:0-256][cs_sensitive:name] */
    HWM_CHAR domain[HWM_MAX_DOMAIN_LEN];                     /**< [en]domain.
                                                                  [cn]域名
                                                                  [cs_sensitive:domain][cs_allowNull:Y] */
    HWM_BOOL isSpeakerOff;                                   /**< [en]is speaker off
                                                                  [cn]是否关闭扬声器，true关闭扬声器；false不关闭扬声器。
                                                                  [cs_allowNull:Y] */
    HWM_BOOL isMicOn;                                        /**< [en]Is mic on
                                                                  [cn]是否开启麦克风，true开启麦克风；false关闭麦克风。
                                                                  [cs_allowNull:Y] */
    HWM_BOOL isCameraOn;                                     /**< [en]Is camera on
                                                                  [cn]是否开启摄像头，true开启摄像头；false关闭摄像头。
                                                                  [cs_allowNull:Y] */
}HwmJoinConfByRandomInfo;

/**
* [en]Add attendee info
* [cn]添加与会者信息
*/
typedef struct tagHwmAddAttendeeInfo
{
    HWM_UINT32 attendeeSize;                  /**< [en]number of attendees. 
                                                   [cn]添加与会者的个数
                                                   [cs_allowNull:N][length for attendees] */
    HwmAttendeeBaseInfo *attendees;           /**< [en]attendees info. 
                                                   [cn]添加与会者的列表
                                                   [cs_allowNull:N] */
}HwmAddAttendeeInfo;

/**
* [en]Share type.
* [cn]共享类型
*/
typedef enum tagHwmShareType
{
    HWM_SHARE_TYPE_MONITOR,          /**< [en]Share monitor(Only for Windows)
                                          [cn]屏幕共享（仅对Windows平台生效） */
    HWM_SHARE_TYPE_APP,              /**< [en]Share app(Only for Windows)
                                          [cn]程序共享（仅对Windows平台生效） */
    HWM_SHARE_TYPE_WHITE_BOARD,      /**< [en]Share white board(Only for Windows)
                                          [cn]白板共享（仅对Windows平台生效） */
    HWM_SHARE_TYPE_ALL,              /**< [en]All Share 
                                          [cn]全类型共享 */
    HWM_SHARE_TYPE_BUTT
} HwmShareType;

/**
* [en]Monitor.
* [cn]显示器类型。
*/
typedef enum tagHwmMonitor
{
    HWM_MONITOR_MAIN = 1,               /**< [en]Main monitor
                                             [cn]主显示器 */
    HWM_MONITOR_SECONDARY,              /**< [en]Secondary monitor
                                             [cn]副显示器 */

    HWM_MONITOR_BUTT
} HwmMonitor;

/**
* [en]Monitor mode
* [cn]屏幕模式
 */
typedef enum tagHwmMonitorMode
{
    HWM_MONITOR_MODE_PRIMARY_SECONDARY = 0,      /**< [en]Primary secondary mode
                                                      [cn]主副屏模式 */
    HWM_MONITOR_MODE_MULTI_MONITOR,              /**< [en]Multi monitor mode
                                                      [cn]多屏模式 */

    HWM_MONITOR_MODE_BUTT
} HwmMonitorMode;

/**
* [en]Monitor Share info.
* [cn]屏幕共享信息
*/
typedef struct tagHwmMonitorShareInfo
{
    HwmMonitor monitor;           /**< [en]Type of monitor
                                       [cn]屏幕类型，1-主屏幕，2-副屏幕，mode为主副屏模式下生效
                                       [cs_allowNull:N] */
    HwmMonitorMode monitorMode;   /**< [en]Mode of monitor
                                       [cn]屏幕模式，0-主副屏模式，1-多屏模式
                                       [cs_allowNull:N] */
    HWM_UINT32 monitorIndex;      /**< [en]Index of monitor
                                       [cn]屏幕索引，mode为多屏模式下生效
                                       [cs_allowNull:N] */
}HwmMonitorShareInfo;

/**
* [en]App share info.
* [cn]程序共享信息
*/
typedef struct tagHwmAppShareInfo
{
    HWM_HWND appHandle;               /**< [en]App window handle
                                           [cn]程序句柄。
                                           [cs_allowNull:N] */
}HwmAppShareInfo;

/**
* [en]Start share info.
* [cn]开启共享信息
*/
typedef struct tagHwmStartShareInfo
{
    HwmShareType shareType;                     /**< [en]Share type
                                                     [cn]共享类型
                                                     [cs_allowNull:N] */
    HwmMonitorShareInfo monitorShareInfo;       /**< [en]Monitor share info
                                                     [cn]屏幕共享信息。当共享类型不是屏幕共享时，该变量不需要赋值。
                                                     [cs_allowNull:Y] */
    HwmAppShareInfo appShareInfo;               /**< [en]App Share info
                                                     [cn]程序共享信息。当共享类型不是程序共享时，该变量不需要赋值。
                                                     [cs_allowNull:Y] */
    HwmMonitorShareInfo operatableUiMonitor;    /**< [en]UI Control Monitor share info
                                                     [cn]共享时，工具栏所在的显示器信息。若monitor值为0，则不指定显示器。
                                                     [cs_allowNull:N] */
}HwmStartShareInfo;

/**
* [en]This enum is used to describe attendee state.
* [cn]成员状态
*/
typedef enum tagHwmConfAttendeeState
{
    CONF_ATTENDEE_STATE_IN_CONF = 0,     /**< [en]Indicates in conference
                                              [cn]会议中
                                              [ios:rename:InConf] */
    CONF_ATTENDEE_STATE_CALLING,         /**< [en]Indicates is calling
                                              [cn]正在呼叫
                                              [ios:rename:Calling] */
    CONF_ATTENDEE_STATE_JOINING,         /**< [en]Indicates is joining conference
                                              [cn]正在加入会议
                                              [ios:rename:Joining] */
    CONF_ATTENDEE_STATE_LEAVED,          /**< [en]Indicates have leaved
                                              [cn]已经离开
                                              [ios:rename:Leaved] */
    CONF_ATTENDEE_STATE_NO_EXIST,        /**< [en]Indicates not exist
                                              [cn]用户不存在
                                              [ios:rename:NoExist] */
    CONF_ATTENDEE_STATE_BUSY,            /**< [en]Indicates callee is busy
                                              [cn]被叫用户忙
                                              [ios:rename:Busy] */
    CONF_ATTENDEE_STATE_NO_ANSWER,       /**< [en]Indicates no answer
                                              [cn]用户无应答
                                              [ios:rename:NoAnswer] */
    CONF_ATTENDEE_STATE_REJECT,          /**< [en]Indicates user reject answer
                                              [cn]用户拒绝接听
                                              [ios:rename:Reject] */
    CONF_ATTENDEE_STATE_CALL_FAILED,     /**< [en]Indicates call failed
                                              [cn]呼叫失败
                                              [ios:rename:Failed] */
}HwmConfAttendeeState;

/**
* [en]This struct is used to describe attendee info.
* [cn]会中与会者信息
*/
typedef struct tagHwmAttendeeInfo
{
    HwmAttendeeBaseInfo baseInfo;              /**< [en]attendee base info.
                                                    [cn]与会者基本信息 */
    HWM_BOOL isSelf;                           /**< [en]is self.
                                                    [cn]是否是自己 */
    HwmConfAttendeeState state;                /**< [en]Indicates participant state.
                                                    [cn]与会者状态 */
    HWM_BOOL isAnonymous;                      /**< [en]Indicates whether is anonymous.
                                                    [cn]是否是匿名入会 */
    HWM_CHAR corpId[HWM_MAX_ORG_ID_LENGTH];    /**< [en]Indicates corp id.
                                                    [cn]企业id，用于识别是否是本企业账号 */
    HWM_INT32 userId;                          /**< [en]Indicates userId
                                                    [cn]与会者唯一标识
                                                    [doc:disable] */
    HWM_BOOL hasModifyCustomInfoInConfCapability; /**< [en]Indicates has modify custom info in conf capbility.
                                                       [cn]是否支持会中修改参数能力
                                                       [doc:disable] */
}HwmAttendeeInfo;

/**
* [en]This struct is used to describe create conference result.
* [cn]创建会议结果
*/
typedef struct tagHwmCreateConfResult
{
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];                         /**< [en]Indicates conference id.
                                                                       [cn]会议id */
    HWM_CHAR vmrConfId[HWM_MAX_CONF_ID_LEN];                      /**< [en]Indicates VMR conference id(only for mac)
                                                                       [cn]VMR会议id(仅对mac平台生效)
                                                                       [win:todo] */
    HWM_CHAR hostPwd[HWM_MAX_CONF_PASSWORD_LEN];                  /**< [en]Indicates host pwd.
                                                                       [cn]主持人密码
                                                                       [cs_sensitive:hide] */
    HWM_CHAR guestPwd[HWM_MAX_CONF_PASSWORD_LEN];                 /**< [en]Indicates guest pwd.
                                                                       [cn]来宾密码
                                                                       [cs_sensitive:hide] */
    HwmMediaType mediaType;                                       /**< [en]Media type(only for mac)
                                                                       [cn]会议媒体类型(仅对mac平台生效)
                                                                       [win:todo] */
    HWM_CHAR guestJoinUri[HWM_MAX_URL_LEN];                       /**< [en]Indicates guest join url.
                                                                       [cn]来宾链接 */
    HwmConfRole selfConfRole;							          /**< [en]Indicates self role.
                                                                       [cn]本人在会议中的角色 */
    HWM_CHAR subject[HWM_MAX_SUBJECT_LEN];                        /**< [en]Indicates conf subject.
                                                                       [cn]会议主题
                                                                       [cs_sensitive:subject]*/
    HWM_INT64 startTime;                                          /**< [en]Indicates conference start time.
                                                                       [cn]（已废弃，推荐使用字段startTimeStamp）会议开始时间，精度秒(0时区) */
    HWM_INT64 endTime;                                            /**< [en]Indicates conference end time.
                                                                       [cn]（已废弃，推荐使用字段endTimeStamp）会议结束时间 */
    HWM_INT64 startTimeStamp;                                     /**< [en]Indicates conference start time
                                                                       [cn]会议开始时间(时间戳，单位秒) */
    HWM_INT64 endTimeStamp;                                       /**< [en]Indicates conference end time
                                                                       [cn]会议结束时间(时间戳，单位秒) */
    HWM_CHAR schedulerName[HWM_MAX_DISPLAY_NAME_LEN];             /**< [en]Indicates conf scheduler name.
                                                                       [cn]预约者名字
                                                                       [cs_sensitive:name] */
    HWM_CHAR customInfo[HWM_MAX_CUSTOM_INFO_LEN];                 /**< [en]Indicates User-defined extended information
                                                                       [cn]自定义扩展信息 */
}HwmCreateConfResult;

/**
* [en]attendee list info.
* [cn]与会者列表信息
*/
typedef struct tagHwmAttendeeListInfo
{
    HWM_UINT32 numOfAttendee;                               /**< [en]length for attendees.
                                                                 [cn]入会者列表的长度
                                                                 [length for attendees] */
    HwmAttendeeInfo* attendees;                             /**< [en]Attendees info
                                                                 [cn]与会者基础信息 */
}HwmAttendeeListInfo;

/**
* [en]user state
* [cn]用户状态枚举 用于判断当前用户是否处于忙碌状态
*/
typedef enum tagHwmUserState
{
    HWM_STATE_IDLE,                                      /**< [en]idle state
                                                              [cn]空闲状态 */
    HWM_STATE_IN_CONF,                                   /**< [en]in conf state
                                                              [cn]会中状态 */
    HWM_STATE_IN_CALL,                                   /**< [en]in call state
                                                              [cn]呼叫中状态 */
    HWM_USER_STATE_BUTT
} HwmUserState;

/**
* [en]Conf state info.
* [cn]会议状态信息
*/
typedef struct tagHwmConfStateInfo
{
    HwmState state;                 /**< [en]The state
                                         [cn]会议状态。 */
}HwmConfStateInfo;

/**
* [en]Conf end reason.
* [cn]会议结束原因
*/
typedef enum tagHwmConfEndReason
{
    HWM_CONF_END_REASON_LEAVE_CONF_HANGUP = 0,                         /**< [en]Leave conf
                                                                            [cn]离开会议。 */
    HWM_CONF_END_REASON_STOP_CONF_HANGUP,                              /**< [en]Conf is end
                                                                            [cn]会议已结束。*/
    HWM_CONF_END_REASON_HOST_HANGUP,                                   /**< [en]Hang up by host
                                                                            [cn]被主持人移出会场。 */
    HWM_CONF_END_REASON_SESSION_TIMER_TIMEOUT,                         /**< [en]Session timer timeout
                                                                            [cn]网络中断。 */
    HWM_CONF_END_REASON_NOSTREAM_HANGUP,                               /**< [en]No stream
                                                                            [cn]网络异常。 */
    HWM_CONF_END_REASON_CORP_CONFERENCE_RESOURCE_HAS_BEEN_RUN_OUT,     /**< [en]Conf resource has been run out
                                                                            [cn]企业资源已达上限。 */
    HWM_CONF_END_REASON_CONF_ONLY_SAME_ENTERPRISE,                     /**< [en]Only the same enterprise can join the conference.
                                                                            [cn]只允许企业内用户加入会议。 */
    HWM_CONF_END_REASON_CONF_ONLY_INVITED,                             /**< [en]Only invited participants can join the conference.
                                                                            [cn]只允许被邀请者加入会议。 */
    HWM_CONF_END_REASON_CONF_IS_LOCKED,                                /**< [en]Conference is locked.
                                                                            [cn]会议被锁定。 */
    HWM_CONF_END_ERR_VMR_NOT_BEEN_HOLD,                                /**< [en]vmr has not been hold.
                                                                            [cn]VMR会议来宾不允许激活会议。 */
    HWM_CONF_END_REASON_VMR_CONF_CONFLICT,                             /**< [en]Vmr conf conflict
                                                                            [cn]VMR会议冲突。 */
    HWM_CONF_END_REASON_SERVER_ERROR,                                  /**< [en]server error
                                                                            [cn]服务器异常。 */
    HWM_CONF_END_REASON_NUM_OF_PAR_EXCEEDS_UPPER_LIMIT,                /**< [en]num of par exceeds upper limit.
                                                                            [cn]与会方超出专用VMR套餐的最大与会人数 */
    HWM_CONF_END_REASON_CONF_NUM_OVER_MAX_ALLOWED = 111071116,         /**< [en]conf number over max allowed.
                                                                            [cn]超出最大在线会议数限制 */
    HWM_CONF_END_REASON_PARTICIPANT_REACH_MAX_NUM = 111072504,         /**< [en]Max participants in a single meeting reached.
                                                                            [cn]达到系统单个会议人数上限 */
    HWM_CONF_END_REASON_PARTICIPANT_CONF_NOT_EXIST = 111081002,        /**< [en]Site meeting does not exist or is ended.
                                                                            [cn]会场所在会议不存在或已结束 */
    HWM_CONF_END_REASON_CONF_CONFLICT = 111111008,                     /**< [en]hangup for joinconf conflict.
                                                                            [cn]多端入会冲突，导致离开会议 */
    HWM_CONF_END_REASON_END_CONF_NET_BREAK = 411000100,                /**< [en]End the meeting with no stream due to disconnection of network.
                                                                            [cn]无码流结束会议网络断开 */
    HWM_CONF_END_REASON_END_EXPIRATION = 414100069,                    /**< [en]Invalid user, enterprise or SP.
                                                                            [cn]用户，企业或SP失效 */
    HWM_CONF_END_REASON_SDK_CHAIR_END_CONF = 414100074,                /**< [en]Chair end conf.
                                                                            [cn]主持人结束会议 */
    HWM_CONF_END_REASON_SDK_TIMEOVER_END_CONF = 414100075,             /**< [en]Timeover end conf.
                                                                            [cn]会议到时间结束会议 */
    HWM_CONF_END_REASON_RTC_SDK_NETWORK_FAILED = 990000053,            /**< [en]network error.
                                                                            [cn]网络问题 */
    HWM_CONF_END_FORCE_OPEN_CAMERA_HANGUP = 414100078,                 /**< [en]force open camera conf hangup.
                                                                            [cn]强制视频会议挂断 */
    HWM_CONF_END_MANAGER_SHARE_VMR_NOT_ENOUGH = 111071151,             /**< [en]Share VMR not enough.
                                                                            [cn]无共享云会议室资源 */
    HWM_CONF_END_MANAGER_NO_UPSCALE_VMR = 112030001,                   /**< [en]No Upscale vmr.
                                                                            [cn]无高档可用的VMR资源 */
    HWM_CONF_END_REASON_OVER_MAX_PARTICIPANT_NUMBER = 111072101,       /**< [en]Max participant amount reached.
                                                                            [cn]超过设置的会议方数 */
    HWM_CONF_END_REASON_PROJECTION_FAILED = 414100080,                 /**< [en]conf end for projection failed.
                                                                            [cn] 投屏失败导致会议结束 */
    HWM_CONF_END_REASON_PROJECTION_CANCEL = 414100082,                 /**< [en]conf end for cancel.
                                                                            [cn] 投屏协商中大屏取消 */
    HWM_CONF_END_REASON_PROJECTION_BY_SENDER_WITH_TWO_MEMBERS = 414100083,  /**< [en]Indicates end projection by sender with two members.
                                                                                 [cn] 两方投屏会议投屏端结束投屏 */
    HWM_CONF_END_REASON_PROJECTION_BY_RECEIVER = 414100084,                 /**< [en]Indicates end projection by receiver.
                                                                                 [cn] 大屏端结束投屏 */
    HWM_CONF_END_REASON_PROJECTION_BY_SENDER_WITH_MULTI_MEMBERS = 414100085,/**< [en]projection by sender with multi members.
                                                                                 [cn] 多方投屏会议投屏端结束投屏 */
    HWM_CONF_END_REASON_BUTT
} HwmConfEndReason;

/**
* [en]This struct is used to describe conference end info
* [cn]会议结束信息
*/
typedef struct tagHwmConfEndInfo
{
    HwmConfEndReason reason;        /**< [en]ConfEndReason
                                         [cn]离会原因。 */
    SDKERR originalReason;          /**< [en]original ConfEndReason
                                         [cn]原始离会原因。
                                         [doc:disable] */
    HWM_BOOL isP2PConf;             /**< [en]is p2p conf
                                         [cn]是否立即会议。 */
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];   /**< [en]Conf id
                                                 [cn]会议id */
}HwmConfEndInfo;

/**
* [en]Conf info.
* [cn]会议信息
*/
typedef struct tagHwmConfInfo
{
    HWM_CHAR hostJoinUri[HWM_MAX_URL_LEN];					/**< [en]Indicates the uri for host to join conference(only for mac)
                                                                 [cn]主持人入会链接(仅对mac平台生效)
                                                                 [cs_sensitive:ip][win:todo] */
    HWM_CHAR guestJoinUri[HWM_MAX_URL_LEN];                 /**< [en]the uri used by guest to join conference.
                                                                 [cn]来宾加入会议uri链接，可以分享通过浏览器打开进行入会。 */
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];                   /**< [en]Conf id
                                                                 [cn]会议id */
    HwmConfRole selfConfRole;							    /**< [en]Indicates self role.
                                                                 [cn]本人在会议中的角色。主持人、来宾等。 */
    HWM_CHAR hostPwd[HWM_MAX_CONF_PASSWORD_LEN];            /**< [en]Indicates host password.
                                                                 [cn]会议主持人密码。
                                                                 [cs_sensitive:hide] */
    HWM_CHAR guestPwd[HWM_MAX_CONF_PASSWORD_LEN];           /**< [en]Indicates guest password.
                                                                 [cn]会议来宾密码。
                                                                 [cs_sensitive:hide] */
    HWM_CHAR subject[HWM_MAX_SUBJECT_LEN];                  /**< [en]conf subject
                                                                 [cn]会议主题。 */
    HWM_INT64 startTime;                                    /**< [en]start conf time
                                                                 [cn]（已废弃，推荐使用字段startTimeStamp）会议开始时间。 */
    HWM_INT64 endTime;                                      /**< [en]edn conf time
                                                                 [cn]（已废弃，推荐使用字段endTimeStamp）会议结束时间。 */
    HWM_INT64 startTimeStamp;                               /**< [en]Indicates conference start time
                                                                 [cn]会议开始时间(时间戳，单位秒) */
    HWM_INT64 endTimeStamp;                                 /**< [en]Indicates conference end time
                                                                 [cn]会议结束时间(时间戳，单位秒) */
    HWM_CHAR schedulerName[HWM_MAX_USER_NAME_LEN];          /**< [en]Indicates scheduler name
                                                                 [cn]会议预定者名称。 */
    HwmMediaType mediaType;                                 /**< [en]Media type(only for mac)
                                                                 [cn]会议类型。(仅对mac平台生效)
                                                                 [win:todo] */
    HWM_CHAR vmrConfId[HWM_MAX_VMR_CONF_ID_LEN];            /**< [en]The vmr conf Id(only for mac)
                                                                 [cn]用于云会议显示的ID。(仅对mac平台生效)
                                                                 [win:todo] */
    HWM_CHAR corpId[HWM_MAX_ORG_ID_LENGTH];                 /**< [en]Indicates corp id(only for mac)
                                                                 [cn]企业id。(仅对mac平台生效)
                                                                 [win:todo] */
    HWM_CHAR customInfo[HWM_MAX_CUSTOM_INFO_LEN];           /**< [en]Indicates User-defined extended information
                                                                 [cn]自定义扩展信息 */
    HWM_BOOL isOtherCorpConf;                               /**< [en]is other corp conf(only for win)
                                                                 [cn]是否是外部企业的会议：如果是加入外部企业的会议或者匿名入会是true，否则是false */
    HWM_BOOL isCreator;                                     /**< [en]Indicates whether the user is the creator of the conference(only for win)
                                                                 [cn]是否是会议预订者(仅对win平台生效)
                                                                 [mac:todo] */
    HWM_BOOL isInBreakoutSubConf;                           /**< [en]Indicates whether the user is in breakout sub conf
                                                                 [cn]是否在讨论组中 */
}HwmConfInfo;

/**
* [en]Share state info.
* [cn]发送端共享状态信息
*/
typedef struct tagHwmShareStateInfo
{
    HwmState state;                             /**< [en]The share state
                                                     [cn]共享状态。 */
    HWM_CHAR nickName[HWM_MAX_DISPLAY_NAME_LEN];/**< [en]The nickName
                                                     [cn]共享者名称。当state为HWM_STATE_ON时才有数据。 */
    HWM_CHAR number[HWM_MAX_NUMBER_LEN];        /**< [en]The number
                                                     [cn]共享者号码。当state为HWM_STATE_ON时才有数据。 */
    HwmMonitor monitor;                         /**< [en]Index of monitor
                                                     [cn]共享显示器类型（主屏、副屏）。 */
    HwmShareType shareType;                     /**< [en]Share type
                                                     [cn]共享类型。 */
    HWM_HWND appHandle;                         /**< [en]App window handle
                                                     [cn]程序句柄。 */
    HwmStopShareReason stopShareReason;         /**< [en]Stop share reason
                                                     [cn]停止共享原因 */
}HwmShareStateInfo;

/**
* [en]Share state info.
* [cn]接收端共享状态信息
*/
typedef struct tagHwmRecvShareStateInfo
{
    HwmState state;                             /**< [en]The share state
                                                     [cn]共享状态 */
    HWM_CHAR nickName[HWM_MAX_DISPLAY_NAME_LEN];/**< [en]The nickName
                                                     [cn]共享者名称。当state为HWM_STATE_ON时才有数据。 */
    HWM_CHAR number[HWM_MAX_NUMBER_LEN];        /**< [en]The number
                                                     [cn]共享者号码。当state为HWM_STATE_ON时才有数据。 */
}HwmRecvShareStateInfo;

/**
* [en]Remote control state info.
* [cn]共享中远程控制状态信息
*/
typedef struct tagHwmRemoteControlStateInfo
{
    HwmState state;                               /**< [en]The share state
                                                       [cn]状态 */
    HWM_CHAR nickName[HWM_MAX_DISPLAY_NAME_LEN];  /**< [en]The nickName
                                                       [cn]权限所有者名称。 */
    HWM_CHAR number[HWM_MAX_NUMBER_LEN];          /**< [en]The number
                                                       [cn]权限所有者号码。 */
}HwmRemoteControlStateInfo;

/**
* [en]This enum is used to describe timezone 
* [cn]时区
*/
typedef enum tagHwmTimezone
{
    HWM_TIMEZONE_DEFAULT = 0,             /**< [en]Indicates default
                                               [cn]Mediax默认时区值
                                               [ios:rename:Default]*/
    HWM_TIMEZONE_INTERDATE_LINE = 1,      /**< [en]Indicates International Date Line West
                                               [cn]日界线西
                                               [ios:rename:InteerdateLine]*/
    HWM_TIMEZONE_MIDWAY_ISLAND,           /**< [en]Indicates Midway Island, Samoa
                                               [cn]中途岛，萨摩亚群岛
                                               [ios:rename:MidwayIsland]*/
    HWM_TIMEZONE_HAWAII,                  /**< [en]Indicates Hawaii
                                               [cn]夏威夷
                                               [ios:rename:Hawii]*/
    HWM_TIMEZONE_ALASKA,                  /**< [en]Indicates Alaska
                                               [cn]阿拉斯加
                                               [ios:rename:Alaska]*/
    HWM_TIMEZONE_PACIFIC_TIME,            /**< [en]Indicates Pacific Time(US&Canada);Tijuana
                                               [cn]太平洋时间(美国和加拿大);蒂华纳
                                               [ios:rename:PacificTime]*/
    HWM_TIMEZONE_ARIZONA,                 /**< [en]Indicates Arizona
                                               [cn]亚利桑那
                                               [ios:rename:Arizona]*/
    HWM_TIMEZONE_MOUNTAIN_TIME,           /**< [en]Indicates Mountain Time(US&Canada)
                                               [cn]山地时间(美国和加拿大)
                                               [ios:rename:MountainTime]*/
    HWM_TIMEZONE_CENTRAL_AMERICA,         /**< [en]Indicates Central America
                                               [cn]中美洲
                                               [ios:rename:CentralAmerica]*/
    HWM_TIMEZONE_CENTRAL_TIME,             /**< [en]Indicates Central Time(US&Canada)
                                                [cn]中间时间(美国和加拿大)
                                                [ios:rename:CentralTime]*/
    HWM_TIMEZONE_MEXICO_CITY,              /**< [en]Indicates Mexico City
                                                [cn]墨西哥城
                                                [ios:rename:MexicoCity]*/
    HWM_TIMEZONE_SASKATCHEWAN,             /**< [en]Indicates Saskatchewan
                                                [cn]萨斯喀彻温
                                                [ios:rename:Saskatchewan]*/
    HWM_TIMEZONE_BOGOTA,                   /**< [en]Indicates Bogota, Lima, Quito
                                                [cn]波哥大，利马，基多
                                                [ios:rename:Bogota]*/
    HWM_TIMEZONE_EASTERN_TIME,             /**< [en]Indicates Eastern Time(US&Canada)
                                                [cn]东部时间(美国和加拿大)
                                                [ios:rename:EasternTime]*/
    HWM_TIMEZONE_INDIANA,                  /**< [en]Indicates Indiana(East)
                                                [cn]印第安纳(东部)
                                                [ios:rename:Indiana]*/
    HWM_TIMEZONE_ATLANTIC_TIME,            /**< [en]Indicates Atlantic time(Canada)
                                                [cn]大西洋时间(加拿大)
                                                [ios:rename:AtlanticTime]*/
    HWM_TIMEZONE_CARACAS,                  /**< [en]Indicates Caracas, La Paz
                                                [cn]加拉加斯，拉巴斯
                                                [ios:rename:Caracas]*/
    HWM_TIMEZONE_SANTIAGO,                 /**< [en]Indicates Santiago
                                                [cn]圣地亚哥
                                                [ios:rename:Santiago]*/
    HWM_TIMEZONE_NEWFOUNDLAND,             /**< [en]Indicates Newfoundland
                                                [cn]纽芬兰
                                                [ios:rename:Newfoundland]*/
    HWM_TIMEZONE_BRASILIA,                 /**< [en]Indicates Brasilia
                                                [cn]巴西利亚
                                                [ios:rename:Brasilia]*/
    HWM_TIMEZONE_BUENOS_AIRES,             /**< [en]Indicates Buenos Aires, Georgetown
                                                [cn]布宜诺斯艾利斯，乔治敦
                                                [ios:rename:BuenosAires]*/
    HWM_TIMEZONE_GREENLAND,                /**< [en]Indicates Greenland
                                                [cn]格陵兰
                                                [ios:rename:GreenLand]*/
    HWM_TIMEZONE_MID_ATLANTIC,             /**< [en]Indicates Mid-Atlantic
                                                [cn]中大西洋
                                                [ios:rename:MidAtlantic]*/
    HWM_TIMEZONE_AZORES,                   /**< [en]Indicates Azores
                                                [cn]亚速尔群岛
                                                [ios:rename:Azores]*/
    HWM_TIMEZONE_CAPEVERDE,                /**< [en]Indicates Cape Verde Is
                                                [cn]佛得角群岛
                                                [ios:rename:Capeverde]*/
    HWM_TIMEZONE_MONROVIA,                 /**< [en]Indicates Monrovia
                                                [cn]蒙罗维亚
                                                [ios:rename:Monrovia]*/
    HWM_TIMEZONE_GREENWICH_MEANTIME,       /**< [en]Indicates Greenwich Mean Time:Dublin, Edinburgh, Lisbon, London
                                                [cn]格林威治标准时：都柏林，爱丁堡，里斯本，伦敦
                                                [ios:rename:GreenwichMeantime]*/
    HWM_TIMEZONE_AMSTERDAM,                /**< [en]Indicates Amsterdam, Berlin, Bern, Rome, Stockholm, Vienna
                                                [cn]阿姆斯特丹，柏林，伯尔尼，罗马，斯德哥尔摩，维也纳
                                                [ios:rename:Amsterdam]*/
    HWM_TIMEZONE_BELGRADE,                 /**< [en]Indicates Belgrade, Bratislava, Budapest, Ljubljana, Prague
                                                [cn]贝尔格莱德，布拉迪斯拉发，布达佩斯，卢布尔亚纳，布拉格
                                                [ios:rename:Belgrade]*/
    HWM_TIMEZONE_BRUSSELS,                 /**< [en]Indicates Brussels, Copenhagen, Madrid, Paris
                                                [cn]布鲁塞尔，哥本哈根，马德里，巴黎
                                                [ios:rename:Brussels]*/
    HWM_TIMEZONE_SARAJEVO,                 /**< [en]Indicates Sarajevo, Skopje, Sofija, Vilnius, Warsaw, Zagreb
                                                [cn]萨拉热窝，斯科普里，维尔纽斯，索非亚，华沙，萨格勒布
                                                [ios:rename:Sarajevo]*/
    HWM_TIMEZONE_WESTCENTRAL_AFRICA,       /**< [en]Indicates West Central Africa
                                                [cn]中非西部
                                                [ios:rename:WestcentralAfrica]*/
    HWM_TIMEZONE_ATHENS,                   /**< [en]Indicates Athens, Istanbul, Minsk
                                                [cn]雅典，伊斯坦布尔，明斯克
                                                [ios:rename:Athens]*/
    HWM_TIMEZONE_BUCHAREST,                /**< [en]Indicates Bucharest
                                                [cn]布加勒斯特
                                                [ios:rename:Bucharest]*/
    HWM_TIMEZONE_CAIRO,                    /**< [en]Indicates Cairo
                                                [cn]开罗
                                                [ios:rename:Cairo]*/
    HWM_TIMEZONE_HARARE,                   /**< [en]Indicates Harare, Pretoria
                                                [cn]哈拉雷，比勒陀利亚
                                                [ios:rename:Harare]*/
    HWM_TIMEZONE_HELSINKI,                 /**< [en]Indicates Helsinki, Riga, Tallinn
                                                [cn]赫尔辛基，里加，塔林
                                                [ios:rename:Helsinki]*/
    HWM_TIMEZONE_JERUSALEM,                /**< [en]Indicates Jerusalem
                                                [cn]耶路撒冷
                                                [ios:rename:Jerusalem]*/
    HWM_TIMEZONE_BAGHDAD,                  /**< [en]Indicates Baghdad
                                                [cn]巴格达
                                                [ios:rename:Baghdad]*/
    HWM_TIMEZONE_KUWAIT,                   /**< [en]Indicates Kuwait, Riyadh
                                                [cn]科威特，利雅得
                                                [ios:rename:Kuwait]*/
    HWM_TIMEZONE_MOSCOW,                   /**< [en]Indicates Moscow, St. Petersburg, Volgograd
                                                [cn]莫斯科，圣彼得堡，喀山，伏尔加格勒
                                                [ios:rename:Moscow]*/
    HWM_TIMEZONE_NAIROBI,                  /**< [en]Indicates Nairobi
                                                [cn]内罗毕
                                                [ios:rename:Nairobi]*/
    HWM_TIMEZONE_TEHRAN,                   /**< [en]Indicates Tehran
                                                [cn]德黑兰
                                                [ios:rename:Tehran]*/
    HWM_TIMEZONE_ABU_DHABI,                /**< [en]Indicates Abu Dhabi, Muscat
                                                [cn]阿布扎比，马斯喀特
                                                [ios:rename:AbuDhabi]*/
    HWM_TIMEZONE_TBILISI,                  /**< [en]Indicates Tbilisi, Yerevan
                                                [cn]第比利斯，埃里温
                                                [ios:rename:Tbilisi]*/
    HWM_TIMEZONE_KABUL,                    /**< [en]Indicates Kabul
                                                [cn]喀布尔
                                                [ios:rename:Kabul]*/
    HWM_TIMEZONE_EKATERINBURG,             /**< [en]Indicates Ekaterinburg
                                                [cn]叶卡捷琳堡
                                                [ios:rename:Ekaterinburg]*/
    HWM_TIMEZONE_ISLAMABAD,                /**< [en]Indicates Islamabad, Karachi, Tashkent
                                                [cn]伊斯兰堡，卡拉奇，塔什干
                                                [ios:rename:Islamabad]*/
    HWM_TIMEZONE_CALCUTTA,                 /**< [en]Indicates Calcutta, Chennai, Mumbai, New Delhi
                                                [cn]加尔各答，马德拉斯，孟买，新德里
                                                [ios:rename:Calcutta]*/
    HWM_TIMEZONE_KATHMANDU,                /**< [en]Indicates Kathmandu
                                                [cn]加德满都
                                                [ios:rename:Kathmandu]*/
    HWM_TIMEZONE_ALMATY,                   /**< [en]Indicates Almaty
                                                [cn]阿拉木图
                                                [ios:rename:Almaty]*/
    HWM_TIMEZONE_ASTANA,                   /**< [en]Indicates Astana, Dhaka
                                                [cn]阿斯塔纳，达卡
                                                [ios:rename:Astana]*/
    HWM_TIMEZONE_SRIJAYA,                  /**< [en]Indicates Sri Jayawardenepura
                                                [cn]斯里哈亚华登尼普拉
                                                [ios:rename:Srijaya]*/
    HWM_TIMEZONE_RANGOON,                  /**< [en]Indicates Rangoon
                                                [cn]仰光
                                                [ios:rename:Rangoon]*/
    HWM_TIMEZONE_BANGKOK,                  /**< [en]Indicates Bangkok, Hanoi, Jakarta
                                                [cn]曼谷，雅加达，河内
                                                [ios:rename:Bangkok]*/
    HWM_TIMEZONE_NOVOSIBIRSK,              /**< [en]Indicates Novosibirsk
                                                [cn]新西伯利亚
                                                [ios:rename:Novosibirsk]*/
    HWM_TIMEZONE_BEIJING,                  /**< [en]Indicates Beijing, Chongqing, Hong Kong, Urumqi, Taipei
                                                [cn]北京，重庆，香港，乌鲁木齐，台北
                                                [ios:rename:Beijing]*/
    HWM_TIMEZONE_KRASNOYARSK,              /**< [en]Indicates Krasnoyarsk, Ulaan Bataar
                                                [cn]克拉斯诺亚尔斯克，乌兰巴托
                                                [ios:rename:Krasnoyarsk]*/
    HWM_TIMEZONE_KUALA_LUMPUR,             /**< [en]Indicates Kuala Lumpur, Singapore
                                                [cn]吉隆坡，新加坡
                                                [ios:rename:KualaLumpur]*/
    HWM_TIMEZONE_PERTH,                    /**< [en]Indicates Perth
                                                [cn]珀斯
                                                [ios:rename:Perth]*/
    HWM_TIMEZONE_OSAKA,                    /**< [en]Indicates Osaka, Sapporo, Tokyo
                                                [cn]东京，大阪，札幌
                                                [ios:rename:Osaka]*/
    HWM_TIMEZONE_SEOUL,                    /**< [en]Indicates Seoul
                                                [cn]汉城
                                                [ios:rename:Seoul]*/
    HWM_TIMEZONE_YAKUTSK,                  /**< [en]Indicates Yakutsk
                                                [cn]雅库茨克
                                                [ios:rename:Yakutsk]*/
    HWM_TIMEZONE_ADELAIDE,                 /**< [en]Indicates Adelaide
                                                [cn]阿德莱德
                                                [ios:rename:Adelaide]*/
    HWM_TIMEZONE_DARWIN,                   /**< [en]Indicates Darwin
                                                [cn]达尔文
                                                [ios:rename:Darwin]*/
    HWM_TIMEZONE_BRISBANE,                 /**< [en]Indicates Brisbane
                                                [cn]布里斯班
                                                [ios:rename:Brisbane]*/
    HWM_TIMEZONE_CANBERRA,                 /**< [en]Indicates Canberra, Melbourne, Sydney
                                                [cn]堪培拉，墨尔本，悉尼
                                                [ios:rename:Canberra]*/
    HWM_TIMEZONE_GUAM,                     /**< [en]Indicates Guam, Port Moresby
                                                [cn]关岛，莫尔兹比港
                                                [ios:rename:Guam]*/
    HWM_TIMEZONE_HOBART,                   /**< [en]Indicates Hobart
                                                [cn]霍巴特
                                                [ios:rename:Hobart]*/
    HWM_TIMEZONE_VLADIVOSTOK,              /**< [en]Indicates Vladivostok
                                                [cn]符拉迪沃斯托克
                                                [ios:rename:Vladivostok]*/
    HWM_TIMEZONE_SOLOMON,                  /**< [en]Indicates Solomon Is., New Caledonia
                                                [cn]所罗门群岛，新喀里多尼亚
                                                [ios:rename:Solomon]*/
    HWM_TIMEZONE_AUCKLAND,                 /**< [en]Indicates Auckland, Welington
                                                [cn]奥克兰，惠灵顿
                                                [ios:rename:Auckland]*/
    HWM_TIMEZONE_FIJI,                     /**< [en]Indicates Fiji, Kamchatka, Marshall Is.
                                                [cn]富士，堪察加半岛，马绍尔群岛
                                                [ios:rename:Fiji]*/
    HWM_TIMEZONE_NUKUALOFA,                /**< [en]Indicates Nuku'alofa
                                                [cn]努库阿洛法
                                                [ios:rename:Nukualofa]*/
    HWM_TIMEZONE_IRKUTSK,                  /**< [en]Indicates Irkutsk
                                                [cn]伊尔库茨克
                                                [ios:rename:Irkutsk]*/
    HWM_TIMEZONE_CASABLANCA,               /**< [en]Indicates Casablanca
                                                [cn]卡萨布兰卡
                                                [ios:rename:Casablanca]*/
    HWM_TIMEZONE_BAKU,                     /**< [en]Indicates Baku
                                                [cn]巴库
                                                [ios:rename:Baku]*/
    HWM_TIMEZONE_MAGADAN,                  /**< [en]Indicates Magadan
                                                [cn]马加丹
                                                [ios:rename:Magadan]*/
}HwmTimezone;

/**
* [en]This struct is used to describe conf common param.
* [cn]会议信息共有参数
*/
typedef struct tagHwmConfCommonParam
{
    HWM_CHAR subject[HWM_MAX_SUBJECT_LEN];            /**< [en]Indicates the conference subject.
                                                           [cn]会议主题。 
                                                           [cs_sensitive:name][cs_allowNull:N][cs_length:1-385] */
    HwmMediaType mediaType;                           /**< [en]The media type
                                                           [cn]会议类型。默认音频会议。
                                                           [cs_allowNull:Y] */
    HWM_BOOL needPassword;                            /**< [en]Is need password
                                                           [cn]是否需要来宾密码。默认不需要来宾密码（仅对随机会议ID生效）。
                                                           [cs_allowNull:Y] */
    HWM_BOOL isAutoRecord;                            /**< [en]Is autorecord
                                                           [cn]会议自动录制开关。默认不开启。
                                                               开启会议自动录制开关，内部会默认支持录制会议，录制会议开关的打开与关闭并不影响。
                                                               说明：仅对云录制生效，客户端本地录制不生效。
                                                           [cs_allowNull:Y] */
    HWM_BOOL allowRecord;                             /**< [en]Is allow record(only for win)
                                                           [cn]是否支持录制会议。(仅对win平台生效)
                                                               说明：仅对云录制生效，客户端本地录制不生效。
                                                           [cs_allowNull:Y] */
    HWM_BOOL allowLive;                               /**< [en]Reserved.This field is used to transparently transmit conference details when editing a conference.
                                                           [cn]保留。编辑会议时透传会议详情此字段
                                                           [cs_allowNull:Y] */
    HwmTimezone timezone;                             /**< [en]Indicates the time zone used in conference notification.
                                                           [cn]时区码。用于发送会议通知。
                                                           [cs_allowNull:N] */
    HWM_CHAR vmrId[HWM_MAX_VMR_ID_LEN];               /**< [en]Indicates VMR conference id .
                                                           [cn]云会议室唯一ID。预约并发会议时，设置为空字符串。编辑会议时不可修改，填入会议详情返回值。
                                                           [cs_allowNull:Y] */
    HwmConfAllowJoinUserType confAllowJoinUser;       /**< [en]Indicates the user type allowed to join the conference.
                                                           [cn]允许入会范围限制。默认所有用户。
                                                           [cs_allowNull:Y]*/
    HWM_BOOL isSendSms;                               /**< [en]Indicates whether it sends the short message.
                                                           [cn]发送短信通知开关。企业配置里面需要开启，否则不生效。
                                                           [cs_allowNull:Y]*/
    HWM_BOOL isSendEmail;                             /**< [en]Indicates whether is send email notify. 
                                                           [cn]发送邮件通知开关。
                                                           [cs_allowNull:Y]*/
    HwmVmrConfIdType vmrConfIdType;                   /**< [en]VMR conf Id type.
                                                           [cn]云会议室的会议ID类型。编辑会议时不可修改会议详情中vmrConfId非空，填入HWM_VMR_CONF_ID_TYPE_FIXED；
                                                               会议详情中vmrConfId为空，填入HWM_VMR_CONF_ID_TYPE_RANDOM
                                                           [cs_allowNull:Y] */
    HWM_CHAR guestPwd[HWM_MAX_CONF_PASSWORD_LEN];     /**< [en]Indicates the password for common participants.If the password is null, 
                                                               the server will generate a password randomly
                                                           [cn]来宾密码，如果空则服务器随机生成（仅对随机会议ID生效）。
                                                           [cs_allowNull:Y][cs_length:0-32][cs_sensitive:hide][win:todo] */
    HWM_BOOL isOpenWaitingRoom;                       /**< [en]Indicates whether is open waiting room
                                                           [cn]等候室开启开关。
                                                               说明：需要等候室功能开通后才能生效
                                                           [cs_allowNull:Y] */
    HWM_UINT32 numOfAttendee;                         /**< [en]length for attendees.
                                                           [cn]与会者列表长度。若不需要带与会者，参数值可以为0。
                                                           [cs_allowNull:N][length for attendees] */
    HwmAttendeeBaseInfo* attendees;                   /**< [en]Attendees info. If no attendees are included, leave the parameter empty.
                                                           [cn]与会者列表。若不需要带与会者，参数值可以为空。
                                                           [cs_allowNull:Y] */
    HWM_BOOL allowGuestStartConf;                     /**< [en]Indicates allow guest start conf. 
                                                           [cn]允许来宾启动会议
                                                           [cs_allowNull:Y] */
    HWM_UINT32 allowGuestStartConfTime;               /**< [en]Indicates join before host time.
                                                           [cn]允许来宾提前入会时间范围(单位：分钟) 0-随时，n-提前n分钟启动会议
                                                           [cs_allowNull:Y] */
    HWM_BOOL forbiddenScreenShots;                    /**< [en]Indicates is whether disable screenshots
                                                           [cn]是否禁止截屏 
                                                           [cs_allowNull:Y] */
    HWM_UINT32 concurrentParticipants;                /**< [en]concurrent participants.
                                                           [cn]会议方数，会议最大与会人数限制. 0: 无限制, 大于0: 会议最大与会人数*/
    HWM_CHAR customInfo[HWM_MAX_CUSTOM_INFO_LEN];     /**< [en]User-defined information
                                                           [cn]用户自定义信息。
                                                           [cs_allowNull:Y] */
    HwmConfResType confResType;                       /**< [en]Indicates conf resource type
                                                           [cn]会议资源类型
                                                           [cs_allowNull:Y] */
    HwmSummaryState defaultSummaryState;              /**< [en]Indicates record include summary by default
                                                           [cn]会议云录制是否包含纪要，初始值（周期会议暂不支持）
                                                           [cs_allowNull:Y] */
    HwmAutoMuteType autoMuteMode;                     /**< [en]Indicates soft terminal auto mute
                                                           [cn]来宾入会，软终端是否自动静音
                                                           [cs_allowNull:Y] */
    HwmAutoMuteType hardTerminalAutoMuteMode;         /**< [en]Indicates hard terminal auto mute
                                                           [cn]来宾入会，硬终端是否自动静音
                                                           [cs_allowNull:Y] */
    HWM_BOOL supportMultiShare;                       /**< [en]Indicates support multi-share.
                                                           [cn]是否支持多人共享
                                                           [cs_allowNull:Y] [doc:disable] */
    HWM_BOOL supportWatermark;                        /**< [en]Indicates is whether support watermark
                                                           [cn]是否支持水印
                                                           [cs_allowNull:Y] */
    HWM_BOOL autoPublishSummary;                      /**< [en]Indicates Whether auto publish summary.
                                                           [cn]是否自动发布纪要（免人工审核）
                                                           [cs_allowNull:Y] */
    HWM_BOOL attendeesExcludeSelf;                    /**< [en]Indicates does the attendee list exclude self.
                                                           [cn]与会者列表是否排除自己，默认不排除自己
                                                           [cs_allowNull:Y] */
    HWM_BOOL supportDataGuardAudio;                   /**< [en]Indicates whether is support data guard audio.
                                                           [cn]是否支持音频暗水印
                                                           [cs_allowNull:Y][doc:disable] */
    HWM_BOOL supportDataGuardVideo;                   /**< [en]Indicates whether is support data guard video.
                                                           [cn]是否支持视频暗水印
                                                           [cs_allowNull:Y][doc:disable] */
}HwmConfCommonParam;

/**
* [en]Book conf parameter.
* [cn]预约会议参数
*/
typedef struct tagHwmBookConfParam
{
    HWM_INT64 startTime;                        /**< [en]Indicates conference start time.
                                                     [cn]（已废弃，推荐使用字段startTimeStamp）会议开始时间时间戳(utc时间)，单位是秒。说明：UTC时间与UTC偏移量（转换为秒）的差值，如东八区为 UTC时间戳 - (8 x 60 x 60)。
                                                     [cs_allowNull:Y] */
    HWM_INT64 startTimeStamp;                   /**< [en]Indicates conference start time
                                                     [cn]会议开始时间，startTimeStamp、startTime二选一，优先使用startTimeStamp(时间戳，单位秒)
                                                     [cs_allowNull:Y] */
    HWM_UINT32 duration;                        /**< [en]Indicates conference length of time, unit is minute  ,this param is optional. 
                                                     [cn]会议时长，单位分钟，最小值15分钟，最大值1440分钟。
                                                     [cs_allowNull:N] */
    HwmConfCommonParam confCommonParam;         /**< [en]Indicates conf common param. 
                                                     [cn]会议公用参数
                                                     [cs_allowNull:N] */
    HWM_BOOL isSendCalendar;                    /**< [en]Indicates whether is send calendar notify. 
                                                     [cn]发送邮件日历开关。
                                                     [cs_allowNull:Y] */
}HwmBookConfParam;

/**
* [en]Cancel conf parameter.
* [cn]取消会议参数
*/
typedef struct tagHwmCancelConfParam
{
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];                    /**< [en]Conf id
                                                                  [cn]会议ID。
                                                                  [cs_allowNull:N] */
    HWM_BOOL isForceCancel;                                  /**< [en]force cancel(only for win) 
                                                                  [cn]是否强制取消，会议开始后可以强制取消(仅对win平台生效)
                                                                  [cs_allowNull:Y][mac:todo] */
}HwmCancelConfParam;

/**
* [en]Edit(modify) conf parameter.
* [cn]编辑（修改）已经预约的会议
*/
typedef struct tagHwmEditConfParam
{
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];           /**< [en]Indicates the conference id.
                                                         [cn]会议ID。
                                                         [cs_allowNull:N] */
    HWM_INT64 startTime;                            /**< [en]Indicates conference start time.
                                                         [cn]（已废弃，推荐使用字段startTimeStamp）会议开始时间，如果没有指定开始时间，则表示会议马上开始，精度秒(0时区)。说明：UTC时间与UTC偏移量（转换为秒）的差值，如东八区为 UTC时间戳 - (8 x 60 x 60)。
                                                         [cs_allowNull:Y] */
    HWM_INT64 startTimeStamp;                       /**< [en]Indicates conference start time
                                                         [cn]会议开始时间，startTimeStamp、startTime二选一，优先使用startTimeStamp(时间戳，单位秒)
                                                         [cs_allowNull:Y] */
    HWM_UINT32 duration;                            /**< [en]Indicates conference length of time, unit is minute
                                                         [cn]会议持续时长，单位分钟
                                                         [cs_allowNull:N] */
    HwmConfCommonParam confCommonParam;             /**< [en]Indicates conf common param.
                                                         [cn]会议公用参数
                                                         [cs_allowNull:N] */
    HWM_BOOL isSendCalendar;                        /**< [en]Indicates whether is send calendar notify.
                                                         [cn]是否发送日历通知
                                                         [cs_allowNull:Y] */
}HwmEditConfParam;

/**
* [en]Get conf detail param.
* [cn]获取会议详情参数体
*/
typedef struct tagHwmGetConfDetailParam
{
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];                /**< [en]Conf id
                                                              [cn]会议ID。
                                                              [cs_allowNull:N] */
}HwmGetConfDetailParam;

/**
* [en]Change vmr info param.
* [cn]修改云会议室参数体
*/
typedef struct tagHwmChangeVmrInfoParam
{
    HWM_CHAR vmrId[HWM_MAX_VMR_ID_LEN];                         /**< [en]Indicates the id of the vmr.
                                                                     [cn]云会议室唯一ID。
                                                                     [cs_allowNull:N] */
    HWM_CHAR vmrName[HWM_MAX_VMR_NAME_LEN];                     /**< [en]Indicates the name of the vmr.
                                                                     [cn]云会议室名称。
                                                                     [cs_allowNull:N] */
    HWM_CHAR guestPwd[HWM_MAX_PASSWORD_LEN];                    /**< [en]Indicates the guest password of the vmr.
                                                                     [cn]云会议室来宾密码。
                                                                     [cs_allowNull:N] */
    HWM_CHAR hostPwd[HWM_MAX_PASSWORD_LEN];                     /**< [en]Indicates the host password of the vmr.
                                                                     [cn]云会议室主持人密码。
                                                                     [cs_allowNull:N] */
    HWM_BOOL allowGuestStartConf;                               /**< [en]Indicates whether is allow guest start conf.
                                                                     [cn]是否允许来宾提前启动会议
                                                                     [cs_allowNull:Y] */
    HwmConfAllowJoinUserType confAllowJoinUser;                 /**< [en]Indicates the user type allowed to join the conference.
                                                                     [cn]允许入会范围限制。默认所有用户。
                                                                     [cs_allowNull:Y]*/
    HWM_BOOL isOpenWaitingRoom;                                 /**< [en]Indicates whether is open waiting room
                                                                     [cn]等候室开启开关。
                                                                         说明：需要等候室功能开通后才能生效
                                                                     [cs_allowNull:Y]*/
}HwmChangeVmrInfoParam;

/**
* [en]This enum is used to describe conference state.
* [cn]会议状态
*/
typedef enum tagHwmConfState
{
    HWM_CONF_STATE_SCHEDULE = 0,        /**< [en]Indicates schedule state
                                             [cn]预定状态。 */
    HWM_CONF_STATE_CREATING,            /**< [en]Indicates be creating state
                                             [cn]正在创建状态。 */
    HWM_CONF_STATE_GOING,               /**< [en]Indicates conf going state
                                             [cn]会议已经开始。 */
    HWM_CONF_STATE_DESTROYED,           /**< [en]Indicates conf is destroyed
                                             [cn]会议已经关闭。 */
}HwmConfState;

/** 
* [en]Indicates conference type.
* [cn]会议类型
*/
typedef enum tagHwmConferenceType
{
    HWM_CONF_TYPE_COMMON = 0,       /**< [en]Indicates common conference.
                                         [cn]普通会议 */
    HWM_CONF_TYPE_CYCLE,            /**< [en]Indicates cycle conference.
                                         [cn]周期会议 */
} HwmConferenceType;

/**
* [en]Record auth type.
* [cn]录播鉴权方式
*/
typedef enum tagHwmRecordAuthType
{
    REOCRD_AUTH_TYPE_ANYONE = 0,        /**< [en]
                                             [cn]可通过连接观看/下载（不拼接nonce）*/
    REOCRD_AUTH_TYPE_IN_COMPANY_USER,   /**< [en]
                                             [cn]企业用户可观看/下载 */
    REOCRD_AUTH_TYPE_ATTENDEE,          /**< [en]
                                             [cn]与会者可观看/下载*/
} HwmRecordAuthType;

/** 
* [en]Indicates cycle conference param.
* [cn]周期子会议的参数
*/
typedef struct tagHwmSubCycleConfParam
{
    HWM_CHAR subConfID[HWM_MAX_CONF_UUID_LEN];          /**< [en]Indicates conference uuid. 
                                                             [cn]子会议唯一标识UUID */
    HwmMediaType mediaType;                             /**< [en]Indicates media type
                                                             [cn]媒体类型*/
    HWM_INT64 startTime;                                /**< [en]Indicates conference start time.
                                                             [cn]（已废弃，推荐使用字段startTimeStamp）会议起始时间 */
    HWM_INT64 endTime;                                  /**< [en]Indicates conference end time.
                                                             [cn]（已废弃，推荐使用字段endTimeStamp）会议结束时间 */
    HWM_INT64 startTimeStamp;                           /**< [en]Indicates conference start time
                                                             [cn]会议开始时间(时间戳，单位秒) */
    HWM_INT64 endTimeStamp;                             /**< [en]Indicates conference end time
                                                             [cn]会议结束时间(时间戳，单位秒) */
    HWM_BOOL isAutoRecord;                              /**< [en]Indicates whether record automatically when conference begin.
                                                             [cn]会议开始后是否自动开启录制  */
    HwmRecordAuthType recordAuthType;                   /**< [en]Indicates record auth type.
                                                             [cn]录播鉴权方式 */
    HwmConfAllowJoinUserType confAllowJoinUser;         /**< [en]Indicates the user type allowed to join the conference.
                                                             [cn]允许入会范围限制。默认所有用户。 */
    HWM_BOOL allowGuestStartConf;                       /**< [en]Indicates is allow guest start conf. 
                                                             [cn]允许来宾启动会议 */
    HWM_UINT32 allowGuestStartConfTime;                 /**< [en]Indicates join before host time. 
                                                             [cn]允许来宾提前入会时间范围(单位：分钟) 0-随时，n-提前n分钟启动会议*/
}HwmSubCycleConfParam;

/**
* [en]cycle type.
* [cn]周期类型
*/
typedef enum tagHwmCycleType
{
    CYCLE_TYPE_DAY = 0,        /**< [en]Indicates cycle by day.
                                    [cn]以天为周期 */
    CYCLE_TYPE_WEEK,           /**< [en]Indicates cycle by week.
                                    [cn]以周为周期 */
    CYCLE_TYPE_MONTH,          /**< [en]Indicates cycle by month.
                                    [cn]以月为周期 */
}HwmCycleType;

/**
* [en]Indicates cycle conference param.
* [cn]周期会议的参数
*/
typedef struct tagHwmCycleConfParam
{
    HWM_INT64 startDate;                                /**< [en]Indicates conference start date, .
                                                             [cn]（已废弃，推荐使用字段startTimeStamp）开始日期-时间戳，精度秒(0时区)
                                                             [cs_allowNull:Y] */
    HWM_INT64 endDate;                                  /**< [en]Indicates conference end date.
                                                             [cn]（已废弃，推荐使用字段endTimeStamp）结束日期-时间戳，精度秒(0时区)
                                                             [cs_allowNull:Y] */
    HWM_INT64 startTimeStamp;                           /**< [en]Indicates conference start time
                                                             [cn]开始时间，startTimeStamp、startDate二选一，优先使用startTimeStamp(时间戳，单位秒)
                                                             [cs_allowNull:Y] */
    HWM_INT64 endTimeStamp;                             /**< [en]Indicates conference end time
                                                             [cn]结束时间，endTimeStamp、endDate二选一，优先使用endTimeStamp(时间戳，单位秒)
                                                             [cs_allowNull:Y] */
    HwmCycleType cycleType;                             /**< [en]Indicates cycle type.
                                                             [cn]周期类型 */
    HWM_UINT32 interval;                                /**< [en]Indicates cycle interval.
                                                             [cn]周期区间  1、周期类型选择了按天，表示每几天召开一次，取值范围[1,15]；2、周期类型选择了Week，表示每几周召开一次，取值范围[1,5]；3、周期类型选择了Month，interval表示隔几月，取值范围[1,3] */
    HWM_CHAR listPoints[HWM_MAX_CONF_POINTS_LENGTH];    /**< [en]Indicates conference points.
                                                             [cn]周期内的会议召开点。仅当按周和月时有效。数据格式(中间用逗号隔开):1,3,5,7 */
    HWM_UINT32 preRemindDays;                           /**< [en]Indicates prior notificate days.
                                                             [cn]周期子会议提前通知天数。 */
}HwmCycleConfParam;

/**
* [en]Conf list information.
* [cn]会议列表信息
*/
typedef struct tagHwmConfListItem
{
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];                   /**< [en]Indicates conference id
                                                                 [cn]会议ID。 */
    HWM_CHAR subject[HWM_MAX_SUBJECT_LEN];                  /**< [en]Indicates conference subject
                                                                 [cn]会议主题。 */
    HWM_CHAR hostPwd[HWM_MAX_CONF_PASSWORD_LEN];            /**< [en]Indicates host pwd.
                                                                 [cn]主持人密码。
                                                                 [cs_sensitive:hide] */
    HWM_CHAR guestPwd[HWM_MAX_CONF_PASSWORD_LEN];           /**< [en]Indicates guest pwd.
                                                                 [cn]来宾密码。
                                                                 [cs_sensitive:hide] */
    HWM_CHAR audiencePwd[HWM_MAX_CONF_PASSWORD_LEN];		/**< [en]Indicates the password of audience(only for win)
                                                                 [cn]观众密码。(仅对win平台生效)
                                                                 [cs_sensitive:hide][mac:todo] */
    HWM_INT64 startTime;                                    /**< [en]Indicates conference start time
                                                                 [cn]（已废弃，推荐使用字段startTimeStamp）会议开始时间(utc时间)，单位秒。*/
    HWM_INT64 endTime;                                      /**< [en]Indicates conference end time
                                                                 [cn]（已废弃，推荐使用字段endTimeStamp）会议结束时间(utc时间)，单位秒。 */
    HWM_INT64 startTimeStamp;                               /**< [en]Indicates conference start time
                                                                 [cn]会议开始时间(时间戳，单位秒) */
    HWM_INT64 endTimeStamp;                                 /**< [en]Indicates conference end time
                                                                 [cn]会议结束时间(时间戳，单位秒) */
    HwmTimezone timezone;                                   /**< [en]Indicates the time zone used in conference notification(only for win)
                                                                 [cn]时区码(仅对win平台生效)
                                                                 [mac:todo] */
    HwmMediaType mediaType;                                 /**< [en]Indicates media type
                                                                 [cn]媒体类型*/
    HWM_CHAR schedulerName[HWM_MAX_USER_NAME_LEN];          /**< [en]Indicates scheduler name
                                                                 [cn]会议预定者名称。 */
    HWM_CHAR vmrConfId[HWM_MAX_VMR_CONF_ID_LEN];            /**< [en]Indicates VMR conference id
                                                                 [cn]用于云会议显示的ID。 */
    HWM_CHAR hostJoinUri[HWM_MAX_URL_LEN];					/**< [en]Indicates the uri for host to join conference.
                                                                 [cn]主持人加入会议uri链接
                                                                 [cs_sensitive:ip] */
    HWM_CHAR guestJoinUri[HWM_MAX_URL_LEN];                 /**< [en]the uri used by guest to join conference.
                                                                 [cn]来宾加入会议uri链接
                                                                 [cs_sensitive:ip] */
    HWM_CHAR audienceJoinUri[HWM_MAX_URL_LEN];				/**< [en]Indicates the uri for audience to join conference(only for win)
                                                                 [cn]观众加入会议的链接(仅对win平台生效)
                                                                 [cs_sensitive:ip][mac:todo] */
    HwmConfState confStateType;                             /**< [en]Indicates conference state(only for win)
                                                                 [cn]会议状态(仅对win平台生效)
                                                                 [mac:todo] */
    HWM_BOOL isWebinar;										/**< [en]Indicates whether the conference is web conferencre(only for win)
                                                                 [cn]是否是网络研讨会(仅对win平台生效)
                                                                 [mac:todo] */
    HwmConfRole selfConfRole;							    /**< [en]Indicates self role.
                                                                 [cn]本人在会议中的角色 */
    HwmConferenceType conferenceType;                       /**< [en]Indicates conference type.
                                                                 [cn]会议类型 */
    HwmCycleConfParam cycleParam;                           /**< [en]Indicates cycle conference param.
                                                                 [cn]周期会议的参数 */
    HWM_UINT32 subConfSize;                                 /**< [en]Indicates cycle sub conference size.
                                                                 [cn]周期子会议数量
                                                                 [length for subConfParam] */
    HwmSubCycleConfParam *subConfParam;                     /**< [en]Indicates cycle sub conference param.
                                                                 [cn]周期子会议参数 */
    HWM_CHAR customInfo[HWM_MAX_CUSTOM_INFO_LEN];           /**< [en]User-defined information
                                                                 [cn]用户自定义信息。 */
} HwmConfListItem;

/**
* [en]Conference information in detail.
* [cn]会议详细信息
*/
typedef struct tagHwmConfDetail
{
    HwmConfListItem confListInfo;                       /**< [en]Indicates conference basic info
                                                             [cn]会议基本信息 */
    HWM_CHAR vmrId[HWM_MAX_VMR_ID_LEN];                 /**< [en]Indicates VMR conference id
                                                             [cn]云会议室唯一ID。 */
    HWM_BOOL vmrFlag;                                   /**< [en]Indicates whether is VMR conference
                                                             [cn]是否是vmr会议。 */
    HWM_BOOL isAutoRecord;                              /**< [en]whether auto record the conference
                                                             [cn]是否开启自动录制会议。 */
    HWM_BOOL allowRecord;                               /**< [en]Is allow record
                                                             [cn]是否支持录制会议 */
    HWM_BOOL allowLive;                                 /**< [en]Reserved.This field is used to transparently transmit conference details when editing a conference.
                                                           [cn]保留。编辑会议时透传会议详情此字段 */
    HWM_BOOL needPassword;                              /**< [en]Is need password
                                                             [cn]是否需要密码。 */
    HWM_BOOL isSendSms;                                 /**< [en]Indicates whether it sends the short message.
                                                             [cn]是否发送短信通知。 */
    HWM_BOOL isSendEmail;                               /**< [en]Indicates whether is send email notify.
                                                             [cn]是否发送邮件通知。 */
    HWM_BOOL isSendCalendar;                            /**< [en]Indicates whether is send calendar notify
                                                             [cn]是否发送日历通知。 */
    HwmConfAllowJoinUserType confAllowJoinUser;         /**< [en]Indicates the user type allowed to join the conference.
                                                             [cn]会议准入设置。 */
    HWM_UINT32 numOfAttendee;                           /**< [en]length for attendees.
                                                             [cn]与会者列表长度。
                                                             [length for attendees] */
    HwmAttendeeBaseInfo* attendees;                     /**< [en]Attendees info
                                                             [cn]与会者列表。 */
    HWM_BOOL isOpenWaitingRoom;                         /**< [en]Indicates waiting room state
                                                             [cn]是否开启等候室 */
    HWM_BOOL allowGuestStartConf;                       /**< [en]Indicates allow guest start conf.
                                                             [cn]允许来宾启动会议 */
    HWM_BOOL forbiddenScreenShots;                      /**< [en]Indicates is whether disable screenshots
                                                             [cn]是否禁止截屏 */                                                         
    HWM_UINT32 allowGuestStartConfTime;                 /**< [en]Indicates join before host time.
                                                             [cn]允许来宾提前入会时间范围(单位：分钟) 0-随时，n-提前n分钟启动会议 */
    HwmVmrConfIdType vmrConfIdType;                     /**< [en]VMR conf Id type.
                                                             [cn]云会议室的会议ID类型 */
    HWM_UINT32 concurrentParticipants;                  /**< [en]concurrent participants.
                                                             [cn]会议方数，会议最大与会人数限制. 0: 无限制, 大于0: 会议最大与会人数*/
    HwmSummaryState defaultSummaryState;                /**< [en]Indicates record include summary by default
                                                             [cn]会议云录制是否包含纪要，初始值（周期会议暂不支持）*/
    HwmAutoMuteType autoMuteMode;                       /**< [en]Indicates is auto mute of soft terminal
                                                             [cn]软终端是否自动静音*/
    HwmAutoMuteType hardTerminalAutoMuteMode;           /**< [en]Indicates is auto mute of hard terminal
                                                             [cn]硬终端是否自动静音*/
    HWM_BOOL supportMultiShare;                         /**< [en]Indicates support multi-share.
                                                             [cn]是否支持多人共享
                                                             [cs_allowNull:Y] [doc:disable] */
    HWM_BOOL supportWatermark;                          /**< [en]Indicates is whether support watermark
                                                             [cn]是否支持水印*/
    HWM_BOOL autoPublishSummary;                        /**< [en]Indicates Whether auto publish summary. 
                                                             [cn]是否自动发布纪要（免人工审核）*/
    HWM_BOOL supportDataGuardAudio;                     /**< [en]Indicates whether is support data guard audio.
                                                             [cn]是否支持音频暗水印
                                                             [doc:disable] */
    HWM_BOOL supportDataGuardVideo;                     /**< [en]Indicates whether is support data guard video.
                                                             [cn]是否支持视频暗水印
                                                             [doc:disable] */
} HwmConfDetail;

/**
* [en]vmr id type.
* [cn]vmr id类型
*/
typedef enum tagHwmVmrIdType
{
    RANDOM_MEETING_ID = 0,                   /**< [en]random meeting id
                                                  [cn]随机会议id */
    PERSONAL_MEETING_ID,                     /**< [en] personal meeting id
                                                  [cn]个人会议id */
    CLOUD_MEETING_ROOM_ID,                   /**< [en]vmr meeting id
                                                  [cn]云会议室id */
} HwmVmrIdType;

/**
* [en]Vmr information.
* [cn]云会议室信息
*/
typedef struct tagHwmVmrInfo
{
    HWM_CHAR vmrConfId[HWM_MAX_CONF_ID_LEN];              /**< [en]Indicates vmr conference id
                                                               [cn]云会议室上的固定会议ID。 */
    HWM_CHAR vmrId[HWM_MAX_VMR_CONF_ID_LEN];              /**< [en]Indicates vmr id
                                                               [cn]云会议室唯一ID。 */
    HWM_CHAR hostPwd[HWM_MAX_CONF_PASSWORD_LEN];          /**< [en]Indicates host pwd.
                                                               [cn]云会议室主持人密码。
                                                               [cs_sensitive:hide] */
    HWM_CHAR guestPwd[HWM_MAX_CONF_PASSWORD_LEN];         /**< [en]Indicates guest pwd.
                                                               [cn]云会议室来宾密码。
                                                               [cs_sensitive:hide] */
    HwmVmrIdType vmrType;                                 /**< [en]Indicates the type of the vmr
                                                               [cn]云会议室类型。 */
    HWM_INT32 maxParties;                                 /**< [en]Indicates the max party size of the vmr
                                                               [cn]云会议室最大方数。 */
    HWM_BOOL isEnable;                                    /**< [en]Indicates whether the vmr is enable
                                                               [cn]云会议室是否可用 */
    HWM_CHAR name[HWM_MAX_VMR_NAME_LEN];                  /**< [en]Indicates the name of the vmr
                                                               [cn]会议室名称。*/
    HWM_BOOL allowGuestStartConf;                         /**< [en]Indicates allow guest start conf.
                                                               [cn]允许来宾启动会议 */
    HwmConfAllowJoinUserType confAllowJoinUser;           /**< [en]Indicates the user type allowed to join the conference.
                                                               [cn]允许入会范围限制。默认所有用户。*/
    HWM_BOOL isOpenWaitingRoom;                           /**< [en]Indicates whether is open waiting room
                                                               [cn]等候室开启开关。
                                                                   说明：需要等候室功能开通后才能生效*/
} HwmVmrInfo;


/**
* [en]This structure is used to describe corp type
* [cn]企业类型
**/
typedef enum tagHwmCorpType
{
    HWM_CORP_TYPE_ENTERPRISE = 0,           /**< [en]Enterprise version, also known as flagship version.
                                                 [cn]企业版，即旗舰版 */
    HWM_CORP_TYPE_PUBLIC_REGISTER,          /**< [en]Public enterprises, phone or email user.
                                                 [cn]公共企业，手机、邮箱注册时会放到该企业内 */
    HWM_CORP_TYPE_PUBLIC_VISION,            /**< [en]Public enterprises, smart screens user.
                                                 [cn]公共企业，智慧屏用户自动开户时会放到该企业内 */
    HWM_CORP_TYPE_PUBLIC_WHITE_BOARD,       /**< [en]Public enterprises, board user.
                                                 [cn]公共企业，大屏用户自动开户时会放到该企业内 */
    HWM_CORP_TYPE_PUBLIC_CONSUMER,          /**< [en]Public TOC Consumers.
                                                 [cn]公共TOC消费者企业 */
    HWM_CORP_TYPE_FREE,                     /**< [en]Free version.
                                                 [cn]免费版 */
    HWM_CORP_TYPE_PROFESSIONAL              /**< [en]Professional version, also known as Flexus version.
                                                 [cn]专业版, 即Flexus版 */
}HwmCorpType;

/**
* [en]Vmr list info.
* [cn]云会议室列表信息
*/
typedef struct tagHwmVmrListInfo
{
    HWM_UINT32 numOfVmrs;                              /**< [en]Indicates the number of specific vmr conference.
                                                            [cn]云会议室个数
                                                            [length for vmrs] */
    HwmVmrInfo* vmrs;                                  /**< [en]Indicates the specific vmr conference information.
                                                            [cn]云会议室信息 */
    HWM_BOOL isEnableShareVmr;                         /**< [en]Indicates whether is share vmr.
                                                            [cn]是否开启共享vmr */
    HwmCorpType corpType;                              /**< [en]Indicates corp type.(only for windows)
                                                            [cn]企业类型(仅对win平台生效)
                                                            [doc:disable]
                                                            [mac:disable] */
}HwmVmrListInfo;

/**
* [en]This struct is used to describe conference list info.
* [cn]会议列表信息
*/
typedef struct tagHwmConfListInfo
{
    HWM_UINT32 confListSize;                            /**< [en]Indicates conference number.
                                                             [cn]会议列表参数信息的长度。
                                                             [length for confListItem] */
    HwmConfListItem* confListItem;                      /**< [en]Indicates conf list.
                                                             [cn]会议列表参数信息。 */
}HwmConfListInfo;

/**
* [en]Conf coming state.
* [cn]来电状态类型
*/
typedef enum tagHwmConfComingState
{
    HWM_CONF_COMING_STATE_START = 1,   /**< [en]Init state
                                            [cn]初始状态，来电振铃中。 */
    HWM_CONF_COMING_STATE_ACCEPT,      /**< [en]Accept state
                                            [cn]接听状态，已接听来电。 */
    HWM_CONF_COMING_STATE_REJECT,      /**< [en]Reject state
                                            [cn]拒绝状态，已拒绝来电。 */
    HWM_CONF_COMING_STATE_TIMEOUT,     /**< [en]Timeout state
                                            [cn]未响应状态，呼叫超时或者主叫已挂断。 */

    HWM_CONF_COMING_STATE_BUTT

} HwmConfComingState;

/**
* [en]Conf coming information.
* [cn]会议来电信息
*/
typedef struct tagHwmConfIncomingInfo
{
    HwmConfComingState state;                              /**< [en]Conf coming state
                                                                [cn]来电状态。 */
    HwmMediaType mediaType;                                /**< [en]Media type
                                                                [cn]会议媒体类型。 */
    HWM_BOOL isP2PConf;                                    /**< [en]is p2p conf
                                                                [cn]是否立即会议。 */
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];                  /**< [en]Conf id
                                                                [cn]会议id。 */
    HWM_CHAR vmrConfId[HWM_MAX_CONF_ID_LEN];               /**< [en]Vmr Conf id
                                                                [cn]Vmr会议id。 */
    HWM_CHAR thirdUserId[HWM_MAX_USER_ID_LEN];             /**< [en]The attendee's third account
                                                                [cn]第三方用户ID，App ID鉴权时使用。 */
    HWM_CHAR invitorName[HWM_MAX_DISPLAY_NAME_LEN];        /**< [en]Invitor name.
                                                                [cn]邀请者的姓名。会议自动外邀时是预订人，会中邀请时是操作人(仅在win平台下有效) */
    HWM_CHAR invitorAliasCN[HWM_MAX_DISPLAY_NAME_LEN];     /**< [en]Invitor alias Chinese name.
                                                                [cn]邀请者的中文别名
                                                                [doc:disable] */
    HWM_CHAR invitorAliasEN[HWM_MAX_DISPLAY_NAME_LEN];     /**< [en]Invitor alias English name.
                                                                [cn]邀请者的英文别名
                                                                [doc:disable] */
    HWM_CHAR invitorExtendedField[HWM_MAX_OPEN_CUSTOM_PARA_LEN]; /**< [en]Invitor Indicates schedule.
                                                                      [cn]邀请者的扩展字段
                                                                      [doc:disable] */
    HWM_CHAR invitorCorpId[HWM_MAX_CORP_ID_LEN];            /**<[en]Indicates Invitor corp id.
                                                                [cn]邀请者的企业id */
    HWM_CHAR schedulerName[HWM_MAX_DISPLAY_NAME_LEN];	   /**< [en]Schedule user name.
                                                                [cn]预订者姓名 */
    HWM_CHAR schedulerAliasCN[HWM_MAX_DISPLAY_NAME_LEN];   /**< [en]Indicates schedule alias Chinese name.
                                                                [cn]预订者中文别名
                                                                [doc:disable] */
    HWM_CHAR schedulerAliasEN[HWM_MAX_DISPLAY_NAME_LEN];   /**< [en]Indicates schedule alias English name.
                                                                [cn]预订者英文别名
                                                                [doc:disable] */
    HWM_CHAR schedulerExtendedField[HWM_MAX_OPEN_CUSTOM_PARA_LEN]; /**< [en]Indicates schedule extended field.
                                                                        [cn]预订者的扩展字段
                                                                        [doc:disable] */
} HwmConfIncomingInfo;

/**
* [en]Rename self info.
* [cn]设置会场名信息
*/
typedef struct tagHwmRenameSelfInfo
{
    HWM_CHAR nickName[HWM_MAX_USER_NAME_LEN];                   /**< [en]Indicates nickName
                                                                     [cn]会场名。
                                                                     [cs_allowNull:N] */
}HwmRenameSelfInfo;

/**
* [en] 
* [cn]预约周期会议参数
*/
typedef struct tagHwmBookCycleConfParam
{
    HwmBookConfParam bookConfParam;                /**< [en] 
                                                        [cn]预约会议参数
                                                        [cs_allowNull:Y]*/
    HwmCycleConfParam cycleParam;                  /**< [en]Indicates sub conference param. 
                                                        [cn]子会议参数
                                                        [cs_allowNull:N]*/
}HwmBookCycleConfParam;

/**
* [en]Modify cycle conf param
* [cn]编辑周期会议参数
*/
typedef struct tagHwmModifyCycleConfParam
{
    HwmEditConfParam editConfParam;             /**< [en] 
                                                     [cn]编辑会议参数
                                                     [cs_allowNull:Y] */
    HwmCycleConfParam cycleParam;               /**< [en]Indicates sub conference param. 
                                                     [cn]子会议参数
                                                     [cs_allowNull:N]*/
}HwmModifyCycleConfParam;

/**
* [en]Modify sub cycle conf param
* [cn]编辑周期子会议参数
*/
typedef struct tagHwmModifySubCycleConfParam
{
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];                 /**< [en]Indicates conference id. 
                                                               [cn]会议id
                                                               [cs_allowNull:N] */
    HWM_CHAR subConfID[HWM_MAX_CONF_UUID_LEN];            /**< [en]Indicates conference uuid. 
                                                               [cn]子会议唯一标识UUID
                                                               [cs_allowNull:N] */
    HwmMediaType mediaType;                               /**< [en]The media type
                                                               [cn]会议类型。
                                                               [cs_allowNull:N] */
    HWM_INT64 startTime;                                  /**< [en]Indicates conference start time, . 
                                                               [cn]（已废弃，推荐使用字段startTimeStamp）会议开始时间（UTC时间），单位为秒。说明：UTC时间与UTC偏移量（转换为秒）的差值，如东八区为 UTC时间戳 - (8 x 60 x 60)。
                                                               [cs_allowNull:Y] */
    HWM_INT64 startTimeStamp;                             /**< [en]Indicates conference start time
                                                               [cn]会议开始时间，startTimeStamp、startTime二选一，优先使用startTimeStamp(时间戳，单位秒)
                                                               [cs_allowNull:Y] */
    HWM_UINT32 duration;                                  /**< [en]Indicates conference length of time, unit is minute
                                                               [cn]会议持续时长，单位分钟
                                                               [cs_allowNull:N] */
    HWM_BOOL isAutoRecord;                                /**< [en]Is autorecord
                                                               [cn]是否开启自动录制会议。开启自动录制会议开关，内部会默认支持录制会议，录制会议开关的打开与关闭并不影响。
                                                               说明：仅对云录制生效，客户端本地录制不生效。
                                                               [cs_allowNull:Y] */
    HwmRecordAuthType recordAuthType;                     /**< [en]Indicates record auth type.
                                                               [cn]录播鉴权方式
                                                               [cs_allowNull:Y] */
    HwmConfAllowJoinUserType confAllowJoinUser;           /**< [en]Indicates the user type allowed to join the conference.
                                                               [cn]允许入会范围限制。默认所有用户。
                                                               [cs_allowNull:Y] */
    HWM_BOOL allowGuestStartConf;                         /**< [en]Indicates allow guest start conf. 
                                                               [cn]允许来宾启动会议
                                                               [cs_allowNull:Y] */
    HWM_UINT32 allowGuestStartConfTime;                   /**< [en]Indicates join before host time.
                                                               [cn]允许来宾提前入会时间范围(单位：分钟) 0-随时，n-提前n分钟启动会议
                                                               [cs_allowNull:Y] */
}HwmModifySubCycleConfParam;

/*
* [en]cancelCycleConfParam
* [cn]取消周期会议参数
*/
typedef struct tagHwmCancelCycleConfParam
{
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];                 /**< [en]Indicates conference id. 
                                                               [cn]会议id
                                                               [cs_allowNull:N]*/
    HWM_BOOL endOnlineConf;                               /**< [en]Indicates is end online conf. 
                                                               [cn]是否需要结束在线会议
                                                               [cs_allowNull:N]*/
}HwmCancelCycleConfParam;

/*
* [en]cancelSubCycleConfParam
* [cn]取消周期子会议参数
*/
typedef struct tagHwmCancelSubCycleConfParam
{
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];                 /**< [en]Indicates conference id. 
                                                               [cn]会议id
                                                               [cs_allowNull:N]*/
    HWM_CHAR subConfID[HWM_MAX_CONF_UUID_LEN];            /**< [en]Indicates conference uuid. 
                                                               [cn]子会议唯一标识UUID
                                                               [cs_allowNull:N]*/
    HWM_BOOL endOnlineConf;                               /**< [en]Indicates is end online conf. 
                                                               [cn]是否需要结束在线会议
                                                               [cs_allowNull:N]*/
}HwmCancelSubCycleConfParam;


/**
* [en]This struct is used to describe the status of low net work
* [cn]网络状态
*/
typedef enum tagHwmLowNetWorkStatus
{
    LOW_NET_WORK_STATUS_NORMAL,         /**< [en]Indicates staus normal
                                             [cn]网络正常*/
    LOW_NET_WORK_STATUS_BAD,            /**< [en]Indicates staus bad
                                             [cn]网络差*/
    LOW_NET_WORK_STATUS_VERY_BAD,       /**< [en]Indicates staus very bad
                                             [cn]网络非常差*/
}HwmLowNetWorkStatus;

/**
* [en]This struct is used to describe the status of low net work
* [cn]弱网状态信息
*/
typedef struct tagHwmLowNetWorkStatusInfo
{
    HWM_UINT32 packetLoss;          /**< [en]Indicates loss rate.
                                         [cn]丢包率*/
    HWM_UINT32 delay;               /**< [en]Indicates delay.
                                         [cn]时延*/
    HWM_UINT32 jitter;              /**< [en]Indicates jitter.
                                         [cn]抖动*/
    HwmLowNetWorkStatus status;     /**< [en]This struct is used to describe the status of low net work
                                         [cn]网络状态 */
}HwmLowNetWorkStatusInfo;

/**
* [en]This struct is used to describe the QoS weak network prompt information
* [cn]qos弱网提示信息
*/
typedef struct tagHwmPoorNetWorkQualityInfo
{
    HWM_CHAR nickName[HWM_MAX_USER_NAME_LEN];   /**< [en]Indicates nickName
                                                     [cn]与会者名称。*/
    HWM_CHAR number[HWM_MAX_NUMBER_LEN];        /**< [en]Indicates number
                                                     [cn]号码。如果填分配给账号的sip号码（如+99111244216210249）则呼叫该账号的App；
                                                         如果填PSTN号码（如18700000000），则通过VoIP网关呼叫该号码，前提是该企业已开通PSTN呼叫权限。
                                                         与thirdUserId二选一，账号密码鉴权时使用。*/
    HWM_CHAR thirdUserId[HWM_MAX_USER_ID_LEN];  /**< [en]The attendee's third account
                                                     [cn]第三方用户ID。与number二选一，App ID鉴权时使用。 */
    HWM_CHAR accountId[MAX_ACCOUNT_ID_LEN];     /**< [en]Indicates account id
                                                     [cn]华为云会议账号*/
    HWM_CHAR corpId[HWM_MAX_ORG_ID_LENGTH];     /**< [en]Indicates corp id.
                                                     [cn]企业id */
    HWM_BOOL isLocal;                           /**< [en]Indicates is local.
                                                     [cn]是否本端*/
    HwmLowNetWorkStatusInfo audioStatus;        /**< [en]Indicates is network status.
                                                     [cn]音频网络状态*/
    HwmLowNetWorkStatusInfo videoStatus;        /**< [en]Indicates is network status.
                                                     [cn]视频网络状态*/
    HwmLowNetWorkStatusInfo shareStatus;        /**< [en]Indicates is network status.
                                                     [cn]共享网络状态*/
}HwmPoorNetWorkQualityInfo;

/**
* [en]watch item param
* [cn]选看入参
*/
typedef enum tagHwmWndType
{
    HWM_WND_TYPE_NORMAL,             /**< [en]Normal window
                                          [cn]普通窗口 */
    HWM_WND_TYPE_MAIN,               /**< [en]Main window
                                          [cn]主要窗口 */
}HwmWndType;

/**
* [en] rect info
* [cn]界面信息
*/
typedef struct tagHwmRectInfo
{
    HWM_INT32 left;                  /**< [en]left
                                          [cn]左边距 */
    HWM_INT32 top;                   /**< [en]top
                                          [cn]顶部边距 */
    HWM_INT32 right;                 /**< [en]right
                                          [cn]右边距*/
    HWM_INT32 bottom;                /**< [en]bottom
                                          [cn]底部边距 */
} HwmRectInfo;

/**
* [en]watch item param
* [cn]选看入参
*/
typedef struct tagHwmWatchItem
{
    HWM_INT32 userId;                 /**< [en]Indicates userId
                                           [cn]与会者唯一标识 */
    HWM_HWND renderHandle;	          /**< [en]The render window handle
                                           [cn]视频窗口附着的父窗口句柄 */
    HwmWndType wndType;               /**< [en]Indicates type of window
                                           [cn]窗口类型 */
    HwmRectInfo padding;              /**< [en]Video rect info
                                           [cn]视频显示区域边距 */
}HwmWatchItem;

/**
* [en]Watch list info.
* [cn]选看列表信息
*/
typedef struct tagHwmWatchListInfo
{
    HWM_UINT32 watchListSize;                          /**< [en]length for watchList.
                                                            [cn]选看个数
                                                            [cs_allowNull:N][length for watchList] */
    HwmWatchItem* watchList;                           /**< [en]watch info
                                                            [cn]选看信息
                                                            [cs_allowNull:N] */
}HwmWatchListInfo;

/**
* [en]Annotation wnd info
* [cn]标注窗口信息
*/
typedef struct tagHwmAnnotationWndInfo
{
    HWM_HWND renderHandle;           /**< [en]The render window handle
                                           [cn]标注窗口附着的父窗口句柄 */
} HwmAnnotationWndInfo;

/**
* [en]Annotation save confg info
* [cn]标注保存配置信息
*/
typedef struct tagHwmAnnotationSaveConfigInfo
{
    HWM_CHAR path[HWM_MAX_FILE_PATH_LEN];       /**< [en]save path
                                                     [cn]保存路径，注：绝对路径，需要UTF8编码
                                                     [cs_allowNull:N] */
} HwmAnnotationSaveConfigInfo;

/**
* [en]Speaker info.
* [cn]发言人信息
*/
typedef struct tagHwmConfSpeaker
{
    HWM_INT32 userId;                                        /**< [en]Indicates userId
                                                                  [cn]与会者唯一标识 */
    HWM_CHAR name[HWM_MAX_DISPLAY_NAME_LEN];                 /**< [en]Display name.
                                                                  [cn]与会者显示名称 */
} HwmConfSpeaker;

/**
* [en]Speakers info.
* [cn]发言人列表信息
*/
typedef struct tagHwmSpeakerList
{
    HWM_UINT32 speakerSize;                                        /**< [en]Speaker number.
                                                                        [cn]发言人个数
                                                                        [length for speakers] */
    HwmConfSpeaker* speakers;                                      /**< [en]Speakers.
                                                                        [cn]发言人列表 */
} HwmSpeakerList;

/**
* [en]Stream type
* [cn]流类型
*/
typedef enum tagHwmStreamType
{
    STREAM_TYPE_NONE = 0,         /**< [en]No stream
                                       [cn]没有流
                                       [ios:rename:none]*/
    STREAM_TYPE_PICTURE,          /**< [en]Picture stream
                                       [cn]图片流
                                       [ios:rename:picture]*/
    STREAM_TYPE_VIDEO,            /**< [en]Video stream
                                       [cn]视频流
                                       [ios:rename:video]*/
} HwmStreamType;

/**
* [en]Stream type info
* [cn]视频流类型信息
*/
typedef struct tagHwmVideoStreamTypeInfo
{
    HWM_INT32 userId;                                       /**< [en]Indicates userId
                                                                 [cn]与会者唯一标识 */
    HwmStreamType streamType;                               /**< [en]Stream type.
                                                                 [cn]流类型 */
} HwmVideoStreamTypeInfo;

/**
* [en]Annotation state info
* [cn]标注状态信息
*/
typedef struct tagHwmAnnotationStateInfo
{
    HwmState annotationState;                               /**< [en]Annotation state
                                                                 [cn]标注状态 */
} HwmAnnotationStateInfo;

/**
* [en]This struct is used to describe basic call member info
* [cn]呼叫基本成员信息
*/
typedef struct tagHwmCallMemberInfo
{
    HWM_CHAR number[HWM_MAX_NUMBER_LEN];                 /**< [en]Indicates number. 
                                                              [cn]号码 */
    HWM_CHAR nickName[HWM_MAX_NAME_LEN];                 /**< [en]Indicates name. 
                                                              [cn]成员名称 */
    HWM_CHAR userUuid[HWM_MAX_ACCOUNT_ID_LEN];           /**< [en]Indicates the uuid. 
                                                              [cn]与会者uuid */
    HWM_CHAR thirdUserId[HWM_MAX_ACCOUNT_ID_LEN];        /**< [en]Indicates the third part account.
                                                              [cn]第三方账号 */
    HWM_CHAR extendedField[HWM_MAX_OPEN_CUSTOM_PARA_LEN]; /**< [en]Indicates extended field.
                                                               [cn]扩展字段 
                                                               [doc:disable] */
    HWM_CHAR corpId[HWM_MAX_CORP_ID_LEN];                 /**< [en]Indicates corp id.
                                                               [cn]企业id */
}HwmCallMemberInfo;

/**
 * [en]This struct is used to describe P2P conference record info
 * [cn]立即会议记录
 */
typedef struct tagHwmP2PConfRecordInfo
{
    HwmCallMemberInfo callerInfo;                        /**< [en]Indicates caller infomation. 
                                                              [cn]主叫方信息 */
    HwmCallMemberInfo calleeInfo;                        /**< [en]Indicates callee infomation. 
                                                              [cn]被叫方信息 */
    HWM_INT32 startTime;                                 /**< [en]Indicates calling start time. 
                                                              [cn]（已废弃，推荐使用字段startTimeStamp）呼叫开始时间，精度秒(0时区) */
    HWM_INT32 endTime;                                   /**< [en]Indicates calling end time. 
                                                              [cn]（已废弃，推荐使用字段endTimeStamp）呼叫结束时间，精度秒(0时区) */
    HWM_INT64 startTimeStamp;                            /**< [en]Indicates conference start time
                                                              [cn]会议开始时间(时间戳，单位秒) */
    HWM_INT64 endTimeStamp;                              /**< [en]Indicates conference end time
                                                              [cn]会议结束时间(时间戳，单位秒) */
    HWM_BOOL isCallOut;                                  /**< [en]Is initiative call out. 
                                                              [cn]是否为主动呼出 */
    HwmMediaType mediaType;                              /**< [en]Indicates call type.
                                                              [cn]媒体类型 */
    SDKERR resultCode;                                   /**< [en]Error code.
                                                              [cn]错误码 */
    HWM_CHAR resultMsg[HWM_MAX_REASON_LEN];              /**< [en]Error describe.
                                                              [cn]错误描述 */
    HWM_BOOL isCallEstablished;                          /**< [en]Indicates whether the call is established.
                                                              [cn]呼叫是否建立 */
}HwmP2PConfRecordInfo;

/**
* [en]This structure is used to describe start projection param.
* [cn]发起投屏参数体
*/
typedef struct tagHwmStartProjectionParam
{
    HWM_CHAR pairCode[HWM_MAX_PAIR_CODE_LEN];       /**< [en]Indicates pair code.
                                                    [cn]配对码
                                                    [cs_allowNull:N] */
    HWM_CHAR randomNumber[HWM_MAX_RANDOM_LEN];      /**< [en]Indicates random number.
                                                    [cn]随机数，跨企业投屏场景需要携带。
                                                    [cs_allowNull:Y] */
    HWM_CHAR verifyCode[HWM_MAX_VERIFY_CODE_LEN];   /**< [en]SMS verification code.
                                                    [cn]验证码，跨企业投屏场景需要携带。
                                                    [cs_allowNull:Y] */
}HwmStartProjectionParam;

/**
* [en]This structure is used to describe auto hide tool bar param.
* [cn]自动隐藏会控栏参数体
*/
typedef struct tagHwmAutoHideToolBarParam
{
    HWM_BOOL isAutoHide;                       /**< [en]Is auto hide
                                                    [cn]是否自动隐藏
                                                    [cs_allowNull:N] */
}HwmAutoHideToolBarParam;

/**
* [en]This structure is used to describe start projection result.
* [cn]发起投屏结果
*/
typedef struct tagHwmStartProjectionResult
{
    HWM_CHAR randomNumber[HWM_MAX_RANDOM_LEN];                 /**< [en]Indicates random number.
                                                               [cn]随机数*/
}HwmStartProjectionResult;


/**
* [en]This enum is used to describe share state
* [cn] 数据共享状态
*/
typedef enum tagHwmShareState
{
    CONF_SHARE_STATE_CLOSE = 0,   /**< [en]Indicates no one is sharingh
                                       [cn]非数据共享状态 */
    CONF_SHARE_STATE_OPEN = 1     /**< [en]Indicates some one is sharing
                                       [cn]数据共享状态 */
}HwmShareState;

/**
* [en]This enum is used to describe conference recording mode.
* [cn]会议录制模式
*/
typedef enum tagHwmConfRecordMode
{
    CONF_RECORD_DISABLE,                /**< [en]Indicates disable record
                                             [cn]禁止录制 */
    CONF_RECORD_LIVE,                   /**< [en]Indicates live
                                             [cn]直播模式 */
    CONF_RECORD_RECORD,                 /**< [en]Indicates record broadcast
                                             [cn]录播模式 */
    CONF_RECORD_LIVE_AND_RECORD,        /**< [en]Indicates live broadcast and record broadcast
                                             [cn]直播 + 录播模式 */
}HwmConfRecordMode;

/**
* [en]This enum is used to describe conference IM mode.
* [cn]会中聊天模式
*/
typedef enum tagHwmConfIMType
{
    CONF_IM_TYPE_RTM = 0,       /**< [en]Indicates use welink-rtm.
                                     [cn]使用welink-rtm实现会中聊天 */
    CONF_IM_TYPE_CIM            /**< [en]Indicates use conf im.
                                     [cn]使用会控自带IM实现会中聊天 */
} HwmConfIMType;

/**
* [en]This enum is used to describe create conference model
* [cn] 创建会议model
*/
typedef enum tagHwmConfCreateMode
{
    CONF_CREATE_MODE_DEFAULT = 0,           /**< [en]Indicates default
                                                 [cn]默认 */
    CONF_CREATE_MODE_P2PTransToConf = 1,    /**< [en]Indicates p2p transfer to conf
                                                 [cn]点对点转会议 */
    CONF_CREATE_MODE_MmrP2PTransToConf = 2, /**< [en]Indicates mmr p2p transfer to conf
                                                 [cn]mmr点对点转会议 */
    CONF_CREATE_MODE_MMR_P2P = 3,           /**< [en]Indicates mmr p2p.
                                                 [cn]mmr点对点*/
    CONF_CREATE_MODE_LOCAL_PROJECT = 4      /**< [en]Indicates local project.
                                                 [cn]本地投屏*/
}HwmConfCreateMode;

/**
* [en]This struct is used to describe conf extend config.
* [cn]会议扩展配置
*/
typedef struct tagHwmConfExtendConfig
{
    HWM_UINT32 maxBitrateValue;                                     /**< [en]Indicates max bitrate. 
                                                                         [cn]探高上限值 */
    HWM_BOOL enableDetectMaxBWE;                                    /**< [en]Indicates is enable detect max bitrate. 
                                                                         [cn]是否开启探高 */
}HwmConfExtendConfig;

/**
* [en]This struct is used to describe conference base info.
* [cn]会议基础信息
*/
typedef struct tagHwmConfBaseInfo
{
    HWM_CHAR confId[HWM_MAX_CONF_ID_LEN];                           /**< [en]Indicates conference id.
                                                                         [cn]会议id,加入会议时用到。如果是VMR会议，则此ID是真实会议ID，不用来显示；如果不是VMR会议，则用来显示。*/
    HWM_CHAR vmrConfId[HWM_MAX_CONF_ID_LEN];                        /**< [en]Indicates virtual conference id.
                                                                         [cn]VMR 会议id，VMR会议下有效 */
    HWM_CHAR confSubject[HWM_MAX_SUBJECT_LEN];                      /**< [cs_sensitive:hide][en]Indicates conference subject.
                                                                         [cn]会议主题 */
    HWM_INT64 startTime;                                            /**< [en]Indicates the conference start time.
                                                                         [cn]（已废弃，推荐使用字段startTimeStamp）会议开始时间-时间戳，精度秒(0时区) */
    HWM_INT64 endTime;                                              /**< [en]Indicates the conference end time.
                                                                         [cn]（已废弃，推荐使用字段endTimeStamp）会议结束时间-时间戳，精度秒(0时区) */
    HWM_INT64 startTimeStamp;                                       /**< [en]Indicates conference start time
                                                                         [cn]会议开始时间(时间戳，单位秒) */
    HWM_INT64 endTimeStamp;                                         /**< [en]Indicates conference end time
                                                                         [cn]会议结束时间(时间戳，单位秒) */
    HWM_INT64 serverCurrentTime;                                    /**< [en]Indicates the conference end time.
                                                                         [cn]当前服务器时间
                                                                         [doc:disable] */
    HWM_CHAR  timeZone[HWM_MAX_TIME_ZONE_LEN];                      /**< [en]Indicates time zone, optional, standard time zone for example "Asia/Shanghai" in PBX internal conference; Beijing is "56" in Mediax conference,refer to CONFCTRL_E_TIMEZONE.
                                                                         [cn]时区 */
    HWM_CHAR hostPwd[HWM_MAX_CONF_PASSWORD_LEN];                    /**< [en]Indicates chairman password.
                                                                         [cn]会议主持人密码 */
    HWM_CHAR guestPwd[HWM_MAX_CONF_PASSWORD_LEN];                   /**< [en]Indicates guest password.
                                                                         [cn]会议来宾密码 */
    HwmMediaType mediaType;                                         /**< [en]Indicates the conferencn media type.
                                                                         [cn]会议媒体类型
                                                                         [doc:disable] */
    HWM_CHAR scheduserUuid[HWM_MAX_ACCOUNT_ID_LEN];                 /**< [en]Indicates schedule uuid.
                                                                         [cn]预订者uuid */
    HWM_CHAR scheduserName[HWM_MAX_DISPLAY_NAME_LEN];               /**< [en]Indicates schedule user name.
                                                                         [cn]预订者姓名 */
    HWM_CHAR hostJoinUri[HWM_MAX_URL_LEN];                          /**< [en]Indicates the uri for chairman to join conference .
                                                                         [cn]主持人入会链接 */
    HWM_CHAR guestJoinUri[HWM_MAX_URL_LEN];                         /**< [en]the uri for guest to join conference.
                                                                         [cn]来宾加入会议的链接 */
    HWM_CHAR accessNumber[HWM_MAX_CONF_ACCESS_LEN];                 /**< [en]Indicates conference access number.
                                                                         [cn]会议接入码
                                                                         [doc:disable] */
    HwmConfRole selfConfRole;                                       /**< [en]Indicates the role of participant.
                                                                         [cn]本人在会议中的角色。主持人、普通与会者等。 */
    HWM_BOOL isWebinar;                                             /**< [en]Indicates whether the conference is web conferencre.
                                                                         [cn]是否是网络研讨会 */
    HWM_BOOL isCreator;                                             /**< [en]Indicates whether the user is the creator of the conference
                                                                         [cn]是否是会议预订者 */
    HWM_CHAR audienceJoinUri[HWM_MAX_URL_LEN];                      /**< [en]Indicates the uri for audience to join conference.
                                                                         [cn]观众加入会议的链接 */
    HWM_CHAR audiencePwd[HWM_MAX_CONF_PASSWORD_LEN];                /**< [en]Indicates the password of audience.
                                                                         [cn]观众密码 */
    HwmConfState confStateType;                                     /**< [en]Indicates conference state.
                                                                         [cn]会议状态
                                                                         [doc:disable] */
    HwmConferenceType conferenceType;                               /**< [en]Indicates conference type.
                                                                         [cn]会议类型 */
    HwmCycleConfParam cycleParam;                                   /**< [en]Indicates cycle conference param.
                                                                         [cn]周期会议的参数
                                                                         [doc:disable] */
    HWM_UINT32 subConfSize;                                         /**< [en]Indicates cycle sub conference size.
                                                                         [cn]周期子会议数量
                                                                         [length for subConfParam][doc:disable] */
    HwmSubCycleConfParam *subConfParam;                             /**< [en]Indicates cycle sub conference param.
                                                                         [cn]周期子会议参数
                                                                         [doc:disable] */
    HWM_CHAR confUuid[HWM_MAX_ACCOUNT_ID_LEN];                      /**< [en]Indicates conference uuid.
                                                                         [cn]会议uuid */
    HWM_BOOL isExperienceConf;                                      /**< [en]Indicates is experience conf.
                                                                         [cn]是否为周期体验会议
                                                                         [doc:disable] */
    HWM_CHAR appAttribute[HWM_MAX_APP_ATTRIBUTE_LEN];               /**< [en]Indicates .
                                                                         [cn]应用属性，UI根据不同应用拼接
                                                                         [doc:disable] */
    HWM_CHAR openCustomPara[HWM_MAX_OPEN_CUSTOM_PARA_LEN];          /**< [en]Indicates User-defined data on the device side attribute.
                                                                         [cn]自定义扩展信息 */
    HWM_CHAR liveRoomAddress[HWM_MAX_URL_LEN];                      /**< [en]Indicates live room address
                                                                         [cn]直播地址 */
    HWM_CHAR breakoutID[HWM_MAX_CONF_ID_LEN];                       /**< [en]Indicates breakoutID.
                                                                         [cn]讨论分组ID，启动分组后，动态分组成员信息才有
                                                                         [doc:disable] */
    HWM_UINT32 breakoutConfLimit;                                   /**< [en]Indicates breakoutConfLimit.
                                                                         [cn]分组会议子会场数目上限
                                                                         [doc:disable] */
    HWM_CHAR recordLiveType[HWM_MAX_RECORD_LIVE_TYPE_LEN];          /**< [en]Indicates record live type.
                                                                         [cn]录制类型（mmr/rtc/rse）
                                                                         [doc:disable] */
    HWM_BOOL isHighResolution;                                      /**< [en]high resolution
                                                                         [cn]是否为超高清(1080p)会议 */
    HWM_BOOL isP2PConf;                                             /**< [en]Indicates p2p conf.
                                                                         [cn]是否立即会议 */
    HWM_BOOL isUseConfctrlTempToken;                                /**< [en]Indicates is use confctlr temporary token.
                                                                         [cn]是否使用会控临时token
                                                                         [doc:disable] */
    HWM_UINT32 concurrentParticipants;                              /**< [en]concurrent participants.
                                                                         [cn]会议方数，会议最大与会人数限制. 0: 无限制, 大于0: 会议最大与会人数
                                                                         [doc:disable] */
    HWM_BOOL isSupportE2E;                                          /**< [en]Indicates is support E2E encrypt.
                                                                         [cn]是否支持端到端加密
                                                                         [doc:disable] */
    HWM_BOOL isChipEncryptCall;                                     /**< [en]Indicates is chip encrypt call.
                                                                         [cn]是否保密通话
                                                                         [doc:disable] */
    HwmConfExtendConfig confExtendConfig;                           /**< [en]conf extend config.
                                                                         [cn]会议扩展配置
                                                                         [doc:disable] */
    HwmConfCreateMode immediatelyConfStartMode;                     /**< [en]Indicates immediatelyConfStartMode.
                                                                         [cn]即时创会类型
                                                                         [doc:disable] */
    HwmShareState shareState;                                       /**< [en]Indicates shareState.
                                                                         [cn]共享状态
                                                                         [doc:disable] */
    HWM_BOOL enableCollaborate;                                     /**< [en]Indicates enableCollaborate.
                                                                         [cn]是否使用辅流
                                                                         [doc:disable] */
    HWM_BOOL supportSummary;                                        /**< [en]Indicates supportSummary.
                                                                         [cn]是否支持会议纪要 */
    HwmSummaryState defaultSummaryState;                            /**< [en]Indicates defaultSummaryState.
                                                                         [cn]会议纪要开启状态 */
    HWM_BOOL supportMultiShare;                                     /**< [en]Indicates support multi-share.
                                                                         [cn]是否支持多人共享
                                                                         [cs_allowNull:Y] [doc:disable] */
    HWM_BOOL corpEnableWatermark;                                   /**< [en]Indicates is corp support water mark.
                                                                         [cn]企业是否支持会议水印 */
    HWM_BOOL corpEnableForbidScreenShots;                           /**< [en]Indicates is corp support forbidden screen shot.
                                                                         [cn]企业是否支持防截屏 */
    HWM_BOOL autoPublishSummary;                                    /**< [en]Indicates Whether auto publish summary. 
                                                                         [cn]是否自动发布纪要（免人工审核）*/
}HwmConfBaseInfo;

/**
* [en]This structure is used to describe conf info.
* [cn]会中信息
*/
typedef struct tagHwmMeetingInfo
{
    HwmConfBaseInfo baseInfo;                                  /**< [en]Indicates base information of conference.
                                                                    [cn]会议基本信息 */
    HWM_CHAR imGroupId[HWM_MAX_GROUP_URI_LEN];                 /**< [en]Indicates im group id.
                                                                    [cn]im群组id
                                                                    [doc:disable] */
    HWM_CHAR orgId[HWM_MAX_ORG_ID_LENGTH];                     /**< [en]Indicates org id.
                                                                    [cn]企业id */
    HwmConfRecordMode recordMode;                              /**< [en]Indicates conference recording mode.
                                                                    [cn]会议媒体录制模式 */
    HWM_BOOL isSupport90P;                                     /**< [en]Indicates is support 90P.
                                                                    [cn]是否支持90P
                                                                    [doc:disable] */
    HWM_BOOL supportBreakoutConf;                              /**< [en]Indicates is support breakout conf.
                                                                    [cn]是否支持分组会议 */
    HWM_BOOL isBreakoutSubConf;                                /**< [en]Indicates is breakout sub conf.
                                                                    [cn]是否是分组会议 */
    HWM_BOOL isCtd;                                            /**< [en]Indicates ctd.
                                                                    [cn]是否是ctd
                                                                    [doc:disable] */
    HwmConfIMType imType;                                      /**< [en]Indicates im type.
                                                                    [cn]会中聊天类型，默认RTM
                                                                    [doc:disable] */
    HWM_CHAR cimGroupId[HWM_MAX_CIM_GROUP_ID_LEN];             /**< [en]Indicates conf im group id.
                                                                    [cn]CIM群组id
                                                                    [doc:disable] */
    HWM_BOOL isSupportCloudVirtualBackground;                  /**< [en]Indicates support cloud virtual background.
                                                                    [cn]是否支持云虚拟背景
                                                                    [doc:disable] */
    HWM_BOOL supportGuestInviteAttendee;                       /**< [en]Indicates support guest invite attendee.
                                                                    [cn]是否支持来宾外邀 */
}HwmMeetingInfo;

/**
* [en]This structure is used to describe add attendee result.
* [cn]邀请与会者结果信息
*/
typedef struct tagHwmAddAttendeeResult
{
    HWM_CHAR number[HWM_MAX_NUMBER_LEN];                 /**< [en]Indicates number.
                                                              [cn]号码 */
    HWM_CHAR thirdUserId[HWM_MAX_USER_ID_LEN];           /**< [en]Indicates the third part account.
                                                              [cn]第三方账号 */
    SDKERR resultCode;                                   /**< [en]Error code.
                                                              [cn]错误码 */
    HWM_CHAR extendedField[HWM_MAX_OPEN_CUSTOM_PARA_LEN]; /**< [en]Indicates extended field.
                                                               [cn]扩展字段
                                                               [doc:disable] */
    HWM_CHAR corpId[HWM_MAX_ORG_ID_LENGTH];               /**< [en]Indicates corp id.
                                                               [cn]企业id */
} HwmAddAttendeeResult;

/**
* [en]This structure is used to describe add attendee result list.
* [cn]邀请与会者结果列表
*/
typedef struct tagHwmAddAttendeeResultList
{
    HWM_UINT32 addAttendeeResultSize;                    /**< [en]add attendee result size.
                                                              [cn]添加与会者结果长度
                                                              [length for addAttendeeResults] */
    HwmAddAttendeeResult* addAttendeeResults;            /**< [en]add attendee result list.
                                                              [cn]添加与会者结果列表 */
} HwmAddAttendeeResultList;

/**
* [en]accept conf param
* [cn]接受会议邀请参数
*/
typedef struct tagHwmAcceptConfParam
{
    HWM_BOOL isAudioAccept;             /**< [en]is audio accept
                                             [cn]是否语音接听（仅立即视频会议时有效）
                                             [cs_allowNull:Y]*/
} HwmAcceptConfParam;

/**
* [en]This enum is used to describe the status of local record.
* [cn]本地录制状态
*/
typedef enum tagHwmLocalRecordStatus
{
    LOCAL_RECORD_STATUS_START = 0,      /**< [en]Indicates start local record
                                             [cn]开始本地录制 */
    LOCAL_RECORD_STATUS_STOPPED = 1,    /**< [en]Indicates stopped local record
                                             [cn]停止本地录制 */
    LOCAL_RECORD_STATUS_PAUSED = 2,     /**< [en]Indicates paused local record
                                             [cn]暂停本地录制 */
    LOCAL_RECORD_STATUS_RESUMED = 3     /**< [en]Indicates resumed local record
                                             [cn]恢复本地录制 */
} HwmLocalRecordStatus;

/**
* [en]This struct is used to describe local record state info.
* [cn]本地录制状态信息
*/
typedef struct tagHwmLocalRecordStateInfo
{
    HwmLocalRecordStatus status;                        /**< [en]Local record status
                                                             [cn]本地录制状态 */
    HWM_CHAR localRecordPath[HWM_MAX_FILE_PATH_LEN];    /**< [en]Local record file path
                                                             [cn]本地录制文件路径 */
    HWM_INT64 startTime;                                /**< [en]Start time
                                                             [cn]录制开始时间(时间戳，单位秒) */
} HwmLocalRecordStateInfo;

/**
* [en]Self share state info
* [cn]自己的开启共享状态信息
*/
typedef struct tagHwmSelfShareStateInfo
{
    HwmState state;                             /**< [en]The share state
                                                     [cn]开启共享状态。 */
}HwmSelfShareStateInfo;

/**
* [en]This enum is used to describe the operate of local record.
* [cn]本地录制操作类型
*/
typedef enum tagHwmLocalRecordOperateType
{
    LOCAL_RECORD_OPERATE_TYPE_START = 0,      /**< [en]Indicates start local record
                                                   [cn]开始本地录制 */
    LOCAL_RECORD_OPERATE_TYPE_STOP = 1,       /**< [en]Indicates stopped local record
                                                   [cn]停止本地录制 */
    LOCAL_RECORD_OPERATE_TYPE_PAUSE = 2,      /**< [en]Indicates paused local record
                                                   [cn]暂停本地录制 */
    LOCAL_RECORD_OPERATE_TYPE_RESUME = 3      /**< [en]Indicates resumed local record
                                                   [cn]恢复本地录制 */
} HwmLocalRecordOperateType;

/**
* [en]Local record status info
* [cn]本地录制状态信息
*/
typedef struct tagHwmLocalRecordStatusInfo
{
    HwmLocalRecordStatus state;                 /**< [en]The local record status
                                                     [cn]本地录制状态。 */
}HwmLocalRecordStatusInfo;

/**
* [en]Local record permission info
* [cn]本地录制权限信息
*/
typedef struct tagHwmLocalRecordPermissionInfo
{
    HWM_BOOL hasPermission;                    /**< [en]The local record permission
                                                    [cn]是否有本地录制权限。 */
}HwmLocalRecordPermissionInfo;

/**
* [en]This enumeration is used to describe the subtitle translation language type.
* [cn]字幕翻译语言类型
*/
typedef enum tagHwmTargetLanguageType
{
    SUBTITLE_LANGUAGE_DEFAULT = 0,  /**<  [en]Indicates is default     lan.
                                          [cn]默认不翻译 */
    SUBTITLE_LANGUAGE_CN,           /**<  [en]Indicates is Chinese     lan.
                                          [cn]中文 */
    SUBTITLE_LANGUAGE_EN,           /**<  [en]Indicates is English     lan.
                                          [cn]英文 */
    SUBTITLE_LANGUAGE_GE,           /**<  [en]Indicates is Germany     lan.
                                          [cn]德文 */
    SUBTITLE_LANGUAGE_FR,           /**<  [en]Indicates is French      lan.
                                          [cn]法语 */
    SUBTITLE_LANGUAGE_RU,           /**<  [en]Indicates is Russian     lan.
                                          [cn]俄语 */
    SUBTITLE_LANGUAGE_JA,           /**<  [en]Indicates is Japanese    lan.
                                          [cn]日语 */
    SUBTITLE_LANGUAGE_KO,           /**<  [en]Indicates is Korean      lan.
                                          [cn]韩语 */
    SUBTITLE_LANGUAGE_ES,           /**<  [en]Indicates is Spanish     lan.
                                          [cn]西班牙语 */
    SUBTITLE_LANGUAGE_PT,           /**<  [en]Indicates is Portuguese  lan.
                                          [cn]葡萄牙语 */
    SUBTITLE_LANGUAGE_AR,           /**<  [en]Indicates is Arabic      lan.
                                          [cn]阿拉伯语 */
    SUBTITLE_LANGUAGE_IT,           /**<  [en]Indicates is Italian     lan.
                                          [cn]意大利语 */
    SUBTITLE_LANGUAGE_TH,           /**<  [en]Indicates is Thai        lan.
                                          [cn]泰语*/
    SUBTITLE_LANGUAGE_ID,           /**<  [en]Indicates is Indonesian  lan.
                                          [cn]印度尼西亚语*/
    SUBTITLE_LANGUAGE_MS,           /**<  [en]Indicates is Malay       lan.
                                          [cn]马来语*/
    SUBTITLE_LANGUAGE_TR            /**<  [en]Indicates is Turkish     lan.
                                          [cn]土耳其语*/
} HwmTargetLanguageType;

/**
* [en]This struct is used to describe the subtitle translation language type info.
* [cn]字幕翻译语言类型信息
*/
typedef struct tagHwmTargetLanguageTypeInfo
{
    HwmTargetLanguageType targetLanguageType;     /**< [en]target language type.
                                                       [cn]字幕翻译语言类型 */
} HwmTargetLanguageTypeInfo;

/**
* [en]This struct is used to describe the subtitle translation language type list.
* [cn]字幕翻译语言类型列表
*/
typedef struct tagHwmTargetLanguageTypeList
{
    HWM_UINT32 targetLanguageTypeSize;                   /**< [en]target language type size.
                                                              [cn]字幕翻译语言类型长度
                                                              [cs_allowNull:N][length for targetLanguageTypes] */
    HwmTargetLanguageTypeInfo* targetLanguageTypes;      /**< [en]target language types.
                                                              [cn]字幕翻译语言类型列表 
                                                              [cs_allowNull:N] */
} HwmTargetLanguageTypeList;

/**
* [en]This struct is used to describe the real time subtitle info.
* [cn]实时字幕信息
*/
typedef struct tagHwmRealTimeSubtitleInfo
{
    HWM_UINT32 userId;                                  /**< [en]Indicates user id.
                                                             [cn]内部用户id */
    HWM_CHAR   originalMsg[HWM_MAX_SUBTITLE_LEN];       /**< [en]Indicates Subtitle.
                                                             [cn]字幕原文内容 */
    HWM_CHAR   translatedMsg[HWM_MAX_SUBTITLE_LEN];     /**< [en]Indicates Subtitle.
                                                             [cn]字幕翻译内容 */
    HWM_BOOL   isFinal;                                 /**< [en]Indicates is final.
                                                             [cn]是否完整语句 */
    HWM_CHAR   name[HWM_MAX_NAME_LEN];                  /**< [en]Indicates user name.
                                                             [cn]用户名称 */
    HWM_CHAR   aliasCN[HWM_MAX_NAME_LEN];               /**< [en]Indicates user aliasCN.
                                                             [cn]用户中文别名 */
    HWM_CHAR   aliasEN[HWM_MAX_NAME_LEN];               /**< [en]Indicates user aliasEN.
                                                             [cn]用户英文别名 */
    HWM_BOOL   isExistVoicePrint;                       /**< [en]Indicates is exist voice print.
                                                             [cn]是否携带声纹识别信息 */
    HWM_BOOL   isRecognize;                             /**< [en]Indicates isRecognize.
                                                             [cn]是否匹配到声纹 */
    HWM_CHAR   speakerCN[HWM_MAX_NAME_LEN];             /**< [en]Indicates speakerCN.
                                                             [cn]发言人中文别名 */
    HWM_CHAR   speakerEN[HWM_MAX_NAME_LEN];             /**< [en]Indicates speakerCN.
                                                             [cn]发言人英文别名 */
    HWM_CHAR   userCN[HWM_MAX_NAME_LEN];                /**< [en]Indicates userCN.
                                                             [cn]声纹所属中文名 */
    HWM_CHAR   userEN[HWM_MAX_NAME_LEN];                /**< [en]Indicates userEN.
                                                             [cn]声纹所属英文名 */
    HWM_CHAR   tag[HWM_MAX_NAME_LEN];                   /**< [en]Indicates tag.
                                                             [cn]用户标识 */
    HwmTargetLanguageType translateLanguage;            /**< [en]Indicates translate language.
                                                             [cn]字幕翻译语言类型 */
} HwmRealTimeSubtitleInfo;

/**
* [en]This struct is used to describe subtitle record info.
* [cn]字幕记录信息
*/
typedef struct tagHwmSubtitleRecordInfo
{
    HwmRealTimeSubtitleInfo realTimeSubtitleInfo;     /**< [en]Indicates real time subtitle.
                                                           [cn]实时字幕信息 */
    HWM_CHAR   msgId[HWM_MAX_MSG_ID_LEN];             /**< [en]Indicates subtitle message id.
                                                           [cn]消息id,可以用来匹配字幕和翻译 */
    HWM_INT64   timeStamp;                            /**< [en]Indicates subtitle time stamp.
                                                           [cn]字幕时间戳,UI可以用来显示字幕时间 */
} HwmSubtitleRecordInfo;

/**
* [en]This enumeration is used to describe the interrupt type.
* [cn]中断类型
*/
typedef enum tagHwmInterruptType
{
    SELECT_SAVE_PATH = 0,                     /**< [en]Indicates select save path
                                                   [cn]选择录制文件存放地址 */
    RISK_TIP = 1,                             /**< [en]Indicates risk tip confirm
                                                   [cn]开启安全录制提示 */
    INFLUENCE_RECORD_FOR_GALLERY = 2,         /**< [en]Indicates influence record for gallery
                                                   [cn]画廊模式录制效果不佳，需切换到演讲者视图 */
    SAVE_PATH_NOT_WRITEABLE = 3,              /**< [en]Indicates save path not writeable
                                                   [cn]路径无写入权限 */
    SAVE_TEMP_RECORD_FILE = 4,                /**< [en]Indicates save temp record file
                                                   [cn]保留临时录制文件提示 */
    HDCLOUD_PERFORMANCE_CONSUMPTION = 5,      /**< [en]Indicates hdcloud performance consumption confirm
                                                   [cn]云桌面会增加性能消耗 */
    NO_SPEAKER_DEVICE = 6                     /**< [en]Indicates no speaker device
                                                   [cn]无扬声器设备 */
} HwmInterruptType;

/**
* [en]This struct is used to describe interrupt info.
* [cn]中断信息
*/
typedef struct tagHwmInterruptInfo
{
    HwmInterruptType type;                            /**< [en]Indicates interrupt type.
                                                           [cn]中断类型 */
} HwmInterruptInfo;

END_HWM_UISDK_NAMESPACE

#endif /* _HWM_UISDK_CONF_SERVICE_DEF_H_ */
