"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IConfMgr = void 0;
var enum_1 = require("./enum");
var util_1 = require("./util");
var electron = window.require('electron');
var IConfMgr = /** @class */ (function () {
    function IConfMgr(addOn) {
        this.callWithIPC = false;
        this.addOn = addOn;
        this._inProcessCallbackSet = new Set();
        this.callWithIPC = util_1.needCallWithIPC();
    }
    IConfMgr.prototype.init = function () {
        this._inProcessCallbackSet.clear();
    };
    /**
    * [en] Notify of the conf incoming [cn] 会议来电通知
    */
    IConfMgr.prototype.setOnConfIncomingNotifyCB = function (onConfIncomingNotify) {
        if (this.callWithIPC) {
            electron.ipcRenderer.sendSync('setOnConfIncomingNotifyCB', {});
            electron.ipcRenderer.on('onConfIncomingNotify', function (event, arg) {
                if (onConfIncomingNotify) {
                    onConfIncomingNotify(arg["confIncomingInfo"]);
                }
            });
            return;
        }
        this._onConfIncomingNotifyCallback = function (confIncomingInfo) {
            if (onConfIncomingNotify) {
                onConfIncomingNotify(confIncomingInfo);
            }
        };
        this.addOn.setOnConfIncomingNotifyCB(this._onConfIncomingNotifyCallback);
    };
    /**
    * [en] Conf list notify. when login/bookconf/cancelconf, will receive this notify [cn] 会议列表通知
    */
    IConfMgr.prototype.setOnConfListChangedCB = function (onConfListChanged) {
        if (this.callWithIPC) {
            electron.ipcRenderer.sendSync('setOnConfListChangedCB', {});
            electron.ipcRenderer.on('onConfListChanged', function (event, arg) {
                if (onConfListChanged) {
                    onConfListChanged(arg["confListInfo"]);
                }
            });
            return;
        }
        this._onConfListChangedCallback = function (confListInfo) {
            if (onConfListChanged) {
                onConfListChanged(confListInfo);
            }
        };
        this.addOn.setOnConfListChangedCB(this._onConfListChangedCallback);
    };
    /**
    * [en] Create Conf Finished Notify. when create conf, will receive this notify [cn] 完成创建会议通知
    */
    IConfMgr.prototype.setOnCreateConfFinishedNotifyCB = function (onCreateConfFinishedNotify) {
        if (this.callWithIPC) {
            electron.ipcRenderer.sendSync('setOnCreateConfFinishedNotifyCB', {});
            electron.ipcRenderer.on('onCreateConfFinishedNotify', function (event, arg) {
                if (onCreateConfFinishedNotify) {
                    onCreateConfFinishedNotify(arg["ret"], arg["reason"], arg["createConfResult"]);
                }
            });
            return;
        }
        this._onCreateConfFinishedNotifyCallback = function (ret, reason, createConfResult) {
            if (onCreateConfFinishedNotify) {
                onCreateConfFinishedNotify(ret, reason, createConfResult);
            }
        };
        this.addOn.setOnCreateConfFinishedNotifyCB(this._onCreateConfFinishedNotifyCallback);
    };
    /**
    * [en] This interface is book a conference. [cn] 预定会议
    */
    IConfMgr.prototype.bookConf = function (bookConfParam, onBookConfResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("bookConf")) {
            if (onBookConfResult) {
                var nullResult = {};
                onBookConfResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("bookConf");
        this._bookConfCallback = function (ret, reason, confDetail) {
            _this._inProcessCallbackSet.delete("bookConf");
            if (onBookConfResult) {
                onBookConfResult(ret, reason, confDetail);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('bookConf', { "bookConfParam": bookConfParam }).then(function (ret) {
                _this._inProcessCallbackSet.delete('bookConf');
                if (onBookConfResult) {
                    onBookConfResult(ret["ret"], ret["reason"], ret["confDetail"]);
                }
            });
        }
        this.addOn.bookConf(bookConfParam, this._bookConfCallback);
    };
    /**
    * [en] This interface is used to cancel a conference booked before. [cn] 取消会议
    */
    IConfMgr.prototype.cancelConf = function (cancelConfParam, onCancelConfResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("cancelConf")) {
            if (onCancelConfResult) {
                onCancelConfResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("cancelConf");
        this._cancelConfCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("cancelConf");
            if (onCancelConfResult) {
                onCancelConfResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('cancelConf', { "cancelConfParam": cancelConfParam }).then(function (ret) {
                _this._inProcessCallbackSet.delete('cancelConf');
                if (onCancelConfResult) {
                    onCancelConfResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.cancelConf(cancelConfParam, this._cancelConfCallback);
    };
    /**
    * [en] This interface is used to modify a conf booked before. [cn] 编辑会议
    */
    IConfMgr.prototype.editConf = function (editConfParam, onEditConfResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("editConf")) {
            if (onEditConfResult) {
                onEditConfResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("editConf");
        this._editConfCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("editConf");
            if (onEditConfResult) {
                onEditConfResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('editConf', { "editConfParam": editConfParam }).then(function (ret) {
                _this._inProcessCallbackSet.delete('editConf');
                if (onEditConfResult) {
                    onEditConfResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.editConf(editConfParam, this._editConfCallback);
    };
    /**
    * [en] This interface is used to get vmr list. [cn] 获取VMR列表
    */
    IConfMgr.prototype.getVmrList = function (onGetVmrListResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("getVmrList")) {
            if (onGetVmrListResult) {
                var nullResult = {};
                onGetVmrListResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("getVmrList");
        this._getVmrListCallback = function (ret, reason, vmrListInfo) {
            _this._inProcessCallbackSet.delete("getVmrList");
            if (onGetVmrListResult) {
                onGetVmrListResult(ret, reason, vmrListInfo);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('getVmrList', {}).then(function (ret) {
                _this._inProcessCallbackSet.delete('getVmrList');
                if (onGetVmrListResult) {
                    onGetVmrListResult(ret["ret"], ret["reason"], ret["vmrListInfo"]);
                }
            });
        }
        this.addOn.getVmrList(this._getVmrListCallback);
    };
    /**
    * [en] This interface is used to get conf detail. [cn] 获取会议详情
    */
    IConfMgr.prototype.getConfDetail = function (getConfDetailParam, onGetConfDetailResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("getConfDetail")) {
            if (onGetConfDetailResult) {
                var nullResult = {};
                onGetConfDetailResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("getConfDetail");
        this._getConfDetailCallback = function (ret, reason, confDetail) {
            _this._inProcessCallbackSet.delete("getConfDetail");
            if (onGetConfDetailResult) {
                onGetConfDetailResult(ret, reason, confDetail);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('getConfDetail', { "getConfDetailParam": getConfDetailParam }).then(function (ret) {
                _this._inProcessCallbackSet.delete('getConfDetail');
                if (onGetConfDetailResult) {
                    onGetConfDetailResult(ret["ret"], ret["reason"], ret["confDetail"]);
                }
            });
        }
        this.addOn.getConfDetail(getConfDetailParam, this._getConfDetailCallback);
    };
    /**
    * [en] This interface is used to change virtual meeting room information. [cn] 修改云会议室信息
    */
    IConfMgr.prototype.changeVmrInfo = function (vmrInfoParam, onChangeVmrInfoResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("changeVmrInfo")) {
            if (onChangeVmrInfoResult) {
                onChangeVmrInfoResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("changeVmrInfo");
        this._changeVmrInfoCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("changeVmrInfo");
            if (onChangeVmrInfoResult) {
                onChangeVmrInfoResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('changeVmrInfo', { "vmrInfoParam": vmrInfoParam }).then(function (ret) {
                _this._inProcessCallbackSet.delete('changeVmrInfo');
                if (onChangeVmrInfoResult) {
                    onChangeVmrInfoResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.changeVmrInfo(vmrInfoParam, this._changeVmrInfoCallback);
    };
    /**
    * [en]This interface is used to get conf list [cn]获取会议列表
    */
    IConfMgr.prototype.getConfList = function (onGetConfListResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("getConfList")) {
            if (onGetConfListResult) {
                var nullResult = {};
                onGetConfListResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("getConfList");
        this._getConfListCallback = function (ret, reason, confListInfo) {
            _this._inProcessCallbackSet.delete("getConfList");
            if (onGetConfListResult) {
                onGetConfListResult(ret, reason, confListInfo);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('getConfList', {}).then(function (ret) {
                _this._inProcessCallbackSet.delete('getConfList');
                if (onGetConfListResult) {
                    onGetConfListResult(ret["ret"], ret["reason"], ret["confListInfo"]);
                }
            });
        }
        this.addOn.getConfList(this._getConfListCallback);
    };
    /**
    * [en] [cn]预约周期会议
    */
    IConfMgr.prototype.bookCycleConf = function (bookCycleConfParam, onBookCycleConfResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("bookCycleConf")) {
            if (onBookCycleConfResult) {
                var nullResult = {};
                onBookCycleConfResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process", nullResult);
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("bookCycleConf");
        this._bookCycleConfCallback = function (ret, reason, confDetail) {
            _this._inProcessCallbackSet.delete("bookCycleConf");
            if (onBookCycleConfResult) {
                onBookCycleConfResult(ret, reason, confDetail);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('bookCycleConf', { "bookCycleConfParam": bookCycleConfParam }).then(function (ret) {
                _this._inProcessCallbackSet.delete('bookCycleConf');
                if (onBookCycleConfResult) {
                    onBookCycleConfResult(ret["ret"], ret["reason"], ret["confDetail"]);
                }
            });
        }
        this.addOn.bookCycleConf(bookCycleConfParam, this._bookCycleConfCallback);
    };
    /**
    * [en] [cn]编辑周期会议
    */
    IConfMgr.prototype.modifyCycleConf = function (modifyParam, onModifyCycleConfResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("modifyCycleConf")) {
            if (onModifyCycleConfResult) {
                onModifyCycleConfResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("modifyCycleConf");
        this._modifyCycleConfCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("modifyCycleConf");
            if (onModifyCycleConfResult) {
                onModifyCycleConfResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('modifyCycleConf', { "modifyParam": modifyParam }).then(function (ret) {
                _this._inProcessCallbackSet.delete('modifyCycleConf');
                if (onModifyCycleConfResult) {
                    onModifyCycleConfResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.modifyCycleConf(modifyParam, this._modifyCycleConfCallback);
    };
    /**
    * [en] [cn]编辑周期子会议
    */
    IConfMgr.prototype.modifySubCycleConf = function (modifySubCycleConfParam, onModifySubCycleConfResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("modifySubCycleConf")) {
            if (onModifySubCycleConfResult) {
                onModifySubCycleConfResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("modifySubCycleConf");
        this._modifySubCycleConfCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("modifySubCycleConf");
            if (onModifySubCycleConfResult) {
                onModifySubCycleConfResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('modifySubCycleConf', { "modifySubCycleConfParam": modifySubCycleConfParam }).then(function (ret) {
                _this._inProcessCallbackSet.delete('modifySubCycleConf');
                if (onModifySubCycleConfResult) {
                    onModifySubCycleConfResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.modifySubCycleConf(modifySubCycleConfParam, this._modifySubCycleConfCallback);
    };
    /**
    * [en] [cn]取消周期会议
    */
    IConfMgr.prototype.cancelCycleConf = function (cancelCycleConfParam, onCancelCycleConfResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("cancelCycleConf")) {
            if (onCancelCycleConfResult) {
                onCancelCycleConfResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("cancelCycleConf");
        this._cancelCycleConfCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("cancelCycleConf");
            if (onCancelCycleConfResult) {
                onCancelCycleConfResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('cancelCycleConf', { "cancelCycleConfParam": cancelCycleConfParam }).then(function (ret) {
                _this._inProcessCallbackSet.delete('cancelCycleConf');
                if (onCancelCycleConfResult) {
                    onCancelCycleConfResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.cancelCycleConf(cancelCycleConfParam, this._cancelCycleConfCallback);
    };
    /**
    * [en] [cn]取消周期子会议
    */
    IConfMgr.prototype.cancelSubCycleConf = function (cancelSubCycleConfParam, onCancelSubCycleConfResult) {
        var _this = this;
        if (this._inProcessCallbackSet.has("cancelSubCycleConf")) {
            if (onCancelSubCycleConfResult) {
                onCancelSubCycleConfResult(enum_1.SDKERR.UISDK_IN_PROCESS, "uisdk in process");
            }
            ;
            return;
        }
        this._inProcessCallbackSet.add("cancelSubCycleConf");
        this._cancelSubCycleConfCallback = function (ret, reason) {
            _this._inProcessCallbackSet.delete("cancelSubCycleConf");
            if (onCancelSubCycleConfResult) {
                onCancelSubCycleConfResult(ret, reason);
            }
        };
        if (this.callWithIPC) {
            electron.ipcRenderer.invoke('cancelSubCycleConf', { "cancelSubCycleConfParam": cancelSubCycleConfParam }).then(function (ret) {
                _this._inProcessCallbackSet.delete('cancelSubCycleConf');
                if (onCancelSubCycleConfResult) {
                    onCancelSubCycleConfResult(ret["ret"], ret["reason"]);
                }
            });
        }
        this.addOn.cancelSubCycleConf(cancelSubCycleConfParam, this._cancelSubCycleConfCallback);
    };
    return IConfMgr;
}());
exports.IConfMgr = IConfMgr;
