﻿/**
* @file HwmHTTPDef.h
*
* Copyright(C), 2024-2025, Huawei Tech. Co., Ltd. ALL RIGHTS RESERVED.
*
* @brief Xxxxxx.
*/

#ifndef __HWM_HTTP_DEF_H__
#define __HWM_HTTP_DEF_H__

#import "HwmSdkCommonDef.h"
#import "HwmGlobalErrorCodeDef.h"

BEGIN_HWM_SDK_NAMESPACE


#define HWM_HTTP_HEADER_NAME_MAX_LEN            (64)
#define HWM_HTTP_HEADER_VALUE_MAX_LEN           (640)
#define HWM_HTTP_AUTH_NAME                      (256)
#define HWM_HTTP_AUTH_PASSWORD                  (256)
#define HWM_HTTP_MAX_CA_PATH_LEN                (512)
#define HWM_HTTP_MAX_CERT_KEY_LEN               (256)
#define HWM_HTTP_REQUEST_ID_LEN                 (33)
#define HWM_HTTP_MAX_URL_LEN                    (2048)
#define HWM_HTTP_MAX_FILE_LEN                   (512)
#define HWM_HTTP_MAX_FILE_FORM_KEY_LEN          (32)
#define HWM_HTTP_MAX_FORM_KEY_LEN               (128)
#define HWM_HTTP_MAX_FORM_CONTENT_LEN           (512)

/**
 * [en]HTTP Request Type
 * [cn]http 请求类型
 */
typedef enum tagHTTPRequestType
{
    HTTP_REQUEST_TYPE_POST,                   /**< [en]Indicates post request. [cn]post请求 */
    HTTP_REQUEST_TYPE_DELETE,                 /**< [en]Indicates delete request. [cn]delete请求 */
    HTTP_REQUEST_TYPE_PUT,                    /**< [en]Indicates put request. [cn]put请求 */
    HTTP_REQUEST_TYPE_GET,                    /**< [en]Indicates get request. [cn]get请求 */
    HTTP_REQUEST_TYPE_HEAD,                   /**< [en]Indicates head request. [cn]head请求 */
    HTTP_REQUEST_TYPE_BUTT                    /**< [en]Indicates butt. [cn]无效值 */
}HTTPRequestType;

/**
 * [en]auth mode
 * [cn]鉴权模式
 */
typedef enum tagHTTPAuthMode
{
    HTTP_AUTH_MODE_AUTO = 0,              /**< [en]Indicates BASIC and DIGEST mode (adapted by internal) . [cn]自动模式, 内部调整 */
    HTTP_AUTH_MODE_BASIC,                 /**< [en]Indicates BASIC mode. [cn]basic模式 */
    HTTP_AUTH_MODE_DIGEST                 /**< [en]Indicates DIGEST mode. [cn]digest模式 */
}HTTPAuthMode;

/**
 * [en]response auth mode
 * [cn]响应结果的鉴权模式
 */
typedef enum tagHTTPRspAuthMode{
    HTTP_EN_AUTH_CLOSE  = 0,            /**< [en]auth closed (default mode) [cn]关闭*/
    HTTP_EN_AUTH_BASIC  = 1,            /**< [en]BASIC mode [en]basic模式*/
    HTTP_EN_AUTH_DIGEST = 2,            /**< [en]DIGEST mode [en]digest模式*/
    HTTP_EN_AUTH_BASIC_DIGEST = 3,      /**< [en]BASIC and DIGEST mode (adapted by internal)[en]basic和digest模式 */
    HTTP_EN_AUTH_NTLM = 4,              /**< [en]NTLM mode [en]NTLM模式*/
    HTTP_EN_AUTH_MODE_BUTT              /**< [en]invalid mode [en]无效值*/
}HTTPRspAuthMode;

/**
 * [en]auth info
 * [cn]鉴权信息
 */
typedef struct tagHTTPAuthInfo
{
    HTTPAuthMode mode;                              /**< [en]Indicates mode. [cn]鉴权模式 */
    HWM_CHAR userName[HWM_HTTP_AUTH_NAME];          /**< [en]Indicates user name. [cn]鉴权用户名 [cs_allowNull:Y] */
    HWM_CHAR password[HWM_HTTP_AUTH_PASSWORD];      /**< [en]Indicates password. [cn]鉴权密码 [cs_allowNull:Y] */
}HTTPAuthInfo;

/**
 * [en]verify mode
 * [cn]验证模式
 */
typedef enum tagHTTPVerifyMode
{
    HTTP_VERIFY_MODE_NONE        = 0,                  /**< [en]Indicates no authentication.[cn]不认证 */
    HTTP_VERIFY_MODE_SERVER      = 1,                  /**< [en]Indicates authentication server.[cn]认证服务端 */
    HTTP_VERIFY_MODE_CLIENT      = 2,                  /**< [en]Indicates authentication server.<br>[cn]认证客户端 */
    HTTP_VERIFY_MODE_BOTH        = 3                   /**< [en]Indicates authentication both server and client.[cn]客户端，服务器双向认证 */
}HTTPVerifyMode;

/**
 * [en]TSL verify service mode
 * [cn]TLS鉴权服务器模式
 */
typedef enum tagHTTPVerifyServerMode
{
    HTTP_VERIFY_SERVER_PEER             = 0,      /**< [en]only verify certificate (default mode). [cn]默认鉴权模式. */
    HTTP_VERIFY_SERVER_PEER_AND_HOST    = 1,      /**< [en]verify certificate and host both.[cn]验证证书和主机. */
    HTTP_VERIFY_SERVER_BUTT                       /**< [en]Indicates butt. [cn]无效值 */
}HTTPVerifyServerMode;

/**
 * [en]TSL verify crl mode
 * [cn]TLS鉴权crl模式
 */
typedef enum tagHTTPVerifyCrlCheck
{
    HTTP_VERIFY_CRL_CHECK_AUTO = 0,               /**< [en]Indicates auto.[cn]HTTP_E_VERIFY_MODE为HTTP_E_VERIFY_MODE_SERVER和HTTP_E_VERIFY_MODE_BOTH时检验 */
    HTTP_VERIFY_CRL_CHECK_NONE = 1                /**< [en]Indicates none. [cn]不检查 */
}HTTPVerifyCrlCheck;

/**
 * [en]TSL param
 * [cn]TLS参数
 */
typedef struct tagHTTPTlsParam
{
    HWM_CHAR    caCertPath[HWM_HTTP_MAX_CA_PATH_LEN];                   /**< [en]CA certificate.[cn]CA证书路径 [cs_allowNull:N] */
    HWM_CHAR    clientCertPath[HWM_HTTP_MAX_CA_PATH_LEN];               /**< [en]client certificate.[cn]客户端证书路径 [cs_allowNull:Y] */
    HWM_CHAR    clientKeyPath[HWM_HTTP_MAX_CA_PATH_LEN];                /**< [en]client key.[cn]客户端key路径 [cs_allowNull:Y] */
    HWM_CHAR    clientKeyPwd[HWM_HTTP_MAX_CERT_KEY_LEN];                /**< [en]client key password.[cn]客户端key密码  [cs_allowNull:Y] */
    HTTPVerifyMode tlsMode;                                             /**< [en]TLS VERIFY MODE.[cn]tls模式 [cs_allowNull:N] */
    HTTPVerifyServerMode verifyServerMode;                              /**< [en]TLS VERIFY SERVER MODE.[cn]服务器验证模式 [cs_allowNull:N] */
    HTTPVerifyCrlCheck checkCrl;                                        /**< [en]TLS VERIFY SERVER MODE.[cn]crl检查模式 [cs_allowNull:N] */
}HTTPTlsParam;

/**
 * [en]proxy param
 * [cn]代理参数
 */
typedef struct tagHTTPProxyParam
{
    HWM_CHAR    address[HWM_HTTP_MAX_URL_LEN];                          /**< [en]proxy addr.[cn]代理地址 [cs_allowNull:N] */
    HWM_UINT32  port;                                                   /**< [en]proxy port.[cn]代理端口 [cs_allowNull:N] */
    HWM_CHAR    userName[HWM_HTTP_AUTH_NAME];                           /**< [en]proxy user name.[cn]代理用户名 [cs_allowNull:Y] */
    HWM_CHAR    password[HWM_HTTP_AUTH_PASSWORD];                       /**< [en]proxy password.[cn]代理密码 [cs_allowNull:Y] */
}HTTPProxyParam;

/**
 * [en]http config param
 * [cn]http配置参数
 */
typedef struct tagHTTPConfigParam
{
    HTTPTlsParam        tlsParam;                                       /**< [en]tls param.[cn]tls参数 [cs_allowNull:Y] */
    HTTPProxyParam      proxyParam;                                     /**< [en]proxy param.[cn]代理参数 [cs_allowNull:Y] */
    HWM_UINT32          reqTimeout;                                     /**< [en]req timeout.[cn]http请求超时时间 [cs_allowNull:Y] */
}HTTPConfigParam;

/**
 * [en]header
 * [cn]头域
 */
typedef struct tagHTTPHeader
{
    HWM_CHAR acHeaderName[HWM_HTTP_HEADER_NAME_MAX_LEN];                /**< [en]header name.[cn]header的key */
    HWM_CHAR acHeaderValue[HWM_HTTP_HEADER_VALUE_MAX_LEN];              /**< [en]header value.[cn]header数据 */
}HTTPHeader;

/**
 * [en]http request param
 * [cn]http 请求参数
 */
typedef struct tagHTTPReqParam
{
    HWM_CHAR            url[HWM_HTTP_MAX_URL_LEN];              /**< [en]url.[cn]请求地址 */
    HTTPRequestType     type;                                   /**< [en]type.[cn]请求类型 */
    HTTPHeader*         header;                                 /**< [en]header.[cn]头域 [cs_allowNull:Y] */
    HWM_UINT32          headerLen;                              /**< [length for header][en]header nums.[cn]头域的个数 [cs_allowNull:Y] */
    HWM_CHAR*           body;                                   /**< [en]body.[cn]body体 [cs_allowNull:Y] */
    HWM_UINT32          bodyLen;                                /**< [length for body][en]body lenght.[cn]body体长度 [cs_allowNull:Y] */
    HTTPAuthInfo        authInfo;                               /**< [en]auth info.[cn]鉴权信息 [cs_allowNull:Y] */
    HWM_CHAR            requestId[HWM_HTTP_REQUEST_ID_LEN];     /**< [en]requestid.[cn]请求唯一标识 [cs_allowNull:Y] */
}HTTPReqParam;

/**
 * [en]HTPP response
 * [cn]HTTP请求响应
 */
typedef struct tagHTTPRsp
{
    HWM_UINT32          statusCode;                             /**< [en]response [cn]响应码 */
    HTTPHeader*         header;                                 /**< [en]header.[cn]响应头域 [cs_allowNull:Y] */
    HWM_UINT32          headerLen;                              /**< [length for header][en]header lengh.[cn]响应头域的个数 [cs_allowNull:Y] */
    HWM_CHAR*           body;                                   /**< [en]body.[cn]body体 [cs_allowNull:Y] */
    HWM_UINT32          bodyLen;                                /**< [length for body][en]body lenght.[cn]body体长度[cs_allowNull:Y]  */
    HWM_CHAR            redirectUrl[HWM_HTTP_MAX_URL_LEN];      /**< [en]redirect url.[cn]重定向url [cs_allowNull:Y] */
    HWM_CHAR            realAuthName[HWM_HTTP_AUTH_NAME];       /**< [en]real auth name.[cn]实际鉴权的用户名 [cs_allowNull:Y] */
    HTTPRspAuthMode     enAuthMode;                             /**< [en]auth mode.[cn]鉴权模式 */
}HTTPRsp;

typedef struct tagHTTPResult
{
    SDKERR              result;                 /**< [en]interface result.[cn]接口调用结果*/
    HWM_UINT32          handle;                 /**< [en]http handle.[cn]http 句柄*/
}HTTPResult;

/**
 * [en]asynchronous request
 * [cn]异步请求参数
 */
typedef struct tagHTTPAsynSendParam
{
    HTTPReqParam        reqParam;              /**< [en]request params.[cn]请求参数 */
    HTTPConfigParam     configParam;           /**< [en]config params.[cn]配置参数 [cs_allowNull:Y] */
}HTTPAsynSendParam;

/**
 * [en]form option
 * [cn]表单选项
 */
typedef struct tagHTTPFormOption
{
    HWM_CHAR    key[HWM_HTTP_MAX_FORM_KEY_LEN];             /**< [en]key.[cn]form option的key */
    HWM_CHAR    content[HWM_HTTP_MAX_FORM_CONTENT_LEN];     /**< [en]content.[cn]form option的content */
}HTTPFormOption;

/**
 * [en]upload file param
 * [cn]上传文件参数
 */
typedef struct tagHTTPUploadFileParam
{
    HTTPReqParam        reqParam;                                       /**< [en]request params.[cn]请求参数 */
    HTTPConfigParam     configParam;                                    /**< [en]config params.[cn]配置参数 [cs_allowNull:Y] */

    HWM_CHAR            filePath[HWM_HTTP_MAX_FILE_LEN];                /**< [en]file path.[cn]文件路径 [cs_allowNull:N] */
    HWM_CHAR            fileFormKey[HWM_HTTP_MAX_FILE_FORM_KEY_LEN];    /**< [en]file form key.[cn]文件key [cs_allowNull:Y] */
    HTTPFormOption*     formOption;                                     /**< [en]form options.[cn]文件格式选项 [cs_allowNull:Y] */
    HWM_UINT32          formOptionLen;                                  /**< [length for formOption][en]form options nums.[cn]文件格式选项个数 [cs_allowNull:Y] */
}HTTPUploadFileParam;

/**
 * [en]download file param
 * [cn]下载文件参数
 */
typedef struct tagHTTPDownloadFileParam
{
    HTTPReqParam        reqParam;                           /**< [en]request params.[cn]请求参数 */
    HTTPConfigParam     configParam;                        /**< [en]config params.[cn]配置参数 [cs_allowNull:Y] */

    HWM_CHAR            filePath[HWM_HTTP_MAX_FILE_LEN];    /**< [en]file path.[cn]文件路径 [cs_allowNull:N] */
}HTTPDownloadFileParam;


END_HWM_SDK_NAMESPACE

#endif  /* __HWM_HTTP_DEF_H__ */