﻿/**
 * @file HwmConfService.h
 *
 * Copyright(C), 2019-2020, Huawei Tech. Co., Ltd. ALL RIGHTS RESERVED.
 *
 * @brief Xxxxxx.
 */

#ifndef __HWM_PRIVATE_CONF_SERVICE_H__
#define __HWM_PRIVATE_CONF_SERVICE_H__


#import "HwmPrivateConfServiceDef.h"
#import "HwmConfServiceShare.h"
#import "HwmGlobalErrorCodeDef.h"

BEGIN_HWM_SDK_NAMESPACE

/**
* @brief [en]This class is about confMgr result callback.
*        [cn]会议管理的结果回调类回调
*/
class HWM_API IHwmPrivateConfMgrResultCallback
{
public:
    /**
    * @brief
    * [en]This class is about the result of booking CTD conference..
    * [cn]预约CTD会议的结果回调
    * @private
    *
    * @UTCallback
    */
    virtual HWM_VOID OnBookCtdConfResult(SDKERR result, const HWM_CHAR* reasonDesc, const CreateConfResult* bookCtdConfResult) = 0;
    
    /**
    * @brief
    * [en]
    * [cn]匿名会议，校验滑块并获取验证码的结果回调
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @since:v1.0
    *
    * @UTCallback
    */
    virtual HWM_VOID OnRequestVerifycodeResult(SDKERR result, const HWM_CHAR* reasonDesc) = 0;

    /**
    * @brief
    * [en]
    * [cn]匿名会议，刷新滑块的结果回调
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @param sliderInfo [type:GetSliderResultInfo][cs_allowNull:N]
    * [en]
    * [cn]滑块结果
    * @see GetSliderResultInfo
    *
    * @since:v1.0
    *
    * @UTCallback
    */
    virtual HWM_VOID OnRefreshSliderResult(SDKERR result, const HWM_CHAR* reasonDesc, const GetSliderResultInfo* sliderInfo) = 0;

    /**
    * @brief
    * [en]This class is about the result of joining a conference successfully.
    * [cn]加入会议成功的的结果回调
    *
    * @param joinConfResultInfo [type:JoinConfResultInfo][cs_allowNull:N]
    * [en]
    * [cn]加入会议成功的结果
    * @see JoinConfResultInfo
    *
    * @since:v1.0
    *
    * @android:initRender
    * @android:setAudioRouteInCall
    * @android:resetAudioRoute
	* @ios:initAudioRoute
    * @Callback ["JoinConfByVerifyCode","JoinConfDirect","JoinConfByLink","JoinConfByNonce","JoinConfByRandom"]
    *
    * @UTCallback ["JoinConfByVerifyCode","JoinConfDirect","JoinConfByLink","JoinConfByNonce","JoinConfByRandom"]
    */
    virtual HWM_VOID OnJoinConfSuccess(const JoinConfResultInfo* joinConfResultInfo) = 0;

    /**
    * @brief
    * [en]This class is about the the result of failing to join a conference.
    * [cn]加入会议失败的的结果回调
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @param joinConfFailedInfo [type:JoinConfFailedInfo][cs_allowNull:N]
    * [en]
    * [cn]加入会议失败的结果
    * @see JoinConfFailedInfo
    *
    * @since:v1.0
    *
    * @android:uninitRender
    * @mobile:leaveDataConf
    * @android:setAudioRouteOutCall
	* @ios:resetAudioRoute
    * @Callback ["JoinConfByVerifyCode","JoinConfDirect","JoinConfByLink","JoinConfByNonce","JoinConfByRandom"]
    *
    * @UTCallback ["JoinConfByVerifyCode","JoinConfDirect","JoinConfByLink","JoinConfByNonce","JoinConfByRandom"]
    */
    virtual HWM_VOID OnJoinConfFailed(SDKERR result, const HWM_CHAR* reasonDesc, const JoinConfFailedInfo* joinConfFailedInfo) = 0;

    /**
    * @brief
    * [en]
    * [cn]匿名会议，获取短信验证码的结果回调
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @param sliderInfo [type:GetSliderResultInfo][cs_allowNull:N]
    * [en]
    * [cn]滑块结果
    * @see GetSliderResultInfo
    *
    * @since:v1.0
    *
    * @UTCallback
    */
    virtual HWM_VOID OnCheckNeedSliderAuthResult(SDKERR result, const HWM_CHAR* reasonDesc, const GetSliderResultInfo* sliderInfo) = 0;

    /**
    * @brief
    * [en]
    * [cn]二维码配对的结果回调
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @since:v1.0
    *
    * @UTCallback
    */
    virtual HWM_VOID OnStartQrCodePairResult(SDKERR result, const HWM_CHAR* reasonDesc) = 0;
    
    /**
    * @brief
    * [en]
    * [cn]强制二维码配对的结果回调
    * 
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    * 
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    * 
    * @since:v1.0
    *
    * @UTCallback
    */
    virtual HWM_VOID OnStartQrCodePairForceResult(SDKERR result, const HWM_CHAR* reasonDesc) = 0;
    
    /**
    * @brief
    * [en]
    * [cn]结束二维码配对的结果回调
    * 
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    * 
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    * 
    * @since:v1.0
    *
    * @UTCallback
    */
    virtual HWM_VOID OnEndQrCodePairResult(SDKERR result, const HWM_CHAR* reasonDesc) = 0;

    /**
    * @brief
    * [en]
    * [cn]二维码入会的结果回调
    * 
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    * 
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    * 
    * @since:v1.0
    *
    * @UTCallback
    */
    virtual HWM_VOID OnJoinPairConfResult(SDKERR result, const HWM_CHAR* reasonDesc) = 0;

    /**
     * 查询历史会议列表，参数时间为毫秒结果
     *
     * @UTCallback
     */
    virtual HWM_VOID OnQueryHistoryConflistResult(SDKERR result, const HWM_CHAR* reasonDesc, const HistoryConfListInfo* confListInfo) = 0;

    /**
     * 查询历史会议与会者信息结果
     *
     * @UTCallback
     */
    virtual HWM_VOID OnQueryHistoryConfAttendeeResult(SDKERR result, const HWM_CHAR* reasonDesc, const HistoryConfAttendeeResult* attendeeResult) = 0;

    /** 
     * 查询录制文件结果 
     *
     * @UTCallback
     */
    virtual HWM_VOID OnQueryHistoryConfRecordFileResult(SDKERR result, const HWM_CHAR* reasonDesc, const HWM_CHAR* fileUrl) = 0;

    /**
     * 查询本地历史会议列表，参数时间为毫秒结果
    *
    * @UTCallback
     */
    virtual HWM_VOID OnQueryLocalHistoryConflistResult(SDKERR result, const HWM_CHAR* reasonDesc, const HistoryConfListInfo* confListInfo) = 0;

    /**
    * @brief
    * [en]
    * [cn]查询基本会议信息
    *
    * @UTCallback
    */
    virtual HWM_VOID OnQueryBasicConfInfoResult(SDKERR result, const HWM_CHAR* reasonDesc, const BasicConfInfo* confListInfo) {};

    /**
    * @brief [en] xxxxxx
    *        [cn] 查询Nonce结果
    *
    * @param result [type:SDKERR]
    * [en]
    * [cn]错误码
    * @see SDKERR
    * @param nonceType [type:NonceType]
    * [en]
    * [cn]Nonce类型
    *
    * @param reasonDesc [type:HWM_CHAR*]
    * [en]
    * [cn]错误描述
    *
    * @param nonce [type:HWM_CHAR*]
    * [en]
    * [cn]nonce值
    **/
    virtual HWM_VOID OnQueryNonceByTypeResult(SDKERR result, NonceType nonceType,const HWM_CHAR* reasonDesc, const HWM_CHAR* nonce) {};

    /**
    * @brief
    * [en]
    * [cn查询文件结果回调
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @param QueryFileListsResult [type:QueryFileListsResult*][cs_allowNull:N]
    * [en]
    * [cn]创建文档的结果
    * @see QueryFileListsResult
    *
    * @android:disable
    * @mac:disable
    * @UTCallback
    */
    virtual HWM_VOID OnQueryDocFileListsResult(SDKERR result, const HWM_CHAR* reasonDesc, const QueryFileListsResult* queryDocResult) {};

    /**
    * @brief
    * [en]
    * [cn删除文件结果回调
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @android:disable
    * @mac:disable
    * @UTCallback
    */
    virtual HWM_VOID OnDeleteDocumentResult(SDKERR result, const HWM_CHAR* reasonDesc) {};

    /**
    * @brief
    * [en]
    * [cn删除文件结果回调
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @param ApplyDocOperRightResult [type:ApplyDocOperRightResult*][cs_allowNull:N]
    * [en]
    * [cn]申请文档权限结果
    * @see ApplyDocOperRightResult
    * @android:disable
    * @mac:disable
    * @UTCallback
    */
    virtual HWM_VOID OnApplyDocOperateRightResult(SDKERR result, const HWM_CHAR* reasonDesc, const ApplyDocOperRightResult* applyDocRightResult) {};


    /**
    * @brief
    * [en]
    * [cn]创建文件结果回调
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @param fileGuid [type:HWM_CHAR*][cs_allowNull:Y]
    * [en]
    * [cn]文件guid
    *
    * @android:disable
    * @mac:disable
    * @UTCallback
    */
    virtual HWM_VOID OnCreateFolderResult(SDKERR result, const HWM_CHAR* reasonDesc, const HWM_CHAR* fileGuid) {};

    /**
    * @brief
    * [en]
    * [cn]查询文档权限链接结果回调
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @param docAccessUrl [type:HWM_CHAR*][cs_allowNull:Y]
    * [en]
    * [cn]文档权限链接
    *
    * @param folderGuid [type:HWM_CHAR*][cs_allowNull:Y]
    * [en]
    * [cn]文件夹id
    *
    * @pc:disable
    * @android:disable
    * @mac:disable
    * @UTCallback
    */
    virtual HWM_VOID OnQueryDocAccessUrlResult(SDKERR result, const HWM_CHAR* reasonDesc, const HWM_CHAR* docAccessUrl, const HWM_CHAR* folderGuid) {};

    
};

/**
* @brief [en]This class is about confMgr notify callback.
*        [cn]会议管理通知类回调
*/
class HWM_API IHwmPrivateConfMgrNotifyCallback
{
public:
    /**
    * @brief
    * [en]This class is about the notification of data conference parameters.(Used by mobile terminal)
    * [cn]数据会议参数通知(移动端使用)
    * @private
    * @mobile:joinDataConf
    * 
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    * 
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    * 
    * @since:v1.0
    */
    virtual HWM_VOID OnJoinShareConfParamNotify(const JoinShareConfParamInfo* joinShareConfParamInfo) = 0;

    /**
    * @brief
    * [en]This class is about pair universal info.
    * [cn]配对通用信息通知
    * @private
    * @ios:disable
    */
    virtual HWM_VOID OnPairUniversalInfoNotify(const PairUniversalInfo* pairUniversalInfo) = 0;

    /**
    * @brief
    * [en]This class is about data conf leave notify.
    * [cn]离开数据会议通知
    * @private
    * @mobile:disable
    */
    virtual HWM_VOID OnDataConfLeaveNotify() {};

    /**
    * @brief
    * [en]This class is about data conf join notify.
    * [cn]加入数据会议通知
    * @private
    * @mobile:disable
    */
    virtual HWM_VOID OnDataConfJoinNotify() {};

    /**
    * @brief
    * [en]This class is about paired or canceled.
    * [cn]配对事件通知
    * @private
    * @ios:disable
    */
    virtual HWM_VOID OnPairedOrCanceledNotify(PairEventInfo* pairEventInfo) = 0;

    /**
    * @brief
    * [en]This class is about the notification of upload key log.
    * [cn]关键日志上传通知
    *
    * @param keyLogInfo [type:UploadKeyLogInfo][cs_allowNull:N]
    * [en]
    * [cn]关键日志上传信息
    * @see UploadKeyLogInfo
    *
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnUploadKeyLogNotify(const UploadKeyLogInfo* keyLogInfo) = 0;

    /**
    * @brief
    * [en]This class is about the notification of anonymous membership account cancellation.
    * [cn]匿名入会账户注销通知
    * @mobile:imLogout
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnAnonyJoinConfLogoutNotify(SDKERR result, const HWM_CHAR* reasonDesc) = 0;

    /**
    * @brief
    * [en]This class is about paired or canceled.
    * [cn]收到配对请求通知
    * @private
    * @ios:disable
    */
    virtual HWM_VOID OnPairRequestNotify(PairReqInfo* pairReqInfo) = 0;

    /**
    * @brief
    * [en]This class is about site info notify.
    * [cn]站点信息通知
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @param siteInfo [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]站点信息
    *
    * @private
    */
    virtual HWM_VOID OnSiteInfoNotify(SDKERR result, const HWM_CHAR* reasonDesc, const HWM_CHAR* siteInfo){};

    /**
    * @brief
    * [en]
    * [cn]音视频媒体传输模式通知
    * @private
    *
    * @param mediaTransMode [type:HWM_UINT32]
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnMediaTransModeNotify(HWM_UINT32 mediaTransMode) {};
	
    /**
    * @brief
    * [en]This class is about the callback of transfering point to point call to conference.
    * [cn]MMR立即会议记录回调
    *
    * @param confInfo [type:P2PConfRecordInfo][cs_allowNull:N]
    * [en]
    * [cn]会议记录信息
    *
    * @return [type:HWM_VOID]
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnP2PConfRecordNotify(SDKERR result, const HWM_CHAR* reasonDesc, P2PConfRecordInfo* recordInfo) {};

    /**
    * @brief
    * [en]
    * [cn]异常断开的会议的通知
    *
    * @param abnormalLeaveMeetingInfo [type:AbnormalLeaveMeetingInfo][cs_allowNull:N]
    * [en]
    * [cn]异常断开的会议信息
    *
    * @return [type:HWM_VOID]
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnAbnormalLeaveMeetingNotify(const AbnormalLeaveMeetingInfo* abnormalLeaveMeetingInfo) {};
};

/**
* @brief [en]This class is about confstate notify callback.
*        [cn]会议状态通知类回调
*/
class HWM_API IHwmPrivateConfStateNotifyCallback
{
public:
    /**
    * @brief
    * [en]
    * [cn]会议状态变更: 是否支持AI录制的通知回调
    *
    * @param supportAiRecord [type:HWM_BOOL][cs_allowNull:N]
    * [en]
    * [cn]是否支持AI录制
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnConfSupportAiRecordChanged(HWM_BOOL supportAiRecord) = 0;

    /**
    * @brief
    * [en]
    * [cn]支持投票功能的通知：返回true则为支持，返回false则认为不支持
    *
    * @param supportVote [type:HWM_BOOL][cs_allowNull:N]
    * [en]
    * [cn]支持投票通知
    *
    */
    virtual HWM_VOID OnVoteInfoNotify(const VoteInfo* voteInfo) {};

    /**
    * @brief
    * [en]
    * [cn]会议状态变更: 共享标注权限变更
    *
    * @param permission [type:AnnotationPermission][cs_allowNull:N]
    * [en]
    * [cn]共享标注权限
    *
    */
    virtual HWM_VOID OnAnnotationPermissionChanged(AnnotationPermission permission, HWM_BOOL isConfLevel) {};

        /**
    * @brief
    * [en]
    * [cn]会议媒体流模式变更
    *
    * @param mode [type:ConfMediaFlowMode][cs_allowNull:N]
    * [en]
    * [cn]会议媒体流模式
    * @mobile:disable
    * @mac:disable
    *
    */
    virtual HWM_VOID OnConfMediaFlowModeChanged(ConfMediaFlowMode mode) {};

    /**
    * @brief
    * [en]
    * [cn] 媒体分离场景是否支持音频ai降噪功能通知
    *
    * @param mode [type:HWM_BOOL][cs_allowNull:N]
    * [en]
    * [cn]是否支持ai降噪功能
    * @mobile:disable
    * @mac:disable
    *
    */
    virtual HWM_VOID OnConfSupportAudioTnrInMediaSeparateNotify(HWM_BOOL supportAudioTnr) {};

    /**
    * @brief [en]Notify of conf event info
    *        [cn]会中事件信息上报通知
    *
    * @param eventInfo [type:HWM_CHAR*]
    * [en] Conf state info
    * [cn] 会议状态信息
    */
    virtual HWM_VOID OnConfEventInfoNotify(const HWM_CHAR* eventInfo) {};

    /**
    * @brief
    * [en]
    * [cn]主被叫保密通话能力不一致
    *
    * @pc:disable
    */
    virtual HWM_VOID OnChipEncryptInconsistentNotify() {};
};

/**
* @brief [en]This class is about confctrl result callback.
*        [cn]会议控制操作类回调
*/
class HWM_API IHwmPrivateConfCtrlResultCallback
{
public:
    /**
    * @brief
    * [en]
    * [cn]邀请硬终端的结果回调
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @since:v1.0
    *
    * @UTCallback
    */
    virtual HWM_VOID OnInviteHardTerminalByScanQRCodeResult(SDKERR result, const HWM_CHAR* reasonDesc) = 0;

    /**
    * @brief
    * [en]
    * [cn]处理TUP上报的AI会议纪要的结果回调
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @param isOpen [type:HWM_UINT32][cs_allowNull:N]
    * [en]
    * [cn]是否上报
    *
    * @since:v1.0
    *
    * @UTCallback
    */
    virtual HWM_VOID OnAiConfRecordResult(SDKERR result, const HWM_CHAR* reasonDesc, HWM_UINT32 isOpen) = 0;

    /**
    * @brief
    * [en]
    * [cn]设置聊天权限结果回调
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @param chatPermission [type:ChatPermission]
    * [en]
    * [cn]聊天权限
    *
    * @since:v1.0
    *
    * @UTCallback
    */
    virtual HWM_VOID OnSetChatPermissionResult(SDKERR result, const HWM_CHAR* reasonDesc, ChatPermission chatPermission) {};

    /**
    * @brief
    * [en]
    * [cn]延长会议时间操作结果
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @UTCallback
    */
    virtual HWM_VOID OnSetExtendConfDurationResult(SDKERR result, const HWM_CHAR* reasonDesc) {};

    /**
    * @brief
    * [en]
    * [cn]设置共享标注权限结果
    *
    * @param result [type:SDKERR][cs_allowNull:N]
    * [en]
    * [cn]错误码
    * @see SDKERR
    *
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]错误描述
    *
    * @UTCallback
    */
    virtual HWM_VOID OnSetAnnotationPermissionResult(SDKERR result, const HWM_CHAR* reasonDesc, HWM_BOOL isConfLevel) {};


};

/**
* @brief [en]This class is about confctrl notify callback.
*        [cn]会议控制通知类回调
*/
class HWM_API IHwmPrivateConfCtrlNotifyCallback
{
public:

    /**
    * @brief
    * [en]xxxx
    * [cn]本地录制收到视频流
    *
    * @mobile:disable
    * @win:disable
    *
    * @param videoFrameData [type:VideoFrameData*]
    * [en]
    * [cn]视频流数据
    * @see VideoFrameData
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnLocalRecordVideoFrameNotify(VideoFrameData * videoFrameData) {};

    /**
    * @brief
    * [en]xxxx
    * [cn]本地录制收到音频流
    *
    * @mobile:disable
    * @win:disable
    *
    * @param audioFrameData [type:AudioFrameData*]
    * [en]
    * [cn]音频流数据
    * @see AudioFrameData
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnLocalRecordAudioFrameNotify(AudioFrameData * audioFrameData) {};

    /**
    * @brief
    * [en]xxxx
    * [cn]本地录制收到数据流，包括白板共享标注
    *
    * @mobile:disable
    * @win:disable
    *
    * @param dataFrameData [type:DataFrameData*]
    * [en]
    * [cn]数据流数据
    * @see DataFrameData
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnLocalRecordDataFrameNotify(VideoFrameData * dataFrameData) {};

    /**
    * @brief
    * [en]xxxx
    * [cn]本地录制收到图片流
    *
    * @mobile:disable
    * @win:disable
    *
    * @param imageFrameData [type:ImageFrameData*]
    * [en]
    * [cn]图片流数据
    * @see ImageFrameData
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnLocalRecordImageFrameNotify(VideoFrameData * imageFrameData) {};

    /**
    * @brief
    * [en]This function is about to info conf end
    * [cn]收到会议即将结束通知(会议结束前五分钟，由服务端计算)
    * @private
    */
    virtual HWM_VOID OnExtendConfNotify(const ConfEndNotifyInfo* confEndNotifyInfo) {};

    /**
    * @brief
    * [en]This function is about leave breakout conf
    * [cn]离开分组会议
    * @private
    */
    virtual HWM_VOID OnEndBreakoutConfNotify() {};

    /**
    * @brief
    * [en]This function is about applications info
    * [cn]收到会议应用通知消息，json字符串
    * @private
    */
    virtual HWM_VOID OnApplicationNotify(const HWM_CHAR*  applicationJsonInfo) {};

    /**
    * @brief
    * [en]This api is about the callback for get hme sample rate.
    * [cn]获取hme的音频采样率
    *
    * @param result [type:SDKERR][cs_allowNull:Y]
    * [en]
    * [cn]接受音频转视频请求操作结果
    * @param reasonDesc [type:HWM_CHAR*][cs_allowNull:Y]
    * [en]
    * [cn]结果的描述信息
    * @param sampleRate [type:HWM_INT32][cs_allowNull:N]
    * [en]
    * [cn]hme音频采样频率
    * @see StartAudioMix
    *
    * @since:v1.0
    *
    * @UTCallback
    */
    virtual HWM_VOID OnStartAudioMixResult(SDKERR result, const HWM_CHAR* reasonDesc, HWM_INT32 sampleRate) {};
};


/**
* @brief [en]This class is about e2ee notify callback.
*        [cn]端到端加密通知回调
*/
class HWM_API IHwmPrivateE2EENotifyCallback
{
public:
    /**
    * @brief
    * [en]
    * [cn]端到端加解密之音频加密回调
    *
    * @param buffer [type:MediaFrameBuffer *]
    * [en]
    * [cn]编码后的音频码流
    *
    * @win:disable
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnE2EEAudioEncryptNotify(MediaFrameBuffer* buffer) {};

    /**
    * @brief
    * [en]
    * [cn]端到端加解密之音频解密回调
    *
    * @param buffer [type:MediaFrameBuffer *]
    * [en]
    * [cn]解码前的音频码流
    *
    * @win:disable
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnE2EEAudioDecryptNotify(MediaFrameBuffer* buffer) {};


    /**
    * @brief
    * [en]
    * [cn]端到端加解密之视频加密回调
    *
    * @param buffer [type:MediaFrameBuffer *]
    * [en]
    * [cn]编码后的视频码流
    *
    * @win:disable
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnE2EEVideoEncryptNotify(MediaFrameBuffer* buffer) {};

    /**
    * @brief
    * [en]
    * [cn]端到端加解密之视频解密回调
    *
    * @param buffer [type:MediaFrameBuffer *]
    * [en]
    * [cn]解码前的视频码流
    *
    * @win:disable
    *
    * @since:v1.0
    */
    virtual HWM_VOID OnE2EEVideoDecryptNotify(MediaFrameBuffer* buffer) {};
};

/**
* @brief [en]This class is about device result callback.
*        [cn]会议设备操作类回调
*/
class HWM_API IHwmPrivateConfDeviceResultCallback
{
public:

};

/**
* @brief [en]This class is about device notify callback.
*        [cn]会议设备通知回调
*/
class HWM_API IHwmPrivateConfDeviceNotifyCallback
{
public:
    /**
    * @brief
    * [en]]This API is about the changes of workspace plugin version.
    * [cn]云桌面插件版本变化信息
    * @mobile:disable
    * @mac:disable
    * @since:v1.0
    */
    virtual HWM_VOID OnHdWorkspacePluginVersionChanged(HdWorkspacePluginVersion* hdWorkspacePluginVersion) {};

    /**
    * @brief
    * [en]]This API is about the changes of Windows Desktop Type.
    * [cn]windows系统的桌面类型变化信息
    * @mobile:disable
    * @mac:disable
    * @since:v1.0
    */
    virtual HWM_VOID OnWinDesktopTypeChanged(WinClientType type) {};
};

/**
* @brief [en]This class is about conference management object.
*        [cn]会议管理对象
* @GetInstance
*        HWM_SDK_NAMESPACE::HwmPrivateNativeSDK::GetPrivateConfMgrApi()
*/
class IHwmPrivateConfMgr
{
public:
    /**
    * @brief
    * [en]This class is about the global callback for the operating results of conference management.
    * [cn]设置会议管理操作类全局回调
    * 
    * @param callback [cs_allowNull:N]
    * [en]
    * [cn]会议管理操作类全局回调
    * 
    * @since:v1.0
    */
    virtual SDKERR SetPrivateConfMgrResultCallback(IHwmPrivateConfMgrResultCallback* callback) = 0;

    /**
    * @brief
    * [en]This class is about the global callback for the notifications of conference management.
    * [cn]设置会议管理通知类全局回调
    * 
    * @param callback [cs_allowNull:N]
    * [en]
    * [cn]会议管理通知类全局回调
    * 
    * @since:v1.0
    */
    virtual SDKERR SetPrivateConfMgrNotifyCallback(IHwmPrivateConfMgrNotifyCallback* callback) = 0;


    /**
    * @brief
    * [en]This class is about booking CTD conference.
    * [cn]预约CTD会议
    * @private
    *
    * @UTInterface:start
    */
    virtual SDKERR BookCtdConf(IN const BookConfParam *bookConfParam) = 0;


    /**
    * @brief
    * [en]This class is about starting QR code pairing.
    * [cn]开始二维码配对
    * @private
    *
    * @UTInterface:start
    */
    virtual SDKERR StartQrCodePair(IN const StartPairParam* pairParam) = 0;

    /**
    * @brief
    * [en]This class is about forcing QR code pairing.
    * [cn]强制二维码配对
    * @private
    *
    * @UTInterface:start
    */
    virtual SDKERR StartQrCodePairForce(IN const StartPairParam* pairParam) = 0;

    /**
    * @brief
    * [en]
    * [cn]验证码入会:匿名入会中间调用接口
    *
    * @param verifyCode [type:HWM_CHAR*][cs_allowNull:N]
    * [en]
    * [cn]验证码
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    *
    * @UTInterface:start
    */
    virtual SDKERR JoinConfByVerifyCode(IN const HWM_CHAR* verifyCode, IN AutoRegisterMode autoRegMode = STATE_DISABLED) = 0;

    /**
    * @brief
    * [en]This class is about ending QR code pairing.
    * [cn]结束二维码配对
    * @private
    */
    virtual SDKERR EndQrCodePair() = 0;

    /**
    * @brief
    * [en]This class is about joining the paring conference.
    * [cn]加入配对会议
    * @private
    *
    * @UTInterface:start
    */
    virtual SDKERR JoinPairConf(IN const HWM_CHAR* confId) = 0;

    /**
    * @brief
    * [en]This class is about the api of verifying the slider and getting verification code for anonymous conferences.
    * [cn]匿名会议 校验滑块并获取验证码接口
    * 
    * @private
    * 
    * @since:v1.0
    *
    * @UTInterface:start
    */
    virtual SDKERR RequestVerifycode(IN HWM_UINT32 sliderPointX, IN HWM_UINT32 costSlideTime) = 0;

    /**
    * @brief
    * [en]This class is about the api of refreshing slider for anonymous conferences.
    * [cn]匿名会议 刷新滑块接口
    * 
    * @private
    * 
    * @since:v1.0
    *
    * @UTInterface:start
    */
    virtual SDKERR RefreshSlider() = 0;

    /**
    * @brief
    * [en]
    * [cn]会议列表直接入会
    *
    * @param joinConfParam [type:JoinConfParam][cs_allowNull:N]
    * [en]
    * [cn]直接入会的参数体
    * @see JoinConfParam
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    *
    * @UTInterface:start
    */
    virtual SDKERR JoinConfDirect(IN const JoinConfParam *joinConfParam) = 0;

    /**
    * @brief
    * [en]
    * [cn]通过会议链接加入会议
    *
    * @param joinConfParam [type:ConfJoinByLinkParam][cs_allowNull:N]
    * [en]
    * [cn]通过会议链接加入会议的参数体
    * @see ConfJoinByLinkParam
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    *
    * @UTInterface:start
    */
    virtual SDKERR JoinConfByLink(IN const ConfJoinByLinkParam *joinConfParam) = 0;

    /**
    * @brief
    * [en]
    * [cn]通过会议random加入会议
    *
    * @param joinConfParam [type:ConfJoinByRandomParam][cs_allowNull:N]
    * [en]
    * [cn]通过会议random加入会议的参数体
    * @see ConfJoinByRandomParam
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    *
    * @UTInterface:start
    */
    virtual SDKERR JoinConfByRandom(IN const ConfJoinByRandomParam *joinConfParam) = 0;

    /**
    * @brief
    * [en]This class is about the api of getting the SMS verification code for anonymous conferences.(SMS :Short Message Service)
    * [cn]匿名会议 获取短信验证码接口
    * 
    * @private
    * 
    * @since:v1.0
    *
    * @UTInterface:start
    */
    virtual SDKERR CheckNeedSliderAuth(IN const HWM_CHAR* phoneNumber, IN const HWM_CHAR* countryCode) = 0;


    /**
    * @brief
    * [en]This class is about getting the pair state.
    * [cn]同步获取会议二维码配对状态
    * @synchronized
    * 
    * @private
    * 
    * @since:v1.0
    */
    virtual ConfPairState GetConfPairState(HWM_VOID) = 0;


    /**
    * @brief
    * [en]
    * [cn]获取会议id的相关信息
    * @synchronized
    * @paramOut
    *
    * @param confId [type:const HWM_CHAR*][cs_allowNull:N]
    * @param confBaseInfo [type:ConfBaseInfo*]
    *
    * @see ConfBaseInfo
    *
    * @private
    *
    * @since:v1.0
    */
    virtual HWM_VOID GetConfBaseInfo(IN const HWM_CHAR* confId, OUT ConfBaseInfo* confBaseInfo) {}

    /**
    * @brief
    * [en]This class is about the api of set upload key log
    * [cn]设置上传关键日志
    * 
    * @private
    * 
    * @UTInterface:disable
    * 
    * @since:v1.0
    */
    virtual SDKERR SetUploadKeyLog(IN HWM_BOOL isUpload) = 0;

    /**
    * @brief
    * [en]This class is about init pair server.
    * [cn]初始化配对服务
    * @private
    * @ios:disable
    */
    virtual SDKERR InitPairServer(IN PairRoleType pairRole, IN HWM_UINT32 maxPairNum, IN HWM_UINT32 codeMode) = 0;

    /**
    * @brief
    * [en]This class is about accept pair request.
    * [cn]是否准许配对
    * @private
    * @ios:disable
    */
    virtual SDKERR AcceptPairRequest(IN const CallPairAcceptInfo* pairAcceptInfo) = 0;

    /**
    * @brief
    * [en]This class is about cancel pair.
    * [cn]是否准许配对
    * @private
    * @ios:disable
    */
    virtual SDKERR CancelPair(IN HWM_INT64 pairId, IN CallPairReasonType reasonCode) = 0;

    /**
    * @brief
    * [en]This class is about pair send join conf info.
    * [cn]发送入会结果
    * @private
    * @ios:disable
    */
    virtual SDKERR PairSendJoinconfInfo(IN HWM_INT64 pairId, IN HWM_UINT32 joinResult) = 0;

    /**
    * @brief
    * [en]This class is about uninit pair server.
    * [cn]配对去初始化
    * @private
    * @ios:disable
    */
    virtual SDKERR UninitPairServer() = 0;

    /**
    * @brief
    * [en]This class is query history conference list.
    * [cn]查询历史会议列表，参数时间为毫秒
    * @private
    *
    * @UTInterface:start
    */
    virtual SDKERR QueryHistoryConflist(IN HWM_UINT64 beginTime, IN HWM_UINT64 endTime) = 0;

    /**
    * @brief
    * [en]This class is about query history conference attendee infomation.
    * [cn]查询历史会议与会者信息
    * @private
    *
    * @UTInterface:start
    */
    virtual SDKERR QueryHistoryConfAttendee(IN const HWM_CHAR* confUuid) = 0;

    /**
    * @brief
    * [en]This class is about query history conference record file.
    * [cn]查询录制文件
    * @private
    * @mobile:disable
    *
    * @UTInterface:start
    */
    virtual SDKERR QueryHistoryConfRecordFile(IN const HWM_CHAR* confUuid) = 0;

    /**
    * @brief
    * [en]This class is about query local history conference list.
    * [cn]查询本地历史会议列表，参数时间为毫秒
    * @private
    *
    * @UTInterface:start
    */
    virtual SDKERR QueryLocalHistoryConflist(IN HWM_UINT64 beginTime, IN HWM_UINT64 endTime) = 0;

    /**
    * @brief
    * [en]This class is about join conf by nonce.
    * [cn]根据随机数加入会议
    * @private
    *
    * @UTInterface:start
    */
    virtual SDKERR JoinConfByNonce(IN const NonceJoinConfParam* joinConfParam) = 0;

    /**
    * @brief
    * [en]
    * [cn]根据会议Id和密码获取会议的基本信息
    *
    * @param confId  password
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    *
    * @UTInterface:start
    */
    virtual SDKERR QueryBasicConfInfo(IN const HWM_CHAR* confId, IN const HWM_CHAR* password) = 0;

    /**
    * @brief
    * [en]
    * [cn]修改历史会议中数据库的本地录制文件路径
    * @private
    * @mobile:disable
    * @win:disable
    * @param historyConfInfo
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    */
    virtual SDKERR UpdateHistoryConfInfo(IN HistoryConfInfoParam* historyConfInfo) = 0;

    /**
    * @brief
    * [en]
    * [cn]获取
    *
    * @param nonceType [type:NonceType]
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    */
    virtual SDKERR QueryNonceByType(NonceType nonceType) { return SDKERR_SUCCESS; };


    /**
    * @brief
    * [en]
    * [cn]查询会议文档
    *
    * @param queryFileListParam [type:QueryFileListsParam][cs_allowNull:N]
    * [en]
    * [cn]查询会议文档参数
    * @see QueryFileListsParam
    *
    * @return [type:SDKERR]
    *
    * @android:disable
    * @mac:disable
    * @UTInterface:start
    */
    virtual SDKERR QueryDocFileLists(IN const QueryFileListsParam* queryFileListParam) { return SDKERR_SUCCESS; };

    /**
    * @brief
    * [en]
    * [cn]删除会议文档
    *
    * @param deleteFileParam [type:DeleteFileParam][cs_allowNull:N]
    * [en]
    *
    * @return [type:SDKERR]
    *
    * @android:disable
    * @mac:disable
    * @UTInterface:start
    */
    virtual SDKERR DeleteDocument(IN const DeleteFileParam* deleteFileParam) { return SDKERR_SUCCESS; };

    /**
    * @brief
    * [en]
    * [cn]获取协作权限
    *
    * @param applyDocOperParam [type:ApplyDocOperRightParam][cs_allowNull:N]
    * [en]
    *
    * @return [type:SDKERR]
    * @android:disable
    * @mac:disable
    *
    * @UTInterface:start
    */
    virtual SDKERR ApplyDocOperateRight(IN const ApplyDocOperRightParam* applyDocOperParam) { return SDKERR_SUCCESS; };

    /**
    * @brief
    * [en]
    * [cn]会中创建文档
    *
    * @param createFolderParam [type:CreateFolderParam][cs_allowNull:N]
    * [en]
    *
    * @return [type:SDKERR]
    *
    * @android:disable
    * @mac:disable
    * @UTInterface:start
    */
    virtual SDKERR CreateFolder(IN const CreateFolderParam* createFolderParam) { return SDKERR_SUCCESS; };

    /**
    * @brief
    * [en]
    * [cn]查询文档权限链接
    *
    * @param QueryDocAccessUrlParam [type:QueryDocAccessUrlParam][cs_allowNull:N]
    * [en]
    *
    * @return [type:SDKERR]
    *
    * @pc:disable
    * @android:disable
    * @mac:disable
    * @UTInterface:start
    */
    virtual SDKERR QueryDocAccessUrl(IN const QueryDocAccessUrlParam* queryDocAccessUrlParam) { return SDKERR_SUCCESS; };
};


/**
* @brief [en]This class is about the object of conference control operation.
*        [cn]会议控制操作对象
* @GetInstance
*        HWM_SDK_NAMESPACE::HwmPrivateNativeSDK::GetPrivateConfCtrlApi()
*/
class IHwmPrivateConfCtrl
{
public:

    /**
    * @brief
    * [en]This class is about the global callback for the operating results of setting the conference control.
    * [cn]设置会议控制操作类全局回调
    * 
    * @param callback [type:IHwmConfCtrlResultCallback][cs_allowNull:N]
    * [en]
    * [cn]会议控制操作类全局回调
    * @see IHwmConfCtrlResultCallback
    * 
    * @since:v1.0
    */
    virtual SDKERR SetPrivateConfCtrlResultCallback(IHwmPrivateConfCtrlResultCallback* callback) = 0;

    /**
    * @brief
    * [en]This class is about the global callback for the notifications of conference control.
    * [cn]设置会议控制通知类全局回调
    * 
    * @param callback [type:IHwmConfCtrlNotifyCallback][cs_allowNull:N]
    * [en]
    * [cn]会议控制通知类全局回调
    * @see IHwmConfCtrlNotifyCallback
    * 
    * @since:v1.0
    */
    virtual SDKERR SetPrivateConfCtrlNotifyCallback(IHwmPrivateConfCtrlNotifyCallback* callback) = 0;


    /**
    * @brief
    * [en]This class is about invite hardterminal user by scan QR Code.
    * [cn]扫码邀请硬终端用户入会
    * @private
    *
    * @UTInterface:start
    */
    virtual SDKERR InviteHardTerminalByScanQRCode(IN const InviteHardTerminalParam* inviteHardTerminalParam) = 0;



    /**
    * @brief
    * [en]This class is about dealing with the result notification of AI conference record reported by TUP.
    * [cn]处理TUP上报的AI会议纪要结果通知
    * @private
    *
    * @UTInterface:start
    */
    virtual SDKERR AiConfRecord(IN HWM_BOOL isOpen) = 0;

    /**
    * @brief
    * [en]This class is about setting chat permission.
    * [cn]主席会控：控制聊天权限
    *
    * @param chatPermission [type:ChatPermission][cs_allowNull:N]
    *
    * @state:inConf
    *
    * [en]
    * [cn]聊天权限
    *
    * @return [type:SDKERR]
    *
    */
    virtual SDKERR SetChatPermission(IN ChatPermission chatPermission) { return SDKERR_SUCCESS; };

    /**
    * @brief
    * [en]This class is about data share window
    * [cn]数据会议共享界面观看情况
    * @param isWatching [type:HWM_BOOL]
    * [en] indicate user is watching data share window or not
    * [cn] 是否在看数据会议共享界面
    * @param resolutionLevel [type:GeneralWatchResolutionLevel]
    * [en] indicate resolution level of data share window
    * [cn]选看数据会议窗口分辨率
    * @see GeneralWatchResolutionLevel
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    */
    virtual SDKERR SetDataShareWatchParam(IN HWM_BOOL isWatching, IN GeneralWatchResolutionLevel resolutionLevel) = 0;

    /**
    * @brief
    * [en]This class is about extending conf duration.
    * [cn]设置会议延长时间
    *
    * @param type [type:ExtendConfType][cs_allowNull:N]
    * @param duration [type:HWM_UINT32][cs_allowNull:N]
    * [en]
    * [cn]手动自动设置会议延长时间
    *
    * @return [type:SDKERR]
    *
    * @UTInterface:start
    */
    virtual SDKERR SetExtendConfDuration(IN HWM_UINT32 duration) { return SDKERR_SUCCESS; };

    /**
    * @brief
    * [en]This class is about setting LabelPermission when sharing or as a host.
    * [cn]控制共享标注权限
    *
    * @param permissionType [type:AnnotationPermission][cs_allowNull:N]
    * [en]
    * [cn]控制共享标注权限
    *
    * @return [type:SDKERR]
    *
    * @UTInterface:start
    */
    virtual SDKERR SetAnnotationPermission(IN AnnotationPermission permissionType, IN HWM_BOOL isConfLevel) { return SDKERR_SUCCESS; };

    /**
    * @brief
    * [en]This class is about setting DevicePermission when in meeting.
    * [cn]控制设备权限
    *
    * @param permissionType [type:DevicePermission][cs_allowNull:N]
    * [en]
    * [cn]控制设备权限
    *
    * @return [type:SDKERR]
    *
    * @pc:disable
    */
    virtual SDKERR SetDevicePermission(IN DevicePermission permissionType) { return SDKERR_SUCCESS; };

    /**
    * @brief
    * [en]This class is about get conf extend config.
    * [cn]获取会议扩展配置
    * @synchronized
    * @paramOut
    * 
    * @param pConfExtendConfig [type:ConfExtendConfig][cs_allowNull:N]
    * [en]
    * [cn]会议扩展配置
    *
    * @return [type:SDKERR]
    *
    * @pc:disable
    */
    virtual HWM_VOID GetConfExtendConfig(OUT ConfExtendConfig *pConfExtendConfig) = 0;

    /**
    * @brief
    * [en]
    * [cn]设置多人共享类型
    * @private
    */
    virtual SDKERR SetMultiShareType(IN MultiShareType type) = 0;

    /**
    * @brief
    * [en]
    * [cn]离会清空数据，依赖UI清理
    * @private
    */
    virtual SDKERR ResetConfData() { return SDKERR_SUCCESS; };

};


/**
* @brief [en]This class is about the object of conference state infomation.
*        [cn]会议状态信息对象
* @GetInstance
*        HWM_SDK_NAMESPACE::HwmPrivateNativeSDK::GetPrivateConfStateApi()
*/
class IHwmPrivateConfState
{
public:
    /**
    * @brief
    * [en]This class is about the notifications callback of conference state.
    * [cn]设置会议状态通知类全局回调
    * 
    * @param callback [type:IHwmPrivateConfStateNotifyCallback][cs_allowNull:N]
    * [en]
    * [cn]回调方法
    * @see IHwmPrivateConfStateNotifyCallback
    * 
    * @since:v1.0
    */
    virtual SDKERR SetPrivateConfStateNotifyCallback(IHwmPrivateConfStateNotifyCallback* callback) = 0;

    

    /**
    * @brief
    * [en]This class is about getting the conference state: whether the conference supports the AI recording.
    * [cn]获取会议状态:是否支持AI录制
    * @synchronized
    * @private
    */
    virtual HWM_BOOL GetConfSupportAiRecord() = 0;

    /**
    * @brief
    * [en]This class is about getting the conference state: whether the conference supports set Annotation permission.
    * [cn]获取会议状态:是否支持共享标志权限控制
    * @synchronized
    * @private
    */
    virtual HWM_BOOL GetSupportAnnotationPermission() { return HWM_FALSE; };

    /**
    * @brief
    * [en]This class is about getting the conference state: Annotation permission.
    * [cn]获取会议状态:获取共享标注权限
    * @synchronized
    * @private
    */
    virtual AnnotationPermission GetAnnotationPermission(HWM_BOOL isConfLevel) { return ANNOTATION_PERMISSION_ALL; };

    /**
    * @brief
    * [en]This class is about getting the conference media flow mode.
    * [cn]获取会议状态:获取会议媒体流模式
    * @mobile:disable
    * @mac:disable
    * @synchronized
    * @private
    */
    virtual ConfMediaFlowMode GetConfMediaFlowMode() { return CONF_MEDIA_FLOW_MODE_COROUTE; };


    /**
    * @brief
    * [en]This class is about getting the conference state: force open camera conf show state.
    * [cn]获取会议状态:强制视频会议显示状态
    * @synchronized
    * @private
    */
    virtual OnlyOpenCameraShowState GetOnlyOpenCameraShowState() { return ONLY_OPEN_CAMERA_NOT; };

};


/**
* @brief [en]This class is about getting the object of device management.
*        [cn]获取设备管理对象
* @GetInstance
*        HWM_SDK_NAMESPACE::HwmPrivateNativeSDK::GetPrivateDeviceMgrApi()
*/
class IHwmPrivateDeviceMgr
{
public:
    /**
    * @brief
    * [en]This class is about setting the callback class of device operation APIs.
    * [cn]设置会议设备操作类全局回调
    *
    * @param callback [type:IHwmPrivateConfDeviceResultCallback*][cs_allowNull:N]
    * [en]
    * [cn] 会议设备操作类回调
    * @see IHwmPrivateConfDeviceResultCallback
    *
    * @since:v1.0
    */
    virtual SDKERR SetPrivateConfDeviceResultCallback(IHwmPrivateConfDeviceResultCallback* callback) = 0;

    /**
    * @brief
    * [en]This class is about setting the callback class of device notifications.
    * [cn]设置会议设备操作类全局回调
    *
    * @param callback [type:IHwmPrivateConfDeviceNotifyCallback*][cs_allowNull:N]
    * [en]
    * [cn] 会议设备通知回调
    * @see IHwmPrivateConfDeviceNotifyCallback
    *
    * @since:v1.0
    */
    virtual SDKERR SetPrivateConfDeviceNotifyCallback(IHwmPrivateConfDeviceNotifyCallback* callback) = 0;

    /**
    * @brief
    * [en]This class is about setting the callback class of e2ee notifications.
    * [cn]设置端到端加密全局回调
    *
    * @param callback [type:IHwmPrivateE2EENotifyCallback*][cs_allowNull:N]
    * [en]
    * [cn] 端到端加密全局回调
    * @see IHwmPrivateE2EENotifyCallback
    *
    * @since:v1.0
    */
    virtual SDKERR SetPrivateE2EENotifyCallback(IHwmPrivateE2EENotifyCallback* callback) = 0;

    /**
    * @brief
    * [en]This api is about video control.
    * [cn]视频控制(这个接口有点难理解，需讨论优化成易理解使用的接口)
    *
    * @param videoControl [type:VideoControlParam][cs_allowNull:N]
    * [en]
    * [cn]视频控制参数
    * @see VideoControlParam
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    *
    * @private
    */
    virtual SDKERR VideoControl(IN const VideoControlParam* videoControl) = 0;

    /**
    * @brief
    * [en]This api is about opening or closing the automatic routing of audio.
    * [cn]开启或关闭音频底层自动路由
    *
    * @param isOpen [type:HWM_BOOL][cs_allowNull:N]
    * [en]
    * [cn]是否开启音频底层自动路由，isOpen-True:开启，isOpen-False:不开启
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    *
    */
    virtual SDKERR SwitchAudioAutoRouter(IN HWM_BOOL isOpen) = 0;

    /**
    * @brief
    * [en]This api is about the enable third-party data acquisition .
    * [cn]启用第三方视频数据采集
    *
    * @param frame [type:VideoFrameParam][cs_allowNull:N]
    * [en]
    * [cn]是否启用
    * @param enable [type:HWM_BOOL][cs_allowNull:N]
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    *
    * @pc:disable
    * @ios:disable
    */
    virtual SDKERR EnableExternalVideoCapture(IN HWM_BOOL enable) = 0;
    /**
    * @brief
    * [en]This api is about the third-party data acquisition .
    * [cn]第三方视频数据采集
    *
    * @param frame [type:VideoFrameParam][cs_allowNull:N]
    * [en]
    * [cn]视频帧参数
    * @see VideoFrameParam
    * @param data [type:HWM_UCHAR][cs_allowNull:N]
    * [en]
    * [cn]存储视频数据的data地址
    * @param dataLen [type:HWM_UINT32][cs_allowNull:N]
    * [en]
    * [cn]视频数据的长度
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    *
    * @pc:disable
    * @ios:disable
    */
    virtual SDKERR SetVideoCaptureInput(IN VideoFrameParam* frame, IN HWM_UCHAR *data, IN HWM_UINT32 dataLen) = 0;

    /**
    * @brief
    * [en]This api is about operating audio preview window.
    * [cn]操作音频预览窗口
    * @private
    * @mobile:disable
    */
    virtual SDKERR OperateAudioPreview(HWM_BOOL isOpen) = 0;

    /**
    * @brief
    * [en]This api is about setting the folding state of device interface.
    * [cn]设置设备界面的折叠状态
    *
    * @param deviceFoldedStateType [type:DeviceFoldedStateType][cs_allowNull:N]
    * [en]
    * [cn] 设备折叠状态类型
    * @see DeviceFoldedStateType
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    */
    virtual SDKERR SetDeviceFoldedState(IN const DeviceFoldedStateType deviceFoldedStateType) = 0;


    /**
    * @brief
    * [en]This api is used to remove all video window for ios.
    * [cn]ios移除所有的窗口句柄
    * @private
    * @pc:disable
    * @android:disable
    */
    virtual SDKERR RemoveAllVideoWindow() = 0;

    /**
    * @brief
    * [en]This api is about setting video resolution mode.
    * [cn]设置视频分辨率模式
    *
    * @param videoResolutionMode [type:VideoResolutionMode][cs_allowNull:N]
    * [en]
    * [cn] 视频分辨率类型
    * @see VideoResolutionMode
    *
    * @return [type:SDKERR]
    *
    * @UTInterface:disable
    * 
    * @since:v1.0
    */
    virtual SDKERR SetVideoResolutionMode(IN const VideoResolutionMode videoResolutionMode) = 0;

    /**
    * @brief
    * [en]This api is about setting AI model param.
    * [cn]设置AI模型参数
    *
    * @param modelParam [type:AiModelParam*][cs_allowNull:N]
    * [en]
    * [cn]AI模型参数
    * @see AiModelParam
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    */
    virtual SDKERR SetAiModelParam(IN const AiModelParam* modelParam) = 0;

    /**
    * @brief
    * [en]This api is about opening or closing the encryption of audio stream chip.
    * [cn]开启或关闭音频流芯片加密
    *
    * @param isOpen [type:HWM_BOOL][cs_allowNull:N]
    * [en]
    * [cn]是否开启音频流芯片加密，isOpen-True:开启，isOpen-False:不开启
    *
    * @return [type:SDKERR]
    * 
    * @UTInterface:disable
    *
    * @since:v1.0
    *
    */
    virtual SDKERR SwitchAudioChipEncrypt(IN HWM_BOOL isOpen) = 0;

    /**
    * @brief
    * [en]This api is about setting the virtual background.
    * [cn]设置虚拟背景
    *
    * @param virtualBackgroundInfo [type:VirtualBackgroundInfo][cs_allowNull:N]
    * [en]
    * [cn]设置虚拟背景参数
    * @see VirtualBackgroundInfo
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    *
    */
    virtual SDKERR SetVirtualBackground(const VirtualBackgroundInfo* virtualBackgroundInfo) = 0;

    /**
    * @brief
    * [en]This api is about setting the virtual background memory.
    * [cn]设置虚拟背景位图内存
    *
    * @param virtualBackgroundInfo [type:VirtualBackgroundInfo][cs_allowNull:N]
    * [en]
    * [cn]设置虚拟背景参数
    * @see VirtualBackgroundInfo
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    *
    */
    virtual SDKERR SetVirtualBackgroundMemory(VirtualBackgroundMode mode, HWM_UCHAR* imageBuffer, HWM_UINT32 width, HWM_UINT32 height) = 0;

    /**
    * @brief
    * [en]This api is about getting AI algorithms capability.
    * [cn]获取AI算法能力
    *
    * @param NULL [cs_allowNull:Y]
    *
    * @return [type:AICapability]
    * [en]
    * [cn]AI算法能力
    * @AICapability
    *
    * @since:v1.0
    *
    * @synchronized
    */
    virtual AICapability GetAICapability() = 0;

    /**
    * @brief
    * [en]This api is to get Windows desktop type.
    * [cn]获取Windows桌面类型
    *
    * @param NULL [cs_allowNull:Y]
    *
    * @return [type:WinClientType]
    * [en]Windows desktop type
    * [cn]Windows桌面类型
    * @WinClientType
    *
    * @since:v1.0
    * @mobile:disable
    * @mac:disable
    * @synchronized
    */
    virtual WinClientType GetWinClientType() = 0;


    /**
    * @brief
    * [en]This api is to get HdWorkspace Plugin Version.
    * [cn]获取云桌面插件版本信息
    *
    * @param NULL [cs_allowNull:Y]
    *
    * @return [type:HdWorkspacePluginVersion]
    * [en]plugin version
    * [cn]插件版本信息
    * @HdWorkspacePluginVersion
    *
    * @since:v1.0
    * @mobile:disable
    * @mac:disable
    * @synchronized
    */
    virtual HdWorkspacePluginVersion GetHdWorkspacePluginVersion() = 0;

    /**
    * @brief
    * [en]This api is about getting hme backgroud process result.
    * [cn]获取hme背景虚化或者背景替换的结果
    *
    * @param NULL [cs_allowNull:Y]
    *
    * @return [type:BgTypeErr]
    * [en]
    * [cn]hme背景虚化或者背景替换的结果
    * @BackgroudProgressResult
    *
    * @since:v1.0
    *
    * @synchronized
    * @mobile:disable
    * @mac:disable
    */
    virtual BgTypeErr GetBackgroudProgressResult() = 0;


    /**
    * @brief
    * [en]This api is about whether the super resolution is supported.
    * [cn]手机是否提供视频超级分辨率功能
    *
    * @param NULL [cs_allowNull:Y]
    *
    * @return [type:HWM_BOOL]
    * [en]
    * [cn] true-手机提供视频超级分辨率功能，false-手机不提供视频超级分辨率功能
    *
    * @since:v1.0
    */
    virtual HWM_BOOL GetIsEnableSuperResolution() = 0;


    /**
    * @brief
    * [en]This api is used to setting video window.
    * [cn]设置视频窗口
    * 
    * @UTInterface:disable
    * 
    * @private
    */
    virtual SDKERR SetVideoWindow(IN HWM_UINT32 wndCount, IN const VideoWndBasicInfo* videoWndInfo) = 0;

    /**
    * @brief
    * [en]This api is used to setting attributes of video display window.
    * [cn]设置视频显示窗口属性----废弃掉
    * @private
    * @mobile:disable
    */
    virtual SDKERR SetVideoRender(IN const VideoRenderInfo* videoRenderInfo) = 0;

    /**
    * @brief
    * [en]This api is used to setting attributes of video display window.
    * [cn]设置视频显示--远端窗口属性(render字段, 此版本不生效)
    * @private
    */
    virtual SDKERR SetRemoteVideoRender(IN const RemoteVideoRenderInfo* videoRenderInfo) = 0;

    /**
    * @brief
    * [en]This api is used to setting attributes of video display window.
    * [cn]设置窗口渲染模式
    * @private
    */
    virtual SDKERR SetVideoDisplayMode(IN const VideoRenderDisplayModeInfo* displayModeInfo) = 0;

    /**
    * @brief
    * [en]This api is used to setting extended attributes of video display window.
    * [cn]设置视频显示窗口扩展属性
    * @private
    * @mobile:disable
    */
    virtual SDKERR SetVideoRenderEx(IN const VideoRenderInfo* videoRenderInfo) = 0;

    /**
    * @brief
    * [en]This api is about setting terminal type
    * [cn]设置终端类型
    * @private
    */
    virtual SDKERR SetTerminalType(IN const HWM_CHAR* terminalType) = 0;

    /**
    * @brief
    * [en]This api is about setting user setting info
    * [cn]用户设置相关信息
    * @private
    * @mobile:disable
    */
    virtual SDKERR SetUserSettingInfo(IN const UserSettingInfo* settingParam) = 0;

    /**
    * @brief
    * [en]This api is about media open audio preview
    * [cn]开始音频预览
    * @private
    * @mobile:disable
    * @paramOut
    */
    virtual SDKERR OpenAudioPreview(OUT HWM_UINT32* sessionId) = 0;

    /**
    * @brief
    * [en]This api is about media stop audio preview
    * [cn]停止音频预览
    * @private
    * @mobile:disable
    */
    virtual SDKERR StopAudioPreview(IN HWM_UINT32 sessionId) = 0;

    /**
    * @brief
    * [en]This api is about media get device volume
    * [cn]获取设备音量
    * @private
    * @mobile:disable
    * @synchronized
    * @paramOut
    */
    virtual SDKERR GetDeviceVolume(IN const DeviceType deviceType, OUT HWM_UINT32* volume) = 0;

    /**
    * @brief
    * [en]This api is about media set device volume
    * [cn]设置设备音量
    * @private
    * @mobile:disable
    */
    virtual SDKERR SetDeviceVolume(IN const DeviceType deviceType, IN HWM_UINT32 volume) = 0;

    /**
    * @brief
    * [en]This api is about media set mic auto volume
    * [cn]设置是否自动调节麦克风音量
    * @private
    * @mobile:disable
    */
    virtual SDKERR SetMicAutoVolume(IN HWM_BOOL isOpen) = 0;

    /**
    * @brief
    * [en]This api is about set high resolution
    * [cn]设置1080P
    * @private
    * @mobile:disable
    */
    virtual SDKERR SetHighResolution(IN HWM_BOOL enableHighResolution) = 0;

    /**
    * @brief
    * [en]This api is about set high resolution priority
    * [cn]设置高清优先
    * @private
    * @mobile:disable
    */
    virtual SDKERR SetHighResolutionPriority(IN HWM_BOOL enableHighResolutionPriority) = 0;

    /**
    * @brief
    * [en]This api is about set high resolution svc
    * [cn]开启高清多流模式
    * @private
    * @mobile:disable
    */
    virtual SDKERR EnableHighResolutionSvcMode(IN HWM_BOOL enableHighResolutionSvcMode) = 0;

    /**
    * @brief
    * [en]This api is about enable conf high resolution
    * [cn]开启会议级1080P
    * @private
    * @mobile:disable
    */
    virtual SDKERR EnableConfHighResolution(IN HWM_BOOL enable) = 0;

    /**
    * @brief
    * [en]This api is about set up audio tnr
    * [cn]设置瞬时噪音抑制开关
    * @private
    */
    virtual SDKERR SetUpAudioTnr(IN HWM_BOOL enable) = 0;

    /**
    * @brief
    * [en]This api is about set up audio tnr level
    * [cn]设置瞬时噪音抑制挡位
    * @private
    * @mobile:disable
    */
    virtual SDKERR SetUpAudioTnrLevel(IN UpAudioTnrLevel level) = 0;

    /**
    * @brief
    * [en]This api is about set up active origin sound
    * [cn]设置启用原始声音
    * @private
    * @ios:disable
    */
    virtual SDKERR SetActiveOriginSound(IN HWM_BOOL enable) = 0;

    /**
    * @brief
    * [en]This api is about set up active origin sound params
    * [cn]启用/关闭原始声音相关参数
    * @private
    * @mobile:disable
    * @mac:disable
    */
    virtual SDKERR SetActiveOriginSoundParams(IN const OriginSoundParams* params) { return SDKERR_SUCCESS; };

    /**
    * @brief
    * [en]This api is about set windows board audio optimize
    * [cn]windows大屏音频优化
    * @private
    * @mobile:disable
    * @mac:disable
    */
    virtual SDKERR SetWinBoardAudioOptimize(IN HWM_BOOL enable) = 0;

    /**
    * @brief
    * [en]This api is about start audio mix.
    * [cn]开始混音
    * @private
    * @pc:disable
    *
    * @UTInterface:start
    */
    virtual SDKERR StartAudioMix() = 0;

    /**
    * @brief
    * [en]This api is about stop audio mix.
    * [cn]停止混音
    * @private
    * @pc:disable
    */
    virtual SDKERR StopAudioMix() = 0;

    /**
    * @brief
    * [en]This api is about send share audio mix data.
    * [cn]发送共享的音频混音数据
    * @param buf [type:HWM_UCHAR][cs_allowNull:N]
    * [en]
    * [cn]存储音频数据的data地址
    * @param buffLen [type:HWM_UINT32][cs_allowNull:N]
    * [en]
    * [cn]音频数据的长度（需要是show int 的整数倍）
    *
    * @return [type:SDKERR]
    *
    * @since:v1.0
    * 
    * @UTInterface:disable
    *
    * @private
    * @pc:disable
    */
    virtual SDKERR SendShareAudioMixData(IN HWM_UCHAR *buf, IN HWM_UINT32 buffLen) = 0;

    /**
    * @brief
    * [en]This api is about start or stop send audio mix data.
    * [cn]开启/关闭共享声音推流
    * @private
    * @pc:disable
    *
    * @UTInterface:start
    */
    virtual SDKERR EnableSendShareAudioMixData(IN HWM_BOOL enable) { return SDKERR_SUCCESS; };

    /**
    * @brief
    * [en]This api is about set cpu level
    * [cn]设置cpu等级
    * @private
    */
    virtual HWM_VOID SetCpuLevel(CpuLevel level) = 0;

    /**
    * @brief
    * [en]This api is about set cpu usage
    * [cn]设置cpu使用率
    * @private
    * @mobile:disable
    * 
    * @UTInterface:disable
    */
    virtual HWM_VOID SetCpuUsage(HWM_UINT32 value) = 0;

    /**
    * @brief
    * [en]This api is about set cpu threshold
    * [cn]设置cpu阈值
    * @private
    * @mobile:disable
    */
    virtual HWM_VOID SetCpuThreshold(HWM_UINT32 above, HWM_UINT32 belowSlow, HWM_UINT32 belowQuick) = 0;

    /**
    * @brief
    * [en]This api is about set super resolution model path.
    * [cn]配置超分模型路径
    * @private
    * @pc:disable
    * @ios:disable
    */
    virtual SDKERR SetSuperResolutionModelPath(IN const HWM_CHAR*modelPath) = 0;

    /**
    * @brief
    * [en]This api is about set wake up sdk model path.
    * [cn]配置唤醒SDK模型路径
    * @private
    * @pc:disable
    * @ios:disable
    */
    virtual SDKERR SetWakeupSDKModelPath(IN const HWM_CHAR*modelPath) { return SDKERR_SUCCESS; };

    /**
    * @brief
    * [en]This api is about set hme virtual background path.
    * [cn]配置hme 虚拟背景模型路径
    * @param paths [type:VirtualBackgroundModelPaths][cs_allowNull:N]
    * @see VirtualBackgroundModelPaths
    * 
    * @since:v1.0
    */
    virtual SDKERR SetVirtualBackGroundModelPath(IN VirtualBackgroundModelPaths* paths) { return SDKERR_SUCCESS; };

    /**
    * @brief
    * [en]xxx
    * [cn]配置视频质量评估模型路径
    * @private
    * @mac:disable
    * @mobile:disable
    */
    virtual SDKERR SetVideoQualityAssessmentModelPath(IN const HWM_CHAR *modelPath) = 0;

    /**
    * @brief
    * [en]This class is about setting the VOIP voice data encrypt function and decrype function.
    * [cn]设置VOIP语言数据加密解密回调函数。
    *
    * @param voipEncDec [type:VoipEncDec*][cs_allowNull:N]
    * [en]
    * [cn] 加密解密回调函数地址
    * @see SetVoipEncDecWithIOS
    * @android:disable
    * @pc:disable
    * @mac:disable
    *
    * @since:v1.0
    */
    virtual SDKERR SetVoipEncDecWithIOS(VoipEncDec* voipEncDec) { return SDKERR_SUCCESS; };

    /**
    * @brief
    * [en]This api is about enable debug mode hook frames.
    * [cn]启用Debug模式Hook码流。
    *
    * @android:disable
    * @ios:disable
    * @mac:disable
    *
    * @since:v1.0
    */
    virtual SDKERR EnableHookFrames(IN const HWM_CHAR * params) { return SDKERR_SUCCESS; }

    /**
    * @brief
    * [en]This api is about set sample trace file path
    * [cn]设置音频trace临时文件
    * 
    * @param filePath [type:HWM_CHAR*][cs_allowNull:N]
    *
    * @android:disable
    * @ios:disable
    * @mac:disable
    * @win:disable
    *
    * @since:v1.0
    */
    virtual HWM_VOID SetSampleTraceFilePath(IN HWM_CHAR* filePath) = 0;

    /**
    * @brief
    * [en]This api is about enable save record frames zip file.
    * [cn]是否启用hook码流压缩文件。
    *
    * @android:disable
    * @ios:disable
    * @mac:disable
    *
    * @since:v1.0
    */
    virtual SDKERR EnableSaveTempAudioRecordFile(IN HWM_BOOL enable, IN RecordAudioLevel recordAudioLevel) { return SDKERR_SUCCESS; }

    /**
    * @brief
    * [en]This api is about enable e2ee.
    * [cn]启用/关闭端到端加密。
    *
    * @win:disable
    *
    * @since:v1.0
    */
    virtual SDKERR EnableE2EE(IN HWM_BOOL enable) = 0;


    /**
    * @brief
    * [en]This api is about set e2ee config.
    * [cn]e2ee配置，给win使用，win进程间通讯，不利于码流传输，使用动态加载dll方式在sdk线程处理。
    *
    * @android:disable
    * @ios:disable
    * @mac:disable
    *
    * @since:v1.0
    */
    virtual SDKERR SetE2EEConfig(IN const E2EEConfig * params) = 0;

    /**
    * @brief
    * [en]This api is used to set the audio routing mode.
    * [cn]设置音频路由模式
    * @private
    * @win:disable
    * @mac:disable
    */
    virtual HWM_VOID SetAudioRouteMode(AudioRouteType mode) = 0;

    /**
    * @brief
    * [en]This api is about set video preview.
    * [cn]设置视频预览
    * @private
    * @win:disable
    * @mac:disable
    * @ios:disable
    */
    virtual HWM_VOID SetVideoPreview(HWM_UPTR view) = 0;

    /**
    * @brief
    * [en]This api is about set video preview.
    * [cn]共享摄像头预览, view为0时, 停止预览; view不为0时, 开始预览
    * @param view [type:HWM_UPTR][cs_allowNull:N]
    * @param index [type:HWM_UINT32][cs_allowNull:N]
    * @private
    * @android:disable
    * @mac:disable
    * @ios:disable
    */
    virtual HWM_VOID SetCameraSharePreview(HWM_UPTR view, HWM_UINT32 index) {};

    /**
    * @brief
    * [en]This api is about setting video quality mode.
    * [cn]设置视频最小分辨率
    *
    * @param videoQualityMode [type:VideoQualityMode][cs_allowNull:N]
    * [en]
    * [cn] 最小分辨率类型
    * @see VideoQualityMode
    *
    * @return [type:SDKERR]
    *
    * @UTInterface:disable
    * 
    * @since:v1.0
    */
    virtual SDKERR SetVideoQualityMode(IN const VideoQualityMode videoQualityMode) = 0;

    /**
    * @brief
    * [en]
    * [cn]设置音频是否开启低时延
    * @private
    * @mobile:disable
    * @mac:disable
    */
    virtual SDKERR SetAudioLowLatency(IN HWM_BOOL bEnable) = 0;

    /**
    * @brief
    * [en]
    * [cn] 设置窗口总宽高, 当前仅支持设置本端窗口宽高.
    * @private
    * @android:disable
    * @ios:disable
    * @mac:disable
    * @win:disable
    */
    virtual HWM_VOID SetWindowResolution(const HwmWindowResolutionParam *wndResolutionParam) = 0;

    /**
    * @brief
    * [en]
    * [cn]设置agc最大增益
    * @private
    * @mobile:disable
    * @mac:disable
    */
    virtual SDKERR SetAudioAgcMaxGain(IN HWM_UINT32 uAgcMaxGain) { return SDKERR_SUCCESS; }

    /**
    * @brief
    * [en]This api is about set super resolution model path.
    * [cn]开启或关闭视频或新辅流超分功能
    * @param mediaType 0:主流，2:辅流
    * 
    * @private
    * @pc:disable
    * @ios:disable
    * @android:disable
    */
    virtual SDKERR EnableSuperResolution(IN HWM_BOOL enable, IN HWM_UINT32 mediaType) { return SDKERR_SUCCESS; }
    
    /**
    * @brief
    * [en]get video preview state
    * [cn]获取主流视频的预览状态
    * @private
    * @mobile:disable
    */
    virtual HWM_BOOL GetVideoPreviewState() { return HWM_FALSE; }
};


END_HWM_SDK_NAMESPACE
#endif /* __HWM_CONF_SERVICE_H__ */