//
//  HWMTextView.m
//  HWMUISDKNativeDemo
//
//  Created by z30074171 on 2025/2/8.
//  Copyright © 2025 Huawei Technologies Co. Ltd. All rights reserved.
//

#import "HWMTextView.h"
#define DEFAULT_FONT_SIZE (14)
#define MAX_COUNT (3000)

@interface HWMTextView()<NSTextViewDelegate>
@property (nonatomic, strong) NSScrollView *scrollView;
@property (nonatomic, strong) NSButton *placeholderLabel;
@end
@implementation HWMTextView
- (instancetype)initWithFrame:(NSRect)frameRect {
    if (self = [super initWithFrame:frameRect]) {
        [self createView];
    }
    return self;
}

- (void)createView {
    self.wantsLayer = YES;
    self.layer.backgroundColor = [NSColor whiteColor].CGColor;
    self.layer.cornerRadius = 6.0;
    self.layer.borderWidth = 1.0;
    self.layer.borderColor = [NSColor colorWithRGB:0xCCCCCC].CGColor;
    
    self.scrollView.documentView = self.textView;
    
    [self addSubview:self.scrollView];
    [self.scrollView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.edges.mas_equalTo(self);
    }];
    
    [self addSubview:self.placeholderLabel];
    [self.placeholderLabel mas_makeConstraints:^(MASConstraintMaker *make) {
        make.top.mas_equalTo(self).offset(3);
        make.leading.mas_equalTo(self).offset(6);
    }];
}

- (void)beginEdit:(NSButton *)button {
    [self.window makeFirstResponder:self.textView];
}

#pragma mark - NSTextViewDelegate
- (BOOL)textView:(NSTextView *)textView shouldChangeTextInRange:(NSRange)affectedCharRange replacementString:(NSString *)replacementString {
    NSString *str = [textView.string stringByReplacingCharactersInRange:affectedCharRange withString:replacementString];
    if (textView.string.length >= MAX_COUNT && str.length > MAX_COUNT) {
        return NO;
    }
    if (self.textDidChangeBlock) {
        self.textDidChangeBlock(str);
    }
    self.placeholderLabel.hidden = str.length > 0 || replacementString.length > 0;
    return YES;
}

- (void)setupAttributedPlaceholderString {
    if (HWMSafeString(self.placeholderString)) {
        NSFont *font = self.placeholderFont ? : [NSFont systemFontOfSize:DEFAULT_FONT_SIZE];
        NSColor *color = self.placeholderColor ? : [NSColor colorWithRGB:0x999999];
        NSMutableAttributedString *attributedString = [[NSMutableAttributedString alloc] initWithString:self.placeholderString];
        NSMutableDictionary *dic = @{}.mutableCopy;
        dic[NSForegroundColorAttributeName] = color;
        dic[NSFontAttributeName] = font;
        [attributedString addAttributes:dic range:NSMakeRange(0, self.placeholderString.length)];
        self.placeholderLabel.attributedTitle = attributedString;
    } else {
        self.placeholderLabel.attributedTitle = [[NSAttributedString alloc] initWithString:@""];
    }
    [self.placeholderLabel sizeToFit];
}

- (void)setPlaceholderString:(NSString *)placeholderString {
    _placeholderString = placeholderString;
    [self setupAttributedPlaceholderString];
}

- (void)setPlaceholderFont:(NSFont *)placeholderFont {
    _placeholderFont = placeholderFont;
    [self setupAttributedPlaceholderString];
}

- (void)setPlaceholderColor:(NSColor *)placeholderColor {
    _placeholderColor = placeholderColor;
    [self setupAttributedPlaceholderString];
}

- (void)setPlaceholderString:(NSString *)string font:(NSFont *)font color:(NSColor *)color {
    _placeholderString = string;
    _placeholderFont = font;
    _placeholderColor = color;
    [self setupAttributedPlaceholderString];
}

- (NSScrollView *)scrollView {
    if (!_scrollView) {
        _scrollView = [[NSScrollView alloc] init];
        _scrollView.autohidesScrollers = NO;
        _scrollView.hasVerticalScroller = YES;
        _scrollView.hasHorizontalScroller = NO;
        _scrollView.verticalScrollElasticity = NSScrollElasticityAutomatic;
        _scrollView.horizontalScrollElasticity = NSScrollElasticityNone;
        _scrollView.wantsLayer = YES;
    }
    return _scrollView;
}

- (NSTextView *)textView {
    if (!_textView) {
        _textView = [[NSTextView alloc] initWithFrame:NSMakeRect(0, 0, 555, 111)];
        _textView.allowsUndo = YES;
        _textView.richText = NO;
        _textView.delegate = self;
        _textView.backgroundColor = [NSColor whiteColor];
        _textView.textColor = [NSColor colorWithRGB:0x333333];
        _textView.textContainerInset = NSMakeSize(5, 5);
        _textView.autoresizingMask = NSViewHeightSizable;
        _textView.font = [NSFont systemFontOfSize:DEFAULT_FONT_SIZE];
        NSMutableParagraphStyle *style = [[NSMutableParagraphStyle alloc] init];
        style.lineSpacing = 5;
        _textView.defaultParagraphStyle = style;
    }
    return _textView;
}

- (NSButton *)placeholderLabel {
    if (!_placeholderLabel) {
        _placeholderLabel = [[NSButton alloc] init];
        _placeholderLabel.font = [NSFont systemFontOfSize:DEFAULT_FONT_SIZE];
        _placeholderLabel.bordered = NO;
        _placeholderLabel.cell.usesSingleLineMode = NO;
        _placeholderLabel.target = self;
        _placeholderLabel.action = @selector(beginEdit:);
    }
    return _placeholderLabel;
}
@end
