//
//  HWMCreateConfViewController.m
//  HWMUISDKNativeDemo
//
//  Created by guomeng on 2020/10/24.
//  Copyright © 2020 Huawei Technologies Co. Ltd. All rights reserved.
//

#import "HWMCreateConfViewController.h"
#import "HWMInputViewCell.h"
#import "HWMConfTypeSelectCell.h"
#import "HWMSwitchCell.h"
#import <HWMUISDK/HWMUISDK.h>
#import "HWMSelectVrmViewController.h"
#import "HWMSelectContactViewController.h"
#import "HWMSDKTableView.h"
#import "HWMJoinConfLimitCell.h"
#import "HWMContactInfoModel.h"
#import "HWMInputInfoTitleCell.h"

typedef NS_ENUM (NSInteger, HWMCreateConfCellType) {
    HWMCreateConfCellTypeSubject = 0,   // 会议主题
    HWMCreateConfCellTypeShareVmr,      // 启用共享云会议室
    HWMCreateConfCellTypeVmr,           // 启用云会议室
    HWMCreateConfCellTypeFixed,         // 使用个人会议ID（是否固定会议ID）
    HWMCreateConfCellTypeNeedAttendee,  // 是否需要携带与会者
    HWMCreateConfCellTypeAutoInvite,    // 会议开始自动呼叫与会者
    HWMCreateConfCellTypeAllowJoin,     // 允许入会
    HWMCreateConfCellTypeCamera,        // 打开摄像头
    HWMCreateConfCellTypeMic,           // 打开麦克风
    HWMCreateConfCellTypeAllowRecord,   // 允许录制
    HWMCreateConfCellTypeAutoRecord,    // 自动录制
    HWMCreateConfCellTypeNeedPassword,  // 是否设置来宾密码
    HWMCreateConfCellTypePassword,      // 来宾密码
    HWMCreateConfCellTypeWaitingRoom,   // 是否开启等候室
    HWMCreateConfCellTypeAutoMute,      // 个人客户端入会自动静音
    HWMCreateConfCellTypeHardTerminalAutoMute,  // 会议室设备入会自动静音
    HWMCreateConfCellTypeSupportWatermark,  // 是否支持水印
    HWMCreateConfCellTypeDefaultSummaryState,   // 录制时开启会议纪要
    HWMCreateConfCellTypeAutoPublishSummary,    // 自动发布会议纪要
    HWMCreateConfCellTypeCustomInfo,    // 自定义信息
    HWMCreateConfCellTypeParticipants   // 最大与会人数
};

@interface HWMCreateConfViewController ()<HWMSDKTableViewDelegate, HWMSDKTableViewDataSourse, HWMSwitchCellDelegate, HWMBizNotificationHandler, HWMInputInfoTitleCellDelegate>

/// 按钮
@property (nonatomic, strong) NSButton *createBtn;
/// 标题
@property (nonatomic, copy) NSDictionary<NSNumber *, NSString *> *titleDic;
/// 创建会议参数
@property (nonatomic, strong) HWMCreateConfParam *confParam;
/// vmrList
@property (nonatomic, strong) NSMutableArray <HWMVmrInfoModel *> *vmrList;
/// 是否支持共享vmr
@property (nonatomic, assign) BOOL isEnableShareVmr;
/// 是否携带与会者
@property (nonatomic, assign) BOOL needAttendee;
/// 是否自动外呼
@property (nonatomic, assign) BOOL isAutoInvite;
/// 与会者
@property (nonatomic, strong) NSMutableArray <HWMAttendeeInfo *> *attenteeArr;
/// tableView
@property (nonatomic, strong) HWMSDKTableView *listView;
/// 当前使用的vmr信息
@property (nonatomic, strong) HWMVmrInfoModel *vmrInfoModel;
/// 是否使用共享vmr
@property (nonatomic, assign) BOOL useShareVmr;
/// 是否使用云会议室
@property (nonatomic, assign) BOOL useCloudVmr;
/// 是否使用固定ID
@property (nonatomic, assign) BOOL useFixedId;

@end

@implementation HWMCreateConfViewController

- (void)viewDidLoad {
    [super viewDidLoad];
    [self setupData];
    [self setupUI];
}

- (void)viewWillAppear {
    [super viewWillAppear];
    /// 订阅要在初始化后根据需要设置
    [self configNotification];
    [self getVmrList];
    [self resetAttendee];
}

#pragma mark - Data

- (void)setupData {
    self.titleDic = @{
        @(HWMCreateConfCellTypeSubject) : @"会议主题",
        @(HWMCreateConfCellTypeShareVmr) : @"启用共享云会议室",
        @(HWMCreateConfCellTypeVmr) : @"启用云会议室",
        @(HWMCreateConfCellTypeFixed) : @"使用个人会议ID",
        @(HWMCreateConfCellTypeNeedAttendee) : @"是否需要携带与会者",
        @(HWMCreateConfCellTypeAutoInvite) : @"会议开始自动呼叫与会者",
        @(HWMCreateConfCellTypeAllowJoin) : @"允许入会",
        @(HWMCreateConfCellTypeCamera) : @"打开摄像头",
        @(HWMCreateConfCellTypeMic) : @"打开麦克风",
        @(HWMCreateConfCellTypeAllowRecord) : @"允许录制",
        @(HWMCreateConfCellTypeAutoRecord) : @"自动录制",
        @(HWMCreateConfCellTypeNeedPassword) : @"是否设置来宾密码",
        @(HWMCreateConfCellTypePassword) : @"来宾密码",
        @(HWMCreateConfCellTypeWaitingRoom) : @"是否开启等候室",
        @(HWMCreateConfCellTypeAutoMute) : @"个人客户端入会自动静音",
        @(HWMCreateConfCellTypeHardTerminalAutoMute) : @"会议室设备入会自动静音",
        @(HWMCreateConfCellTypeSupportWatermark) : @"是否支持水印",
        @(HWMCreateConfCellTypeDefaultSummaryState) : @"录制时开启会议纪要",
        @(HWMCreateConfCellTypeAutoPublishSummary) : @"自动发布会议纪要",
        @(HWMCreateConfCellTypeCustomInfo) : @"自定义信息",
        @(HWMCreateConfCellTypeParticipants) : @"最大与会人数"
    };
}

- (void)resetAttendee {
    self.needAttendee = NO;
    [self.attenteeArr removeAllObjects];
    [self.listView reloadData];
}

- (void)getVmrList {
    if (![HWMSdk hasInit]) {
        return;
    }
    @weakify(self)
    [[HWMBizSdk getBizOpenApi] getVmrInfoList:^(NSError * _Nullable error, HWMVmrInfoListModel * _Nullable result) {
        @strongify(self)
        if (error) {
            [HWMConsoleManager shareInstance].error = error;
        } else {
            self.isEnableShareVmr = result.isEnableShareVmr;
            if (!self.isEnableShareVmr) {
                self.useShareVmr = NO;
            }
            [self.vmrList removeAllObjects];
            for (HWMVmrInfoModel *vmrInfo in result.vmrs) {
                if (vmrInfo.usable && (vmrInfo.confId.length > 0 || vmrInfo.vmrId.length > 0)) {
                    [self.vmrList addObject:vmrInfo];
                }
            }
            self.vmrInfoModel = self.vmrList.firstObject;
            self.useCloudVmr = NO;
            self.confParam.vmrId = self.useFixedId ? self.vmrInfoModel.vmrId : @"";
            [self.listView reloadData];
        }
    }];
}

#pragma mark - UI

- (void)setupUI {
    self.listView = [[HWMSDKTableView alloc] initTableViewClomuIdentifier:@"createConf"];
    self.listView.delegate = self;
    self.listView.dataSource = self;
    [self.view addSubview:self.listView];
    [self.listView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.leading.and.top.equalTo(self.view).offset(15);
        make.trailing.equalTo(self.view).offset(-15);
        make.bottom.equalTo(self.view).offset(-100);
    }];

    [self.view addSubview:self.createBtn];
    [self.createBtn mas_makeConstraints:^(MASConstraintMaker *make) {
        make.centerX.equalTo(self.view);
        make.bottom.equalTo(self.view).offset(-20);
        make.height.mas_equalTo(35);
        make.width.mas_equalTo(250);
    }];
}

#pragma mark - Delegate

- (NSInteger)numberOfRowsInTableView:(NSTableView *)tableView {
    return self.titleDic.count;
}

- (NSView *)tableView:(NSTableView *)tableView viewForTableColumn:(NSTableColumn *)tableColumn row:(NSInteger)row {
    if (row == HWMCreateConfCellTypeSubject) {
        // 会议主题
        HWMInputViewCell *cell = [HWMInputViewCell initInputViewCell:tableView ower:self];
        cell.placeholder = @"会议主题（必填）";
        cell.content = self.confParam.subject;
        @weakify(self)
        [cell setInputViewDidEndEditerHandler:^(NSString *_Nonnull content, NSInteger index) {
            @strongify(self)
            self.confParam.subject = content;
        }];
        cell.enabled = YES;
        return cell;
    }

    if (row == HWMCreateConfCellTypePassword) {
        // 来宾密码
        HWMInputViewCell *cell = [HWMInputViewCell initInputViewCell:tableView ower:self];
        cell.placeholder = @"来宾密码（不填使用随机密码）";
        cell.content = self.confParam.guestPwd;
        @weakify(self)
        [cell setInputViewDidEndEditerHandler:^(NSString *_Nonnull content, NSInteger index) {
            @strongify(self)
            self.confParam.guestPwd = content;
        }];
        cell.enabled = !self.useFixedId && !self.confParam.noPassword;
        return cell;
    }

    if (row == HWMCreateConfCellTypeCustomInfo) {
        // 自定义信息
        HWMInputViewCell *cell = [HWMInputViewCell initInputViewCell:tableView ower:self];
        cell.placeholder = @"自定义信息";
        cell.content = self.confParam.customInfo;
        @weakify(self)
        [cell setInputViewDidEndEditerHandler:^(NSString *_Nonnull content, NSInteger index) {
            @strongify(self)
            self.confParam.customInfo = content;
        }];
        cell.enabled = YES;
        return cell;
    }
    
    if (row == HWMCreateConfCellTypeParticipants) {
        // 最大与会人数
        HWMInputInfoTitleCell *cell = [HWMInputInfoTitleCell inputInfoTitleCell:tableView ower:self];
        cell.delegate = self;
        cell.index = row;
        cell.title = self.titleDic[@(HWMCreateConfCellTypeParticipants)];
        cell.placeholder = @"默认0";
        cell.content = [NSString stringWithFormat:@"%ld", self.confParam.concurrentParticipants];
        cell.formatterStyle = HWMFormatterStyleNumber;
        return cell;
    }

    if (row == HWMCreateConfCellTypeAllowJoin) {
        // 允许入会
        HWMJoinConfLimitCell *cell = [HWMJoinConfLimitCell joinConfLimitCell:tableView ower:self];
        cell.title = self.titleDic[@(row)];
        cell.menuTitles = @[@"所有人", @"仅企业内人员", @"仅会议邀请人员"];
        cell.currentIndex = self.confParam.callInRestrictionType > 0 ? self.confParam.callInRestrictionType - 1 : 0;
        @weakify(self)
        [cell setSelectMenuItemHandler:^(NSInteger index, NSString * _Nullable itemTitile) {
            @strongify(self)
            self.confParam.callInRestrictionType = index > 0 ? index + 1 : 0;
        }];
        return cell;
    }
    
    if (row == HWMCreateConfCellTypeAutoMute) {
        // 个人客户端入会自动静音
        HWMJoinConfLimitCell *cell = [HWMJoinConfLimitCell joinConfLimitCell:tableView ower:self];
        cell.title = self.titleDic[@(row)];
        cell.menuTitles = @[@"默认", @"开启", @"关闭"];
        cell.currentIndex = self.confParam.autoMuteMode;
        @weakify(self)
        [cell setSelectMenuItemHandler:^(NSInteger index, NSString * _Nullable itemTitile) {
            @strongify(self)
            self.confParam.autoMuteMode = (HWMAutoMuteType)index;
        }];
        return cell;
    }
    
    if (row == HWMCreateConfCellTypeHardTerminalAutoMute) {
        // 会议室设备入会自动静音
        HWMJoinConfLimitCell *cell = [HWMJoinConfLimitCell joinConfLimitCell:tableView ower:self];
        cell.title = self.titleDic[@(row)];
        cell.menuTitles = @[@"默认", @"开启", @"关闭"];
        cell.currentIndex = self.confParam.hardTerminalAutoMuteMode;
        @weakify(self)
        [cell setSelectMenuItemHandler:^(NSInteger index, NSString * _Nullable itemTitile) {
            @strongify(self)
            self.confParam.hardTerminalAutoMuteMode = (HWMAutoMuteType)index;
        }];
        return cell;
    }

    HWMSwitchCell *cell = [HWMSwitchCell initSwitchCell:tableView ower:self];
    cell.index = row;
    cell.delegate = self;
    cell.title = self.titleDic[@(row)];
    
    if (row == HWMCreateConfCellTypeShareVmr) {
        // 启用共享云会议室
        cell.state = self.useShareVmr;
        cell.enabled = self.isEnableShareVmr;
    } else if (row == HWMCreateConfCellTypeVmr) {
        // 启用云会议室
        cell.state = self.useCloudVmr;
        cell.enabled = !self.useShareVmr && self.vmrList.count > 1;
    } else if (row == HWMCreateConfCellTypeFixed) {
        // 使用个人会议ID（是否固定会议ID）
        NSString *title = [NSString stringWithFormat:@"%@ %@", self.useCloudVmr ? @"使用云会议室ID" : @"使用个人会议ID", self.vmrInfoModel.confId.length ? self.vmrInfoModel.confId : @""];
        if (self.useShareVmr) {
            title = @"使用共享云会议室资源时，会议号会随机生成";
        }
        cell.title = title;
        cell.state = self.useFixedId;
        cell.enabled = !self.useShareVmr;
    } else if (row == HWMCreateConfCellTypeNeedAttendee) {
        // 是否需要携带与会者
        cell.state = self.needAttendee;
        cell.enabled = YES;
    } else if (row == HWMCreateConfCellTypeAutoInvite) {
        // 会议开始自动呼叫与会者
        cell.state = self.isAutoInvite;
        cell.enabled = YES;
    } else if (row == HWMCreateConfCellTypeCamera) {
        // 打开摄像头
        cell.state = self.confParam.isCameraOn;
        cell.enabled = YES;
    } else if (row == HWMCreateConfCellTypeMic) {
        // 打开麦克风
        cell.state = self.confParam.isMicOn;
        cell.enabled = YES;
    } else if (row == HWMCreateConfCellTypeAllowRecord) {
        // 允许录制
        cell.state = self.confParam.isRecordOn;
        cell.enabled = YES;
    }  else if (row == HWMCreateConfCellTypeAutoRecord) {
        // 自动录制
        cell.state = self.confParam.isAutoRecord;
        cell.enabled = YES;
    } else if (row == HWMCreateConfCellTypeNeedPassword) {
        // 是否设置来宾密码
        cell.state = !self.confParam.noPassword;
        cell.enabled = !self.useFixedId;
    } else if (row == HWMCreateConfCellTypeWaitingRoom) {
        // 是否开启等候室
        cell.state = self.confParam.isOpenWaitingRoom;
        cell.enabled = YES;
    } else if (row == HWMCreateConfCellTypeSupportWatermark) {
        // 是否支持水印
        cell.state = self.confParam.supportWatermark;
        cell.enabled = YES;
    } else if (row == HWMCreateConfCellTypeDefaultSummaryState) {
        // 录制时开启会议纪要
        cell.state = self.confParam.defaultSummaryState;
        cell.enabled = YES;
    } else if (row == HWMCreateConfCellTypeAutoPublishSummary) {
        // 自动发布会议纪要
        cell.state = self.confParam.autoPublishSummary;
        cell.enabled = YES;
    }
    return cell;
}

- (CGFloat)tableView:(NSTableView *)tableView heightOfRow:(NSInteger)row {
    return 40.0;
}

- (void)switchStateChanged:(BOOL)state index:(NSInteger)index {
    NSLog(@"%hhd, %lu", state, index);
    HWMSwitchCell *cell = [self.listView.listView viewAtColumn:0 row:index makeIfNecessary:YES];
    switch (index) {
        case HWMCreateConfCellTypeShareVmr: {
            // 启用共享云会议室
            if (state) {
                self.useCloudVmr = NO;
                self.useFixedId = NO;
                self.confParam.vmrId = @"";
            }
            self.useShareVmr = state;
            self.confParam.confResType = state ? HWMSDKConfRestypeShareVmr : HWMSDKConfRestypeDefault;
            [self.listView reloadData];
        }
            break;
        case HWMCreateConfCellTypeVmr: {
            // 启用云会议室
            if (state) {
                cell.state = NO;
                HWMSelectVrmViewController *selectVrm = [[HWMSelectVrmViewController alloc] init];
                NSMutableArray *vmrArr = [NSMutableArray arrayWithArray:self.vmrList];
                [vmrArr removeObjectAtIndex:0];
                selectVrm.vmrArr = vmrArr;
                @weakify(self)
                [selectVrm setSelectVrmInfoHandler:^(HWMVmrInfoModel *_Nonnull infoModel) {
                    @strongify(self)
                    if (infoModel) {
                        cell.state = YES;
                        self.vmrInfoModel = infoModel;
                        self.useCloudVmr = YES;
                        self.confParam.vmrId = infoModel.vmrId;
                        self.confParam.callInRestrictionType = infoModel.confAllowJoinUser;
                        self.confParam.isOpenWaitingRoom = infoModel.isOpenWaitingRoom;
                        [self.listView reloadData];
                    }
                }];
                [self presentViewControllerAsModalWindow:selectVrm];
            } else {
                self.vmrInfoModel = self.vmrList.firstObject;
                self.useCloudVmr = NO;
                self.confParam.vmrId = self.useFixedId ? self.vmrInfoModel.vmrId : @"";
                [self.listView reloadData];
            }
        }
            break;
        case HWMCreateConfCellTypeFixed: {
            // 使用个人会议ID（是否固定会议ID）
            if (self.vmrList.count == 0) {
                [HWMConsoleManager shareInstance].console = @"没有可用个人会议ID";
                cell.state = NO;
            } else {
                self.confParam.vmrId = state || self.useCloudVmr ? self.vmrInfoModel.vmrId : @"";
                self.useFixedId = state;
                HWMSwitchCell *pwdCell = [self.listView.listView viewAtColumn:0 row:HWMCreateConfCellTypeNeedPassword makeIfNecessary:YES];
                HWMInputViewCell *pwdTextCell = [self.listView.listView viewAtColumn:0 row:HWMCreateConfCellTypePassword makeIfNecessary:YES];
                pwdCell.enabled = !self.useFixedId;
                pwdTextCell.enabled = !self.useFixedId && !self.confParam.noPassword;
                if (state) {
                    [self updateConfAllowJoinUser];
                    [self updateOpenWaitingRoom];
                }
            }
        }
            break;
        case HWMCreateConfCellTypeNeedAttendee: {
            // 是否需要携带与会者
            if (state == NO) {
                self.needAttendee = NO;
                [self.attenteeArr removeAllObjects];
                return;
            }
            cell.state = NO;
            HWMSelectContactViewController *contact = [[HWMSelectContactViewController alloc] init];
            @weakify(self)
            [contact setSelectAttendeeInfoHandler:^(NSArray *_Nullable attendees) {
                @strongify(self)
                for (HWMContactInfoModel *model in attendees) {
                    HWMAttendeeInfo *info = [[HWMAttendeeInfo alloc] init];
                    info.accountId = model.accountId;
                    info.name = model.name;
                    info.aliasCN = model.aliasCN;
                    info.aliasEN = model.aliasEN;
                    info.number = model.number;
                    info.thirdUserId = model.thirdUserId;
                    info.email = model.email;
                    info.mute = model.mute;
                    info.role = model.role;
                    info.sms = model.sms;
                    info.orgId = model.corpId;
                    info.isAutoInvite = self.isAutoInvite;
                    [self.attenteeArr addObject:info];
                }
                if (attendees.count > 0) {
                    self.needAttendee = YES;
                    cell.state = YES;
                } else {
                    self.needAttendee = NO;
                }
            }];
            [self presentViewControllerAsModalWindow:contact];
        }
            break;
        case HWMCreateConfCellTypeAutoInvite: {
            // 会议开始自动呼叫与会者
            self.isAutoInvite = state;
            for (HWMAttendeeInfo *info in self.attenteeArr) {
                info.isAutoInvite = self.isAutoInvite;
            }
        }
            break;
        case HWMCreateConfCellTypeCamera:
            // 打开摄像头
            self.confParam.isCameraOn = state;
            break;
        case HWMCreateConfCellTypeMic:
            // 打开麦克风
            self.confParam.isMicOn = state;
            break;
        case HWMCreateConfCellTypeAllowRecord:
            // 允许录制
            self.confParam.isRecordOn = state;
            break;
        case HWMCreateConfCellTypeAutoRecord:
            // 自动录制
            self.confParam.isAutoRecord = state;
            break;
        case HWMCreateConfCellTypeNeedPassword: {
            // 是否设置来宾密码
            self.confParam.noPassword = !state;
            HWMInputViewCell *pwdCell = [self.listView.listView viewAtColumn:0 row:HWMCreateConfCellTypePassword makeIfNecessary:YES];
            pwdCell.enabled = !self.useFixedId && !self.confParam.noPassword;
        }
        case HWMCreateConfCellTypeWaitingRoom:
            // 是否开启等候室
            self.confParam.isOpenWaitingRoom = state;
            break;
        case HWMCreateConfCellTypeSupportWatermark:
            // 是否支持水印
            self.confParam.supportWatermark = state;
            break;
        case HWMCreateConfCellTypeDefaultSummaryState:
            // 录制时开启会议纪要
            self.confParam.defaultSummaryState = state;
            break;
        case HWMCreateConfCellTypeAutoPublishSummary:
            // 自动发布会议纪要
            self.confParam.autoPublishSummary = state;
            break;
        default:
            break;
    }
}

/// 会议状态变更通知
- (void)onConfStatusChanged:(HWMConfStatus)status {
    switch (status) {
        case HWMConfStatusIdle:
            [HWMConsoleManager shareInstance].console = @"会议结束";
            break;
        case HWMConfStatusIncoming:
            [HWMConsoleManager shareInstance].console = @"正在来电";
            break;
        case HWMConfStatusCallingOut:
            [HWMConsoleManager shareInstance].console = @"正在呼出";
            break;
        case HWMConfStatusConnected:
            [HWMConsoleManager shareInstance].console = @"已接通";
            break;
        default:
            break;
    }
}

- (void)updateConfAllowJoinUser {
    self.confParam.callInRestrictionType = _vmrInfoModel.confAllowJoinUser;
    HWMJoinConfLimitCell *confAllowJoinUserCell = [self.listView.listView viewAtColumn:0 row:HWMCreateConfCellTypeAllowJoin makeIfNecessary:YES];
    if (confAllowJoinUserCell && [confAllowJoinUserCell isKindOfClass:[HWMJoinConfLimitCell class]]) {
        confAllowJoinUserCell.currentIndex = self.confParam.callInRestrictionType > 0 ? self.confParam.callInRestrictionType - 1 : 0;
    }
}

- (void)updateOpenWaitingRoom {
    self.confParam.isOpenWaitingRoom = _vmrInfoModel.isOpenWaitingRoom;
    HWMSwitchCell *waitingRoomCell = [self.listView.listView viewAtColumn:0 row:HWMCreateConfCellTypeWaitingRoom makeIfNecessary:YES];
    if (waitingRoomCell && [waitingRoomCell isKindOfClass:[HWMSwitchCell class]]) {
        waitingRoomCell.state = self.confParam.isOpenWaitingRoom;
        waitingRoomCell.enabled = YES;
    }
}

#pragma mark - HWMInputInfoTitleCellDelegate

- (void)inputInfoCellDidEdite:(NSString *)content index:(NSInteger)index {
    if (index == HWMCreateConfCellTypeParticipants) {
        self.confParam.concurrentParticipants = [content integerValue];
    }
}

#pragma mark - Event

- (void)configNotification {
    [[HWMSdk getSdkConfig] subscribeBizNotificationHandler:self];
}

- (void)createConf {
    if (self.confParam.subject.length == 0) {
        [HWMConsoleManager shareInstance].console = @"subject can not be nil";
        return;
    }

    NSString *str = self.confParam.vmrId;
    [HWMConsoleManager shareInstance].console = [NSString stringWithFormat:@"conf vmr id : %@", str];

    self.confParam.members = self.attenteeArr;
    self.confParam.vmrConfIdType = self.useFixedId ? HWMVmrConfIdTypeFixed : HWMVmrConfIdTypeRandom;

    [[HWMSdk getOpenApi] createConf:self.confParam callback:^(NSError *_Nullable error, HWMCreateConfResult *_Nullable result) {
        if (error) {
            [HWMConsoleManager shareInstance].error = error;
        } else {
            [HWMConsoleManager shareInstance].console = [NSString stringWithFormat:@"%@", [result yy_modelDescription]];
        }
    }];
}

#pragma mark - Lazy

- (NSButton *)createBtn {
    if (!_createBtn) {
        _createBtn = [NSButton buttonWithTitle:@"创建会议" font:nil fontColor:[NSColor whiteColor] target:self action:@selector(createConf)];
        _createBtn.layer.backgroundColor = [NSColor colorWithRGB:0x0D94FF].CGColor;
        _createBtn.layer.cornerRadius = 4.0;
        _createBtn.layer.masksToBounds = YES;
    }
    return _createBtn;
}

- (HWMCreateConfParam *)confParam {
    if (!_confParam) {
        _confParam = [[HWMCreateConfParam alloc] init];
    }
    return _confParam;
}

- (NSMutableArray<HWMVmrInfoModel *> *)vmrList {
    if (!_vmrList) {
        _vmrList = [NSMutableArray array];
    }
    return _vmrList;
}

- (NSMutableArray<HWMAttendeeInfo *> *)attenteeArr {
    if (!_attenteeArr) {
        _attenteeArr = [NSMutableArray array];
    }
    return _attenteeArr;
}

@end
