﻿/**
 * @file HwmSdkAgent.h
 *
 * Copyright(C), 2019-2020, Huawei Tech. Co., Ltd. ALL RIGHTS RESERVED.
 *
 * @brief [en] Interface file.
 *        [cn] 接口文件
 */

#ifndef _HWM_SDK_AGENT_H_
#define _HWM_SDK_AGENT_H_

#ifdef HWM_SDK_AGENT_API_EXPORTS
#define HWM_SDK_AGENT_API __declspec(dllexport)
#else
#define HWM_SDK_AGENT_API __declspec(dllimport)
#endif

#include "HwmAgentDef.h"


#ifdef __cplusplus
extern "C" {
#endif

namespace hwmsdkagent
{

    /**
     * @brief [en] This interface is used to init sdk.
     *        [cn] 初始化
     *
     * @param [in] HwmInitInfo* initInfo         [en] Init info
     *                                           [cn] 初始化参数
     * @attention [en] NA
     *            [cn] NA
     **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode Init(HwmInitInfo *initInfo);

    /**
     * @brief [en] This interface is used to exit sdk.
     *        [cn] 退出
     *
     * @attention [en] NA
     *            [cn] NA
     **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode Exit();

    /**
     * @brief [en] This interface is used to config sdk.
     *        [cn] 配置定制化参数
     * @param [in] char* config            [en] Config info
     *                                     [cn] 配置信息，为Json格式字符串
     * @attention [en] More detail in doc file
     *            [cn] 字符串格式见文档
     **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode Config(const char uiConfig[HWM_MAX_CONFIG_LEN]);
    
    /**
     * @brief [en] This interface is used to clean all config.
     *        [cn] 清除定制化参数
     * @attention [en] NA
     *            [cn] NA
     **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode CleanConfig();

    /**
    * @brief [en] This interface is used to enable
    *        [cn] 会议相关功能配置开关
    *
    * @param [in] HwmEnableFeatureType type                         [en] Enable type
    *                                                               [cn] 开启的功能枚举
    * @param [in] bool isEnable                                     [en] Enable state
    *                                                               [cn] 开启状态
    * @attention  [en] NA
    *             [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode EnableFeature(HwmEnableFeatureType type, bool isEnable);

    /**
    * @brief [en] This interface is used to set language.
    *        [cn] 设置语言
    *
    * @param [in] HwmLanguage language            [en] HwmLanguage type
    *                                             [cn] 语言类型
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetLanguage(HwmLanguage language);

    /**
    * @brief [en] This interface is used to set language.
    *        [cn] 设置语言
    *
    * @param [in] HwmLanguage language      [en] HwmLanguage type
    *                                       [cn] 语言类型，自定义时 path 不可为空。
    * @param [in] const char *languageCode  [en] Language code for customization. If the language is Chinese or English, this parameter can be a null character string..
    *                                       [cn] 自定义时对应的语言代码,若是中文/英文,该参数可以传空字符串。
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetLanguageWithCode(HwmLanguage language, const char *languageCode);

    /**
    * @brief [en] This interface is used to setting share permissions.
    *        [cn] 设置共享权限
    *
    * @param [in] bool isAllowed              [en] is sharing allowed
    *                                         [cn] 是否允许
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetShareScreenPermission(bool isAllowed);

    /**
    * @brief [en] This interface is used to rename self.
    *        [cn] 设置会场名
    *
    * @param [in] const char *data            [en] Name
    *                                         [cn] 会场名
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode RenameSelf(const char name[HWM_MAX_DISPLAY_NAME_LEN]);

    /**
     * @brief [en] This interface is used to login sdk.
     *        [cn] 登录
     *
     * @param [in] HwmLoginInfo* loginInfo            [en] Login info
     *                                                [cn] 登录信息
     * @attention [en] NA
     *            [cn] NA
     **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode Login(HwmLoginInfo *loginInfo);

    /**
     * @brief [en] This interface is used to logout sdk.
     *        [cn] 退出登录
     *
     * @attention [en] NA
     *            [cn] NA
     **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode Logout();

   /**
    * @brief [en] This interface is used to create conf.
    *        [cn] 创建会议
    *
    * @param [in] HwmCreateConfInfo *createConfInfo            [en] Create conf info
    *                                                          [cn] 创建会议参数
    * @param [in] HwmConfAttendee *attendees                   [en] Attendee info
    *                                                          [cn] 与会者信息，可以为空
    * @param [in] unsigned int count                           [en] Attendee list count
    *                                                          [cn] 与会者个数，可以为0
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode CreateConf(HwmCreateConfInfo *createConfInfo, HwmConfAttendee *attendees, unsigned int count);

   /**
    * @brief [en] This interface is used to join conf by id.
    *        [cn] 通过会议id入会
    *
    * @param [in] HwmJoinConfByIdInfo *joinConfByIdInfo            [en] Join conf by id
    *                                                              [cn] 通过会议id加入会议
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode JoinConfById(HwmJoinConfByIdInfo *joinConfByIdInfo);

    /**
    * @brief [en] This interface is used to join conf by random.
    *        [cn] 通过会议random入会
    *
    * @param [in] HwmJoinConfByRandomInfo *joinConfByRandomInfo            [en] Join conf by random
    *                                                              [cn] 通过会议random加入会议
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode JoinConfByRandom(HwmJoinConfByRandomInfo *joinConfByRandomInfo);

    /**
    * @brief [en] This interface is used to add attendee.
    *        [cn] 添加与会者
    *
    * @param [in] HwmConfAttendee *attendees         [en] Attendee info
    *                                                [cn] 与会者信息
    * @param [in] unsigned int count                 [en] Attendee list count
    *                                                [cn] 与会者个数
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode AddAttendee(HwmConfAttendee *attendees, unsigned int count);

   /**
    * @brief [en] This interface is used to leave conf.
    *        [cn] 离开会议
    *
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode LeaveConf();

   /**
    * @brief [en] This interface is used to end conf.
    *        [cn] 结束会议
    *
    * @attention [en] NA
    *            [cn] NA
    **/ 
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode EndConf();

   /**
    * @brief [en] This interface is used to start share.
    *        [cn] 开启共享
    *
    * @param [in] HwmStartShareInfo *startShareInfo            [en] Start share info
    *                                                          [cn] 开启共享参数
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode StartShare(HwmStartShareInfo *startShareInfo);

    /**
    * @brief [en] This interface is used to stop share.
    *        [cn] 停止共享
    *
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode StopShare();

   /**
    * @brief [en] This interface is book a conference.
    *        [cn] 预定会议
    *
    * @param [in] const HwmBookConfParam *bookConfParam        [en] book conf parameters
    *                                                          [cn] 预定会议参数
    *
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode BookConf(const HwmBookConfParam *bookConfParam);

    /**
    * @brief [en] This interface is used to cancel a conference booked before.
    *        [cn] 取消会议预定
    *
    * @param [in] const char confId[HWM_MAX_CONF_ID_LEN]            [en] the id of the conference
    *                                                               [cn] 需要取消的会议的id
    *
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode CancelConf(const char confId[HWM_MAX_CONF_ID_LEN]);

    /**
    * @deprecated [en] Deprecated API.
    *             [cn] 废弃接口, 推荐使用接口EditConf
    *
    * @brief [en] This interface is used to modify a conf booked before.
    *        [cn] 修改已经预定的会议
    *
    * @param [in] const HwmEditConfParam *param           [en] the new setting parameters of the conference
    *                                                     [cn] 需要取消的会议的id
    *
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode EidtConf(const HwmEditConfParam *confParam);

    /**
    * @brief [en] This interface is used to modify a conf booked before.
    *        [cn] 修改已经预定的会议
    *
    * @param [in] const HwmEditConfParam *param           [en] the new setting parameters of the conference
    *                                                     [cn] 需要取消的会议的id
    *
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode EditConf(const HwmEditConfParam *confParam);

    /**
    * @deprecated [en] Deprecated API.
    *             [cn] 废弃接口, 推荐使用新接口GetVmrInfoList
    *
    * @brief [en] This interface is used to get vmr list.
    *        [cn] 获取Vmr列表
    *
    * @attention [en] NA
    *            [cn] NA
    *
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetVmrList();

    /**
    * @brief [en] This interface is used to get vmr list.
    *        [cn] 获取Vmr列表
    *
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetVmrInfoList();

    /**
    * @brief [en] This interface is used to get conf detail.
    *        [cn] 获取会议详情
    *
    * @param [in] confId            [en] the id of the conference
    *                               [cn] 需要获取会议详情的会议id
    *
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetConfDetail(const char confId[HWM_MAX_CONF_ID_LEN]);

    /**
    * @brief [en] This interface is used to change virtual meeting room information.
    *        [cn] 修改虚拟会议室信息
    *
    * @param [in] const ChangeVmrInfoParam *vmrInfoParam               [en] the new value of the vmr info
    *                                                                  [cn] 虚拟会议室的修改信息
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode ChangeVmrInfo(const ChangeVmrInfoParam *vmrInfoParam);

    /**
    * @brief [en] This interface is used to set watermark inject information.
    *        [cn] 设置水印注入信息
    *
    * @param [in] const HwmWaterMarkInjectInfo *       [en] the param of the WaterMark Inject Info
    *                                                  [cn] 需要设置的水印注入信息
    *
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetWaterMark(const HwmWaterMarkInfo *wtInfo);

    /**
    * @deprecated [en] Deprecated API.
    *             [cn] 废弃接口
    * @brief [en] This interface is used to start call.
    *        [cn] 发起呼叫
    *
    * @param [in] HwmStartCallInfo* startCallInfo               [en] start call param
    *                                                           [cn] 发起呼叫参数
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode StartCall(const HwmStartCallInfo* startCallInfo);

    /**
    * @brief [en] This interface is used to set call incoming answer result.
    *        [cn] 设置呼叫来电应答:弹框、拒绝或接听
    *
    * @param [in] HWMInComingAnswerType type                   [en] incoming answer type
    *                                                          [cn] 来电应答类型
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetCallIncomingAnswerResult(HWMInComingAnswerType type);

    /**
    * @brief [en] This interface is used to set conf incoming answer result.
    *        [cn] 设置会议来电应答:弹框、拒绝或接听
    *
    * @param [in] HWMInComingAnswerType type                   [en] incoming answer type
    *                                                          [cn] 来电应答类型
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetConfIncomingAnswerResult(HWMInComingAnswerType type);

    /**
    * @brief [en] This interface is used to set file save path.
    *        [cn] 设置用户保存路径
    *
    * @param [in] HWMSavePathType pathType                     [en] save path type
    *                                                          [cn] 保存路径类型
    * @param [in] char* path                                   [en] save path
    *                                                          [cn] 保存路径
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetUserSavePath(HWMSavePathType pathType, const char* path);

    /**
    * @brief [en] This interface is used to get user state.
    *        [cn] 获取用户状态
    *
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetUserState();

    /**
    * @brief [en] This interface is used to set share strategy.
    *        [cn] 设置屏幕共享策略
    *
    * @param [in] HwmShareStrategyType shareStrategyType       [en] share strategy type
    *                                                          [cn] 屏幕共享策略类型
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetShareStrategy(HwmShareStrategyType shareStrategyType);

    /**
    * @brief [en] This interface is used to set share audio.
    *        [cn] 设置共享音频
    *
    * @param [in] bool isEnable                                [en] is enable
    *                                                          [cn] 是否开启
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetShareAudio(bool isEnable);

    /**
    * @brief
    * [en]
    * [cn]预约周期会议
    *
    * @param HwmbookCycleConfParam [type:HwmBookCycleConfParam][cs_allowNull:N]
    * [en]
    * [cn]预约周期会议的参数体
    * @see HwmBookCycleConfParam
    *
    * @return [type:HwmErrCode]
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode BookCycleConf(const HwmBookCycleConfParam* bookCycleConfParam);

    /**
    * @brief
    * [en]
    * [cn]编辑周期会议
    *
    * @param HwmmodifyParam [type:HwmModifyCycleConfParam][cs_allowNull:N]
    * [en]
    * [cn]编辑周期会议的参数体
    *
    * @return [type:HwmErrCode]
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode ModifyCycleConf(const HwmModifyCycleConfParam* modifyParam);

    /**
    * @brief
    * [en]
    * [cn]编辑周期子会议
    *
    * @param HwmmodifySubCycleConfParam [type:HwmModifySubCycleConfParam][cs_allowNull:N]
    * [en]
    * [cn]编辑周期会议的参数体
    *
    * @return [type:HwmErrCode]
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode ModifySubCycleConf(const HwmModifySubCycleConfParam* modifySubCycleConfParam);

    /**
    * @brief
    * [en]
    * [cn]取消周期会议
    *
    * @param HwmcancelCycleConfParam [type:HwmCancelCycleConfParam][cs_allowNull:N]
    * [en]
    * [cn]取消周期会议参数
    *
    * @return [type:HwmErrCode]
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode CancelCycleConf(const HwmCancelCycleConfParam* cancelCycleConfParam);

    /**
    * @brief
    * [en]
    * [cn]取消周期子会议
    *
    * @param HwmcancelSubCycleConfParam [type:HwmCancelSubCycleConfParam][cs_allowNull:N]
    * [en]
    * [cn]取消周期子会议参数
    *
    * @return [type:HwmErrCode]
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode CancelSubCycleConf(const HwmCancelSubCycleConfParam* cancelSubCycleConfParam);

    /**
    * @brief [en] This interface is config setting window.
    *        [cn] 定制设置界面
    *
    * @param [in] HwmSettingWindowConfig* configInfo            [en] Setting window config info
    *                                                           [cn] 设置窗口定制信息
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode ConfigSettingWindow(HwmSettingWindowConfig* configInfo);

    /**
    * @brief [en] This interface is used to set need transparent windows in share
    *        [cn] 设置屏幕共享中需透明的窗口
    *
    * @param [in] HWND handle[]                                [en] Window handle
    *                                                          [cn] 窗口句柄
    * @param [in] unsigned int count                           [en] Window handle list count
    *                                                          [cn] 窗口句柄个数
    * @attention [en] The count of windows must less than 4
    *            [cn] 设置窗口句柄个数不超过4个
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetTransparentWnd(HWND handle[], unsigned int count);

    /**
    * @brief [en] This interface is used to set login server info.
    *        [cn] 设置登录服务器信息
    * @param [in] HwmServerInfo* serverInfo            [en] Server info
    *                                                  [cn] 服务器信息
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetServerAddress(const HwmServerInfo* serverInfo);

    /**
    * @brief [en] This interface is used to set proxy info.
    *        [cn] 设置代理信息
    *
    * @param [in] HwmProxyInfo *proxyInfo            [en] Proxy info
    *                                                [cn] 代理信息
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetProxyInfo(const HwmProxyInfo *proxyInfo);

    /*********************************************以下是保留的接口*********************************************/

    /**
    * @brief [en] This interface is used to login sdk by account and token.
    *        [cn] 登录（token登录，大屏使用）
    *
    * @param [in] HwmAccountAndTokenAuthInfo* loginInfo            [en] Login info
    *                                                              [cn] 登录信息
    * @attention [en] Reserved
    *            [cn] 保留接口
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode LoginByToken(HwmAccountAndTokenAuthInfo *loginInfo);

    /**
    * @brief [en] This interface is used to to enable or disable pair feature.
    *        [cn] 开启或关闭配对功能
    *
    * @param [in] bool enable                               [en] If enable
    *                                                       [cn] 是否开启
    * @param [in] LOGIN_E_PROJECTION_CODE_MODE pairCodeMode [en] projection code mode
    *                                                       [cn] 投影码模式
    * @attention [en] Reserved
    *            [cn] 保留接口
    * @see Release
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode EnablePair(bool isEnable);

    /**
    * @brief [en] This interface is used to join conf by link.
    *        [cn] 通过链接入会
    *
    * @param [in] HwmJoinConfByLinkInfo *joinConfByLinkInfo            [en] Join conf by link
    *                                                                  [cn] 通过链接入会
    * @attention [en] Reserved
    *            [cn] 保留接口
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode JoinConfByLink(HwmJoinConfByLinkInfo *joinConfByLinkInfo);

    /**
    * @brief [en] This interface is used to ShowMeetingWindow.
    *        [cn] 显示会议窗口
    *
    * @attention [en] Reserved
    *            [cn] 保留接口
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode ShowMeetingWindow();

    /**
    * @brief [en] This interface is used to AVDeviceCtrl.
    *        [cn] 音视频设备控制功能
    *
    * @param [in] HwmDeviceType deviceType          [en] Device type
    *                                               [cn] 设备类型
    * @param [in] bool isOpen                       [en] is open
    *                                               [cn] 是否开启
    * @attention [en] Reserved
    *            [cn] 保留接口
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode AVDeviceCtrl(HwmDeviceType deviceType, bool isOpen);

    /**
    * @brief        [en] Callback of GetAttendeeList interface.
    *               [cn] 获取与会者列表
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetAttendeeList();

    /**
    * @brief        [en] Callback of GetAudienceList interface.
    *               [cn] 获取观众列表
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetAudienceList();

    /**
    * @brief        [en] This interface is used to SetAudioRawDataOutputConfig.
    *               [cn] 设置开启或关闭音频码流
    * @param        [in] bool isOpen                        [en] is open
    *                                                       [cn] 是否开启
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetAudioRawDataOutputConfig(bool isOpen);

    /**
    * @brief
    * [en]This interface is about general watch
    * [cn]选看
    *
    * @param watchListInfo [type:HwmWatchListInfo*][cs_allowNull:N]
    * [en]
    * [cn]选看列表
    *
    * @attention [en] Reserved
    *            [cn] 保留接口
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GeneralWatch(const HwmWatchListInfo* watchListInfo);
 /*********************************************以上是保留的接口*********************************************/

    /**
    * @brief [en] Obtaining System Agent Information. This method is used for synchronous processing
    *        [cn] 获取系统代理信息。此接口为同步处理
    * @param [out] char *proxy              [en] proxy server as addr[:port]
    *                                       [cn] 代理地址，格式为 addr[:port]
    * @param [in] int cchProxy              [en] proxy size in byte
    *                                       [cn] proxy 参数的字节数
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetSystemProxy(char *proxy, int cchProxy);

    /**
    * @brief [en] This method is used to get window handle list.
    *        [cn] 获取窗口句柄列表
    *
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetRenderWindowInfoList();

    /**
    * @brief [en] This method is used to get window handle list.
    *        [cn] 更新自定义按钮信息
    *
    * @param [in] HwmCustomButtonInfo *btnInfo   [en] Custom button info
    *                                            [cn] 自定义按钮信息
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode UpdateCustomButtonInfo(const HwmCustomButtonInfo *btnInfo);
    
    /**
    * @brief [en] This method is used to get SSO authorise url.
    *        [cn] 根据企业域名获取鉴权中心Url
    *
    * @param [in] const char *domain             [en] domain
    *                                            [cn] 企业域名
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetSSOAuthorizeUrl(const char *domain);

    /**
    * @brief [en] This method is used to login by SSO.
    *        [cn] SSO登录接口
    *
    * @param [in] HwmSSOAuthInfo *ssoAuthInfo    [en] sso auth info
    *                                            [cn] SSO鉴权信息
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode LoginBySSO(HwmSSOAuthInfo *ssoAuthInfo);

    /**
    * @brief [en] This interface is used to set network mode.
    *        [cn] 设置网络模式
    *
    * @param [in] HwmNetWorkModeInfo networkInfo    [en] network mode info
    *                                               [cn] 网络模式信息
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetNetworkMode(const HwmNetWorkModeInfo* networkInfo);

    /**
    * @brief [en] This interface is used to set raw data external config.
    *        [cn] 设置端到端加密接口配置信息
    *
    * @param [in] HwmRawDataExternalConfigInfo rawDataExternalConfigInfo     [en] raw data external config
    *                                                                        [cn] 端到端加密接口配置信息
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetRawDataExternalConfig(const HwmRawDataExternalConfigInfo* rawDataExternalConfigInfo);

    /**
    * @brief [en] This interface is used to show setting window.
    *        [cn] 显示设置窗口接口
    *
    * @param [in] HwmShowSettingWindowInfo *showInfo                          [en] Show setting window info
    *                                                                         [cn] 显示设置窗口信息
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode ShowSettingWindow(const HwmShowSettingWindowInfo* showInfo);

    /**
    * @brief [en] This interface is used to disable voice prompts.
    *        [cn] 禁用提示音
    *
    * @param [in] HwmVoicePromptsInfo *voicePromptsInfo        [en] Voice prompts info
    *                                                          [cn] 语音提示信息
    * @attention [en] NA
    *            [cn] NA
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode DisableVoicePrompts(const HwmVoicePromptsInfo* voicePromptsInfo);

    /**
    * @brief 
    * [en] This interface is used to upload self avatar.
    * [cn] 上传头像
    *
    * @detail
    * [en] This interface is used to upload self avatar.
    * [cn] 该接口用于上传当前自己的头像到会议服务器的通讯录。注意事项：1、请在已登录状态下调用。2、上传的图片文件格式仅支持jpg/png格式，大小需要限制在200KB以下；3、为了头像展示效果，建议图片宽高比为1:1
    *
    * @param selfAvatarInfo [type:HwmSelfAvatarInfo*]
    * [en] Self avatar info
    * [cn] 自己的头像信息
    *
    * @norepeat
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode UploadSelfAvatar(const HwmSelfAvatarInfo* selfAvatarInfo);

    /**
    * @brief [en] This interface is used to set voiceprint statement agreement.
    *        [cn] 签署声纹功能声明
    *
    * @param [in] HWM_BOOL isAgreed                  [en] Is agreed
    *                                                [cn] 是否同意
    *
    * @mac:disable
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode AgreeVoiceprintStatement(bool isAgreed);

    /**
    * @brief
    * [en] This interface is config voiceprint record window.
    * [cn] 定制声纹录入窗口
    *
    * @detail
    * [en] config voiceprint record window
    * [cn] 该接口用于定制声纹录入窗口。
    *
    * @param configInfo [type:HwmVoiceprintRecordWindowConfig*]
    * [en] Voiceprint window config info
    * [cn] 声纹录入窗口定制信息
    *
    * @mac:disable
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode ConfigVoiceprintRecordWindow(const HwmVoiceprintRecordWindowConfig* configInfo);

    /**
    * @brief
    * [en] This interface is used to show voiceprint record window.
    * [cn] 显示声纹录入窗口接口
    *
    * @detail
    * [en] show voiceprint record window
    * [cn] 该接口用于显示声纹录入窗口。
    *
    * @param isShow [type:HWM_BOOL]
    * [en] is show
    * [cn] 是否显示
    *
    * @mac:disable
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode ShowVoiceprintRecordWindow(bool isShow);

    /**
    * @brief 
    * [en] This interface is used to get file save path.
    * [cn] 获取用户保存路径
    *
    * @detail
    * [en]
    * [cn] 该接口用于获取用户文件（包括本地录制文件）的保存路径。
    *
    * @param pathType [type:HWMSavePathType*]
    * [en] Save path type
    * [cn] 保存路径类型
    *
    * @mac:disable
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetUserSavePath(HWMSavePathType pathType);

    /**
    * @brief [en] This interface is used to start p2p conf.
    *        [cn] 发起立即会议
    *
    * @param startCallInfo [type:HwmP2PStartCallInfo*][cs_allowNull:N]
    * [en] Create p2p conf info
    * [cn] 创建立即会议信息
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode StartP2pConf(const HwmP2PStartCallInfo* startCallInfo);

    /**

    * @brief 

    * [en] This interface is used to delete self avatar.

    * [cn] 删除头像

    *

    * @detail
    * [en] This interface is used to delete self avatar.
    * [cn] 该接口用于删除自定义的通讯录头像。注意事项：1、请在已登录状态下调用

    *

    * @norepeat

    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode DeleteSelfAvatar();

    /**
    * @brief
    * [en] This interface is used to set log save path.
    * [cn] 设置日志保存路径
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置日志的保存路径。注意事项：1、该接口非必需调用，若不设置，则使用默认路径。
    *
    * @param logPathInfo [type:HwmLogPathInfo*]
    * [en] Log path info
    * [cn] 日志路径信息
    *
    * @mac:transToSync
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetLogPath(const HwmLogPathInfo* logPathInfo);

    /**
    * @brief
    * [en] This interface is used to get error message.
    * [cn] 获取错误提示
    *
    * @param errorCode [type:HWM_INT32]
    * [en] Error code
    * [cn] 错误码
    *
    * @param errorMessageInfo [type:HwmErrorMessageInfo*]
    * [en] error message
    * [cn] 错误提示
    *
    * @synchronized
    * @paramOut
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetErrorMessage(int errorCode, OUT HwmErrorMessageInfo* errorMessageInfo);

    /**
    * @brief
    * [en] This interface is used to show toast info.
    * [cn] 显示提示信息
    *
    * @detail
    * [en]
    * [cn] 该接口用于显示提示信息
    *
    * @param toastInfo [type:HwmToastInfo*]
    * [en] Toast info
    * [cn] 提示信息
    *
    * @mac:transToSync
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode ShowToast(const HwmToastInfo* toastInfo);

    /**
    * @brief
    * [en] This interface is used to get main window info.
    * [cn] 获取主窗口信息
    *
    * @detail
    * [en]
    * [cn] 该接口用于获取主窗口信息
    *
    * @mac:transToSync
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetMainWindowInfo();

    /**
    * [en]This interface is about set annotation window
    * [cn]设置标注窗口
    *
    * @param annoWndInfo [type:HwmAnnotationWndInfo*][cs_allowNull:N]
    * [en]
    * [cn]标注窗口信息
    *
    * @js:disable
    * @mac:transToSync
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetAnnotationWindow(const HwmAnnotationWndInfo* annoWndInfo);

    /**
    * @brief
    * [en]This interface is about start annotation
    * [cn]开启标注
    *
    * @param annoWndInfo [type:HwmAnnotationWndInfo*][cs_allowNull:N]
    * [en]
    * [cn]标注窗口信息
    *
    * @js:disable
    * @mac:transToSync
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode StartAnnotation(const HwmAnnotationWndInfo* annoWndInfo);

    /**
    * @brief
    * [en]This interface is about save annotation
    * [cn]保存标注
    *
    * @param configInfo [type:HwmAnnotationSaveConfigInfo*][cs_allowNull:N]
    * [en]
    * [cn]标注保存配置信息
    *
    * @js:disable
    * @mac:transToSync
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SaveAnnotation(const HwmAnnotationSaveConfigInfo* configInfo);

    /**
    * @brief
    * [en] This interface is used to set guaranteed ip addresses.
    * [cn] 设置(会控)保底ip地址
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置(会控)保底ip地址。注意事项：1、多个地址中间用分号隔开，最多支持4个地址
    *
    * @param guaranteedIpAddresses [type:char*]
    * [en] Guarantee ip Addresses
    * [cn] 保底ip地址信息
    *
    * @synchronized
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetGuaranteedIpAddresses(const char* guaranteedIpAddresses);

    /**
    * @brief
    * [en]This interface is about stop annotation
    * [cn]停止标注
    *
    * @js:disable
    * @mac:transToSync
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode StopAnnotation();

    /**
    * @brief
    * [en] This interface is used to start projection.
    * [cn] 发起投屏
    *
    * @detail
    * [en] This interface is used to start projection.
    * [cn] 该接口用于发起投屏。注意事项：1、该接口在已登录状态下才能调用。
    *
    * @param startProjectionParam [type:HwmStartProjectionParam*][cs_allowNull:N]
    * [en] Start projection param
    * [cn] 发起投屏的参数体
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode StartProjection(const HwmStartProjectionParam* startProjectionParam);

    /**
    * @brief [en] This interface is used to set CA verify info.
    *        [cn] 设置证书校验信息
    *
    * @param CAVerifyInfo [type:HwmCAVerifyInfo*]
    * [en] CA Verify info
    * [cn] 证书校验信息
    *
    * @mac:transToSync
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetCAVerifyInfo(const HwmCAVerifyInfo* CAVerifyInfo);

    /**
    * @brief [en] This interface is used to customize the text of the specified scene
    *        [cn] 自定义场景文案
    *
    * @param customSceneTextInfoList [type:HwmCustomSceneTextInfoList*]
    * [en] Customize scene content
    * [cn] 自定义场景文案
    *
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetCustomUIText(const HwmCustomSceneTextInfoList* customSceneTextInfoList);

    /**
    * @brief
    * [en] This interface is used to set auto hide tool bar.
    * [cn] 设置会控栏自动隐藏
    *
    * @detail
    * [en] This interface is used to set auto hide tool bar.
    * [cn] 该接口用于设置会控栏是否自动隐藏。注意事项：1、当次会议生效，离会后恢复默认体验，即鼠标无操作后一段时间后自动隐藏会控栏。
    *
    * @param autoHideToolBarParam [type:HwmAutoHideToolBarParam*][cs_allowNull:N]
    * [en] Auto Hide ToolBar param
    * [cn] 自动隐藏会控栏参数体
    *
    * @synchronized
    * @mac:disable
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetAutoHideToolBar(const HwmAutoHideToolBarParam* autoHideToolBarParam);

    /**
    * @brief [en] This interface is used to customize the image of the specified scene
    *        [cn] 自定义指定场景图标
    *
    * @param customSceneImageInfoList [type:HwmCustomSceneImageInfoList*]
    * [en] Customize scene image
    * [cn] 自定义场景图标信息列表
    *
    * @mac:disable
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetCustomUIImage(const HwmCustomSceneImageInfoList* customSceneImageInfoList);

    /**
    * @brief
    * [en] This interface is used to set call incoming answer type.
    * [cn] 设置来电应答方式
    *
    * @detail
    * [en]
    * [cn] 该接口用于设置来电应答: 原始弹框、自动拒绝或自动接听。
    *
    * @param inComingAnswerTypeInfo [type:HwmInComingAnswerTypeInfo*]
    * [en] Incoming answer info
    * [cn] 来电应答信息
    *
    * @mac:transToSync
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetAnswerType(const HwmInComingAnswerTypeInfo* inComingAnswerTypeInfo);

    /**
    * @brief
    * [en] This interface is used to accept conf.
    * [cn] 接受会议邀请
    *
    * @detail
    * [en] This interface is used to accept conf.
    * [cn] 该接口用于接受会议邀请
    *
    * @param acceptConfParam [type:HwmAcceptConfParam*]
    * [en] accept conf param
    * [cn] 接受会议邀请的参数体
    *
    * @mac:disable
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode AcceptConf(const HwmAcceptConfParam* acceptConfParam);

    /**
    * @brief
    * [en] This interface is used to reject conf.
    * [cn] 拒绝会议邀请
    *
    * @detail
    * [en] This interface is used to reject conf.
    * [cn] 该接口用于拒绝会议邀请
    *
    * @mac:disable
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode RejectConf();

    /**
    * @brief
    * [en] This interface is used to set guaranteed ip addresses of global domain.
    * [cn] 设置管理面域名对应的保底ip地址
    *
    * @detail
    * [en] This interface is used to set guaranteed ip addresses of global domain.
    * [cn] 该接口用于设置管理面域名对应的保底ip地址。
    *
    * @param guaranteedIpAddressesParam [type:HwmGuaranteedIpAddressesParam*]
    * [en] Guarantee ip Addresses
    * [cn] 保底ip地址信息
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetGlobalGuaranteedIpAddresses(const HwmGuaranteedIpAddressesParam* guaranteedIpAddressesParam);

    /**
    * @brief
    * [en] This interface is used to get self share state.
    * [cn] 获取自己的开启共享状态
    *
    * @detail
    * [en] This interface is used to get self share state.
    * [cn] 获取自己的开启共享状态
    *
    * @mac:transToSync
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetSelfShareState();

    /**
    * @brief 
    * [en] This interface is used to get login state.
    * [cn] 获取登录状态
    *
    * @detail
    * [en] This interface is used to get login state.
    * [cn] 该接口用于获取登录状态
    *
    * @mac:transToSync
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetLoginState();

    /**
    * @brief
    * [en] This interface is used to set avatar style.
    * [cn] 设置头像风格
    *
    * @detail
    * [en] This interface is used to set avatar style.
    * [cn] 该接口用于设置头像风格。
    *
    * @param avatarStyle [type:HwmAvatarStyle*]
    * [en] Avatar style
    * [cn] 头像风格
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetAvatarStyle(const HwmAvatarStyle* avatarStyle);

    /**
    * @brief [en] This interface is used to get token info.
    *        [cn] 获取登录token
    *
    * @mac:transToSync
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetLoginToken();

    /**
    * @brief
    * [en] This interface is used to operate subtitle.
    * [cn] 开启/关闭字幕
    *
    * @detail
    * [en] This interface is used to operate subtitle.
    * [cn] 该接口用于开启/关闭字幕
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode OpenSubtitle(bool isOpen);

    /**
    * @brief
    * [en] This interface is used to operate waiting room.
    * [cn] 开启/关闭等候室
    *
    * @detail
    * [en] This interface is used to operate waiting room.
    * [cn] 该接口用于开启/关闭等候室
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode OpenWaitingRoom(bool isOpen);

    /**
    * @brief
    * [en] This interface is used to operate local record.
    * [cn] 操作本地录制
    *
    * @detail
    * [en] This interface is used to operate local record.
    * [cn] 该接口用于操作本地录制
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode OperateLocalRecord(HwmLocalRecordOperateType type);

    /**
    * @brief
    * [en] This interface is used to open virtual background setting page.
    * [cn] 打开虚拟背景设置
    *
    * @detail
    * [en] This interface is used to open virtual background setting page.
    * [cn] 该接口用于打开虚拟背景设置
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode OpenVirtualBackgroundSettingPage();

    /**
    * @brief
    * [en] This interface is used to get local record status.
    * [cn] 获取本地录制状态
    *
    * @detail
    * [en] This interface is used to get local record status.
    * [cn] 该接口用于获取本地录制状态
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetConfLocalRecordStatus();

    /**
    * @brief
    * [en] This interface is used to get local record permission.
    * [cn] 获取本地录制权限
    *
    * @detail
    * [en] This interface is used to get local record permission.
    * [cn] 该接口用于获取本地录制权限
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetSelfHasLocalRecordPermission();

    /**
    * @brief
    * [en] This interface is used to request local record permission.
    * [cn] 申请本地录制权限
    *
    * @detail
    * [en] This interface is used to request local record permission.
    * [cn] 该接口用于申请本地录制权限
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode RequestLocalRecordPermission();

    /**
    * @brief
    * [en] This interface is used to request host role.
    * [cn] 申请主持人
    *
    * @detail
    * [en] This interface is used to request host role.
    * [cn] 该接口用于申请主持人
    *
    * @param password [type:char*]
    * [en] host password
    * [cn] 主持人密码
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode ApplyHostRole(const char* password);

    /**
    * @brief
    * [en] This interface is used to get subtitle state.
    * [cn] 获取字幕状态
    *
    * @detail
    * [en] This interface is used to get subtitle state.
    * [cn] 获取字幕状态
    *
    * @mac:transToSync
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetSubtitlePageOpened();

    /**
    * @brief
    * [en] This interface is used to get waiting room state.
    * [cn] 获取等候室状态
    *
    * @detail
    * [en] This interface is used to get waiting room state.
    * [cn] 获取等候室状态
    *
    * @mac:transToSync
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetWaitingRoomOpened();

    /**
    * @brief
    * [en] This interface is used to get virtual background setting page state.
    * [cn] 获取是否已经打开虚拟背景设置
    *
    * @detail
    * [en] This interface is used to get virtual background setting page state.
    * [cn] 获取是否已经打开虚拟背景设置
    *
    * @mac:transToSync
    **/
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetVirtualBackgroundSettingPageOpened();

    /**
    * @brief
    * [en]set virtual background default images
    * [cn]设置虚拟背景默认图片
    *
    * @detail
    * [en]This interface is used to set virtual background default images
    * [cn]该接口用于设置虚拟背景默认图片
    *
    * @param info [type:HwmVirtualBackgroundInfo*][cs_allowNull:N]
    * [en]Virtual background info.
    * [cn]虚拟背景信息
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SetVirtualBackgroundDefaultImages(const HwmVirtualBackgroundInfo* defaultImagesInfo);

    /**
    * @brief
    * [en] This interface is used to subscribe subtitle language.
    * [cn] 订阅字幕语言
    *
    * @detail
    * [en] This interface is used to subscribe subtitle languague.
    * [cn] 该接口用于订阅字幕语言
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SubscribeSubtitleLanguage(HwmTargetLanguageType targetLanguageType);

    /**
    * @brief
    * [en] This interface is used to subscribe subtitle.
    * [cn] 订阅/取消订阅字幕
    *
    * @detail
    * [en] This interface is used to subscribe subtitle.
    * [cn] 该接口用于订阅/取消订阅字幕
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode SubscribeSubtitle(bool isSubscribe);

    /**
    * @brief
    * [en] This interface is used to get subtitle support languagues.
    * [cn] 获取字幕支持的语言
    *
    * @detail
    * [en] This interface is used to get subtitle support languagues.
    * [cn] 该接口用于获取字幕支持的语言
    *
    * @mac:transToSync
    */
    HWM_SDK_AGENT_API hwmsdk::HwmErrCode GetSubtitleSupportLanguage();

/* auto generate code insert, forbit delete */
}

#ifdef __cplusplus
}
#endif

#endif /* _HWM_SDK_AGENT_H_ */

