﻿/**
 * @file HwmAgentDef.h
 *
 * Copyright(C), 2019-2020, Huawei Tech. Co., Ltd. ALL RIGHTS RESERVED.
 *
 * @brief [en] define file.
 *        [cn] 结构体、枚举定义文件
 */

#ifndef _HWM_AGENT_DEF_H_
#define _HWM_AGENT_DEF_H_

#include "HwmAgentCallback.h"
#include "HwmAgentNotify.h"


namespace hwmsdkagent
{
    /**
    * [en]Log level.
    * [cn]日志级别
    */
    typedef enum tagHwmLogLevel
    {
        HWM_LOG_LEVEL_INFO = 0,                    /**< [en]Info level
                                                   [cn]信息（一般）级别 */
        HWM_LOG_LEVEL_WARNING,                     /**< [en]Warning level
                                                   [cn]警告级别 */
        HWM_LOG_LEVEL_ERROR,                       /**< [en]Info level
                                                   [cn]错误级别 */
    } HwmLogLevel;

    /** 
     * [en]Init info.
     * [cn]初始化信息
     */
    typedef struct tagHwmInitInfo
    {
        HwmAgentNotify* notify;                            /**< [en]Notify object
                                                                [cn]通知对象 */
        HwmAgentCallback* callback;                        /**< [en]Callback object
                                                                [cn]回调对象 */
        char exePath[HWM_MAX_FILE_PATH_LEN];               /**< [en]The path of the exe file
                                                                [cn]执行文件路径 */
        char logPath[HWM_MAX_FILE_PATH_LEN];               /**< [en]The path of the log file
                                                                [cn]日志文件路径 */
        char userDataPath[HWM_MAX_FILE_PATH_LEN];          /**< [en]The path of the user data file
                                                                [cn]用户数据文件路径 */
        char appId[HWM_MAX_APPID_LEN];                     /**< [en]The app identity
                                                                [cn]应用程序的唯一标识 */
        HwmAuthType authType;                              /**< [en]The auth type
                                                                [cn]鉴权方式 */
        HwmSiteType siteType;                              /**< [en]The site type
                                                                [cn]站点类型 */
        unsigned int logKeepDays;                          /**< [en]Indicates the log keeping days.
                                                                [cn]配置项-日志保存天数，0为不按天数删除日志，非0情况范围3-30 */
        char preferredCamera[HWM_MAX_PREFERRED_CAMERA_LEN];/**< [en]preferred camera
                                                                [cn]优选摄像头列表，填写后，初次打开摄像头按列表中的顺序选择第一个可用的摄像头。
                                                                    可以不填，不填按照默认顺序打开可用的摄像头*/
        bool isSupportMultiAppLogin;                       /**< [en]is support multi client login.
                                                                [cn]是否支持同终端多应用登录 */
        char workSpacePluginDownloadUrl[HWM_MAX_URL_LEN];  /**< [en]WorkSpace Plugin Download Url.
                                                                [cn]云桌面插件的下载地址配置  */
        HwmLogLevel logLevel;                              /**< [en]Indicates log level。
                                                                [cn]日志级别 */
        char appVersion[HWM_MAX_APPVERSION_LEN];           /**< [en]Indicates app version。
                                                                [cn]应用版本号 */
    }HwmInitInfo;

    /** 
     * [en]Account and password auth info.
     * [cn]账号密码登录鉴权信息
     */
    typedef struct tagHwmAccountAndPasswordAuthInfo
    {
        char account[HWM_MAX_ACCOUNT_LEN];          /**< [en]The account
                                                         [cn]账号 */
        char password[HWM_MAX_PASSWORD_LEN];        /**< [en]The password
                                                         [cn]密码 */
    }HwmAccountAndPasswordAuthInfo;

    /**
    * [en]appId auth info.
    * [cn]appId登录鉴权信息
    */
    typedef struct tagHwmAppIdAuthInfo
    {
        char corpId[HWM_MAX_CORP_ID_LEN];          /**< [en]Corporation Id
                                                        [cn]企业Id*/
        char thirdUserId[HWM_MAX_USER_ID_LEN];     /**< [en]The user id
                                                        [cn]第三方账号唯一标识 */
        char userName[HWM_MAX_USER_NAME_LEN];      /**< [en]The user name
                                                        [cn]用户名称 */
        char signature[HWM_MAX_SIGNATURE_LEN];     /**< [en]The signature
                                                        [cn]鉴权信息 */
        char nonce[HWM_APPID_NONCE_LEN];           /**< [en]The nonce
                                                        [cn]随机字符串，用于计算鉴权信息 */
        long long expireTime;                      /**< [en]The expire time(s)
                                                        [cn]鉴权信息时间戳（s） */
        char email[HWM_MAX_EMAIL_LEN];             /**< [en]The nonce
                                                        [cn]邮箱信息 */
        char phoneNumber[HWM_MAX_PHONE_NUM_LEN];   /**< [en]The nonce
                                                        [cn]手机号码 */
        char deptCode[HWM_MAX_CORPID_LEN];         /**< [en]Indicates the deptCode.
                                                        [cn]部门id */
    }HwmAppIdAuthInfo;

    /** 
     * [en]Account and token auth info.
     * [cn]账号token登录鉴权信息
     */
    typedef struct tagHwmAccountAndTokenAuthInfo
    {
        char account[HWM_MAX_USERNAME_LEN];              /**< [en]The account
                                                              [cn]账号 */
        char token[HWM_MAX_TOKEN_LEN];                   /**< [en]The token
                                                              [cn]token */
    }HwmAccountAndTokenAuthInfo;

    /** 
     * [en]Login info.
     * [cn]登录信息
     */
    typedef struct tagHwmLoginInfo
    {
        HwmAuthType authType;                                            /**< [en]The auth type
                                                                              [cn]鉴权方式 */
        HwmAccountAndPasswordAuthInfo accountAndPasswordAuthInfo;        /**< [en]The account and password info
                                                                              [cn]账号密码鉴权信息 */
        HwmAppIdAuthInfo appIdAuthInfo;                                  /**< [en]AppId info
                                                                              [cn]appId鉴权信息 */
    }HwmLoginInfo;


    /**
     * [en]conference resource type
     * [cn]会议资源类型
     */
    typedef enum tagHwmConfResType
    {
        HWM_CONF_RESTYPE_DEFAULT = 0,                 /**< [en]default
                                                           [cn]默认*/
        HWM_CONF_RESTYPE_SHARE_VMR = 3,               /**< [en]share vmr
                                                           [cn]共享vmr*/
    }HwmConfResType;

    /** 
     * [en]Creat conf info.
     * [cn]创会信息
     */
    typedef struct tagHwmCreateConfInfo
    {
        char subject[HWM_MAX_SUBJECT_LEN];              /**< [en]The conf suject
                                                             [cn]会议主题 */
        HwmConfMediaType mediaType;                     /**< [en]The media type
                                                             [cn]会议类型 */
        bool needPassword;                              /**< [en]Is need password
                                                             [cn]是否需要密码 */
        bool isAutoRecord;                              /**< [en]Is autorecord
                                                             [cn]是否开启自动录制 */
        bool allowRecord;                               /**< [en]Is allow record
                                                             [cn]是否允许录制 */
        HwmJoinConfPermissionType callInRestriction;    /**< [en]Call in Restriction type
                                                             [cn]呼入限制 */
        char vmrId[HWM_MAX_VMR_CONF_ID_LEN];            /**< [en]The vmr conf Id
                                                             [cn]VMR会议ID */
        bool supportTerminal;                           /**< [en]Indicates whether is support invite Terminal.
                                                             [cn]是否邀请硬终端入会 */
        HwmVmrConfIdType vmrConfIdType;                 /**< [en]VMR conf Id type.
                                                             [cn]云会议室的会议ID类型 */
        char guestPwd[HWM_MAX_CONF_PASSWORD_LEN];       /**< [en]Indicates the password for common participants.If the password is null, the server will generate a password randomly.
                                                             [cn]普通与会者密码，如果空则服务器随机生成 */
        bool isOpenWaitingRoom;                         /**< [en]Indicates whether is open waiting room. 
                                                             [cn]是否开启等候室 */
        int duration;                                   /**< [en]Indicates the duration of the conference. (minute)
                                                             [cn]会议时长（单位：分钟） */
        unsigned int concurrentParticipants;            /**< [en]concurrent participants.
                                                             [cn]会议方数，会议最大与会人数限制. 0: 无限制, 大于0: 会议最大与会人数*/
        char applicationsAttribute[HWM_MAX_APP_ATTRIBUTE_LEN];    /**< [en]Indicates appServer attribute
                                                                       [cn]第三方应用参数属性 */
        char customInfo[HWM_MAX_CUSTOM_INFO_LEN];       /**< [en]Indicates User-defined extended information
                                                             [cn]自定义扩展信息 */
        HwmConfResType confResType;                     /**< [en]Indicates conf resource type
                                                             [cn]会议资源类型 */
        HwmSummaryState defaultSummaryState;            /**< [en]Indicates record include summary by default
                                                             [cn]会议云录制是否包含纪要，初始值 */
        HwmAutoMuteType autoMuteMode;                   /**< [en]Indicates is auto mute of soft terminal
                                                             [cn]来宾入会，软终端是否自动静音 */
        HwmAutoMuteType hardTerminalAutoMuteMode;       /**< [en]Indicates is auto mute of hard terminal
                                                             [cn]来宾入会，硬终端是否自动静音 */
        bool supportMultiShare;                         /**< [en]Indicates support multi-share.
                                                             [cn]是否支持多人共享 */
        bool supportWatermark;                          /**< [en]Indicates whether is support watermark.
                                                             [cn]是否支持水印 */
        bool forbiddenScreenShots;                      /**< [en]Indicates is whether disable screenshots
                                                             [cn]是否禁止截屏 */
        bool autoPublishSummary;                        /**< [en]Indicates Whether auto publish summary. 
                                                             [cn]是否自动发布纪要（免人工审核）*/
        bool supportDataGuardAudio;                     /**< [en]Indicates whether is support data guard audio.
                                                             [cn]是否支持音频暗水印 */
        bool supportDataGuardVideo;                     /**< [en]Indicates whether is support data guard video.
                                                             [cn]是否支持视频暗水印 */
    }HwmCreateConfInfo;

    /** 
     * [en]Monitor Share info.
     * [cn]屏幕共享信息
     */
    typedef struct tagHwmMonitorShareInfo
    {
        HwmMonitor monitor;           /**< [en]Type of monitor
                                           [cn]屏幕类型，1-主屏幕，2-副屏幕，mode为主副屏模式下生效
                                           [cs_allowNull:N] */
        HwmMonitorMode monitorMode;   /**< [en]Mode of monitor
                                           [cn]屏幕模式，0-主副屏模式，1-多屏模式
                                           [cs_allowNull:N] */
        unsigned int monitorIndex;    /**< [en]Index of monitor
                                           [cn]屏幕索引，mode为多屏模式下生效
                                           [cs_allowNull:N] */
    }HwmMonitorShareInfo;

     /** 
     * [en]App share info.
     * [cn]程序共享信息
     */
    typedef struct tagHwmAppShareInfo
    {
        HWND appHandle;               /**< [en]App window handle
                                           [cn]程序窗口句柄 */
    }HwmAppShareInfo;

    /** 
     * [en]Start share info.
     * [cn]开启共享信息
     */
    typedef struct tagHwmStartShareInfo
    {
        HwmShareType shareType;                     /**< [en]Share type
                                                         [cn]共享类型 */
        HwmMonitorShareInfo monitorShareInfo;       /**< [en]Monitor share info
                                                         [cn]屏幕共享信息 */   
        HwmAppShareInfo appShareInfo;               /**< [en]App Share info
                                                         [cn]程序共享信息 */ 
        HwmMonitorShareInfo operatableUiMonitor;    /**< [en]UI Control Monitor
                                                         [cn]可操作UI所在屏幕（会控条、弹框等） */
    }HwmStartShareInfo;

    /** 
     * [en]Join conf info.
     * [cn]会议id入会信息
     */
    typedef struct tagHwmJoinConfByIdInfo
    {
        char confId[HWM_MAX_CONF_ID_LEN];                /**< [en]Conf id
                                                              [cn]会议id */
        char password[HWM_MAX_CONF_PASSWORD_LEN];        /**< [en]Conf password
                                                              [cn]会议密码 */
        char name[HWM_MAX_DISPLAY_NAME_LEN];             /**< [en]name
                                                              [cn]会场名 */
        char aliasCN[HWM_MAX_DISPLAY_NAME_LEN];          /**< [en]Indicates chinese alias.
                                                              [cn]中文别名 */
        char aliasEN[HWM_MAX_DISPLAY_NAME_LEN];          /**< [en]Indicates english alias.
                                                              [cn]英文别名 */
        char extendedField[HWM_MAX_EXTENDED_FIELD_LEN];  /**< [en]Indicates extended field.
                                                              [cn]扩展字段 */
        char corpTrustAppId[HWM_MAX_CORP_TRUST_APP_ID_LEN];  /**< [en]Indicates corp trust app id.
                                                                  [cn]企业互信应用ID */
        bool isCloseSpk;                                     /**< [en]is close loudspeaker
                                                                  [cn]是否关闭扬声器 */
        bool isStopConflictConf;                             /**< [en]Indicates whether stop confilct conf.
                                                                  [cn]强制结束和当前VMR会议资源冲突的会议（必需是vmr资源的拥有者才有权限，否则无效） */
        bool isCascadingMeeting;                        /**< [en]Indicates Cascading Meeting. (Anonymous is not supported; Only for Windows).
                                                             [cn]是否级联会议（不支持匿名入会，仅对Windows平台生效） */
        char confPrefix[HWM_MAX_CONFID_PREFIX_LEN];     /**< [en]Indicates settled conf id (Only for Windows).
                                                             [cn]入驻式会议企业前缀（仅对Windows平台生效） */
    }HwmJoinConfByIdInfo;

    /**
    * [en]Join conf by random info.
    * [cn]会议random入会信息
    */
    typedef struct tagHwmJoinConfByRandomInfo
    {
        char confId[HWM_MAX_CONF_ID_LEN];                    /**< [en]Conf id
                                                                  [cn]会议id */
        char name[HWM_MAX_DISPLAY_NAME_LEN];                 /**< [en]name
                                                                  [cn]会场名 */
        char random[HWM_MAX_RANDOM_LEN];                     /**< [en]random
                                                                  [cn]会议链接后48位随机数*/
        char domain[HWM_MAX_DOMAIN_LEN];                     /**< [en]domain.
                                                                  [cn]域名 */
        bool isCloseSpk;                                     /**< [en]is open loudspeaker
                                                                  [cn]是否关闭扬声器*/
    }HwmJoinConfByRandomInfo;

    /** 
     * [en]Join conf info.
     * [cn]会议链接入会信息
     */
    typedef struct tagHwmJoinConfByLinkInfo
    {
        char siteUrl[HWM_MAX_URL_LEN];                      /**< [en]Conf link
                                                                 [cn]根据链接解析出的服务器地址 */
        char random[HWM_MAX_CONF_PASSWORD_LEN];             /**< [en]Conf random
                                                                 [cn]根据链接解析出的随机数 */
        char name[HWM_MAX_DISPLAY_NAME_LEN];                /**< [en]name
                                                                 [cn]会场名 */
        char aliasCN[HWM_MAX_DISPLAY_NAME_LEN];             /**< [en]Indicates chinese alias.
                                                                 [cn]中文别名 */
        char aliasEN[HWM_MAX_DISPLAY_NAME_LEN];             /**< [en]Indicates english alias.
                                                                 [cn]英文别名 */
        char extendedField[HWM_MAX_EXTENDED_FIELD_LEN];     /**< [en]Indicates extended field.
                                                                 [cn]扩展字段 */
        bool isCloseSpk;                                    /**< [en]Is close speaker
                                                                 [cn]是否关闭扬声器 */
        char idPrefix[HWM_MAX_CONFID_PREFIX_LEN];           /**< [en]prefix of conf id
                                                                 [cn]会议id所属站点的前缀 */
        char confId[HWM_MAX_CONF_ID_LEN];                   /**< [en]Conf id
                                                                 [cn]会议id, 离线入会时使用 */
        bool isOfflineJoin;                                 /**< [en]Indicates is offline join conf.
                                                                 [cn]是否离线入会 */
    }HwmJoinConfByLinkInfo;

    /**
     * [en]Conference attendee type.
     * [cn]会议用户类型
     */
    typedef enum tagHwmConfAttendeeType
    {
        HWM_ATTENDEE_TYPE_NORMAL = 0,                               /**< [en]Normal
                                                                         [cn]会议软终端用户 */
        HWM_ATTENDEE_TYPE_MOBILE = 5,                               /**< [en]Mobile
                                                                         [cn]用户手机号码  */
    } HwmConfAttendeeType;

    /**
    * [en]Btn pos info.
    * [cn]按钮位置信息
    */
    typedef struct tagHwmBtnPosInfo
    {
        int left;                                 /**< [en]The left coordinates
                                                  [cn]左侧坐标 */
        int top;                                  /**< [en]The top coordinates
                                                  [cn]上侧坐标 */
        unsigned int width;                       /**< [en]The width
                                                  [cn]宽度 */
        unsigned int height;                      /**< [en]The height
                                                  [cn]高度 */
    }HwmBtnPosInfo;

    /** 
     * [en]Conf attendee info.
     * [cn]与会者信息
     */
    typedef struct tagHwmConfAttendee
    {
        char name[HWM_MAX_DISPLAY_NAME_LEN];              /**< [en]The attendee's name
                                                               [cn]与会者名称 */
        char aliasCN[HWM_MAX_DISPLAY_NAME_LEN];           /**< [en]Indicates chinese alias.
                                                               [cn]中文别名（获取会中与会者信息使用） */
        char aliasEN[HWM_MAX_DISPLAY_NAME_LEN];           /**< [en]Indicates english alias.
                                                               [cn]英文别名（获取会中与会者信息使用） */
        char extendedField[HWM_MAX_EXTENDED_FIELD_LEN];   /**< [en]Indicates extended field.
                                                               [cn]扩展字段 */
        char number[HWM_MAX_NUMBER_LEN];                  /**< [en]The attendee's number
                                                               [cn]与会者号码 */
        char thirdUserId[HWM_MAX_USER_ID_LEN];            /**< [en]The attendee's third account
                                                               [cn]与会者三方账号，appId登录时使用*/
        char accountId[MAX_ACCOUNT_ID_LEN];               /**< [en]Indicates account id.
                                                               [cn]可选，华为云会议账号 */
        char userUuid[HWM_MAX_ACCOUNT_ID_LEN];            /**< [en]Indicates account uuid.
                                                               [cn]与会者uuid */
        HwmConfAttendeeType type;                         /**< [en]Indicates conference attendee type.
                                                               [cn]与会者类型 */
        HwmBtnPosInfo posInfo;                            /**< [en]Buttton pos info
                                                               [cn]按钮位置信息 */
        char orgId[HWM_MAX_ORG_ID_LENGTH];                /**< [en]Indicates org id.
                                                               [cn]企业id */
        HwmAutoInviteType autoInviteType;                 /**< [en]Indicates auto invite type.
                                                               [cn]自动邀请类型（仅创建会议生效） */
    }HwmConfAttendee;

    /**
    * [en]Proxy config policy
    * [cn]代理配置策略
    */
    typedef enum tagHwmProxyConfigPolicy
    {
        HWM_PROXY_CONFIG_POLICY_CUSTOM,                    /**< [en]Use proxy with custom configuration
                                                                [cn]使用具有自定义配置的代理 */
        HWM_PROXY_CONFIG_POLICY_OFF,                       /**< [en]Do not use proxy
                                                                [cn]不使用代理 */
        HWM_PROXY_CONFIG_POLICY_AUTO,                      /**< [en]Use system proxy
                                                                [cn]使用系统代理 */
    }HwmProxyConfigPolicy;

    /** 
     * [en]Proxy info.
     * [cn]代理信息
     */
    typedef struct tagHwmProxyInfo
    {
        HwmProxyConfigPolicy proxyConfigPolicy;              /**< [en]The proxy config policy
                                                                  [cn]代理配置策略 */
        char server[HWM_MAX_URL_LEN];                        /**< [en]The server address
                                                                  [cn]代理地址 */
        unsigned short port;                                 /**< [en]The server port
                                                                  [cn]代理端口 */
        char account[HWM_MAX_USERNAME_LEN];                  /**< [en]The account
                                                                  [cn]代理账号 */
        char password[HWM_MAX_PASSWORD_LEN];                 /**< [en]The password
                                                                  [cn]代理密码 */
    }HwmProxyInfo;

    /**
     * [en]This structure is used to describe login parameters.
     * [cn]登录信息参数
     */
    typedef struct tagHwmServerInfo
    {
        char serverAddr[HWM_MAX_URL_LEN];               /**< [en]Indicates the server address.
                                                             [cn]服务器地址 */
        unsigned short serverPort;                      /**< [en]The server port
                                                             [cn]服务器端口 */
    }HwmServerInfo;

    /**
     * [en]This structure is used to describe deploy parameters.
     * [cn]部署信息参数
    */
    typedef struct tagHwmDeployInfo
    {
        HwmServerInfo serverInfo;    /**< [en]Indicates the server info.
                                          [cn]服务器地址和端口信息 */
        HwmDeployType deployType;    /**< [en]Indicates the deploy mode.
                                          [cn]部署模式 */
    }HwmDeployInfo;

     /** 
     * [en]Book conf parameter.
     * [cn]预约会议参数
     */
    typedef struct tagHwmBookConfParam
    {
        char subject[HWM_MAX_SUBJECT_LEN];                      /**< [en]Indicates the conference subject.
                                                                     [cn]会议主题 */
        long long startTime;                                    /**< [en]Indicates the begin time of the conference. Timestamp(second)
                                                                     [cn]（已废弃，推荐使用字段startTimeStamp）会议开始时间，时间戳（单位：秒） */
        long long startTimeStamp;                               /**< [en]Indicates conference start time
                                                                     [cn]会议开始时间，startTimeStamp、startTime二选一，优先使用startTimeStamp(时间戳，单位秒) */
        int duration;                                           /**< [en]Indicates the duration of the conference. (minute)
                                                                     [cn]会议时长（单位：分钟） */
        HwmConfMediaType confType;                              /**< [en]Indicates the type of the conference.
                                                                     [cn]会议类型 */
        bool isNeedConfPwd;                                     /**< [en]Indicates whether guest password is needed entering the conference.
                                                                     [cn]来宾入会是否需要密码 */
        bool isAutoRecordOn;                                    /**< [en]Indicates whether taking a autorecord of the conference.
                                                                     [cn]会议是否开启自动录制 */
        bool isRecordOn;                                        /**< [en]Indicates whether allow taking a record of the conference.
                                                                     [cn]会议是否允许开启录制 */
        int timeZone;                                           /**< [en]Indicates the time zone used in conference notification.
                                                                     [cn]时区码 */
        bool vmrFlag;                                           /**< [en]Indicates whether vmr is used.
                                                                     [cn]（废弃）是否使用云会议室 */
        char vmrId[HWM_MAX_VMR_CONF_ID_LEN];                    /**< [en]Indicates the vmr if vmr used.
                                                                     [cn]云会议室id */           
        HwmJoinConfPermissionType joinConfRestrictionType;      /**< [en]Indicates the admission of the conference.
                                                                     [cn]会议接入限制 */ 
        bool isSmsOn;                                           /**< [en]Indicates if sms notification sent.
                                                                     [cn]是否发送短信通知 */
        bool isMailOn;                                          /**< [en]Indicates if email notification sent.
                                                                     [cn]是否发送邮件通知 */ 
        bool isEmailCalendarOn;                                 /**< [en]Indicates if calendar notification sent.
                                                                     [cn]是否发送日历通知 */ 
        HwmAttendeeInfo* attendees;                             /**< [en]Indicates attendees to the conference.
                                                                     [cn]与会者列表 */ 
        unsigned int attendeeLen;                               /**< [en]Indicates the size of the attendee list.
                                                                     [cn]与会者列表长度 */ 
        bool supportTerminal;                                   /**< [en]Indicates whether is support invite Terminal.
                                                                     [cn]是否邀请硬终端入会 */
        HwmVmrConfIdType vmrConfIdType;                         /**< [en]VMR conf Id type.
                                                                     [cn]云会议室的会议ID类型 */
        char guestPwd[HWM_MAX_CONF_PASSWORD_LEN];               /**< [en]Indicates the password for common participants.If the password is null, the server will generate a password randomly.
                                                                     [cn]普通与会者密码，如果空则服务器随机生成 */
        bool isOpenWaitingRoom;                                 /**< [en]Indicates whether is open waiting room.
                                                                     [cn]是否开启等候室 */
        bool allowGuestStartConf;                               /**< [en]Indicates whether is allow guest start conf.
                                                                     [cn]是否允许来宾提前启动会议*/
        bool forbiddenScreenShots;                              /**< [en] disable screenshots
                                                                     [cn] 是否禁止截屏 */
        unsigned int allowGuestStartConfTime;                     /**< [en]Indicates join before host time.
                                                                     [cn]允许来宾提前入会时间范围(单位：分钟) 0-随时，n-提前n分钟启动会议*/
        char applicationsAttribute[HWM_MAX_APP_ATTRIBUTE_LEN];  /**< [en]Indicates appServer attribute
                                                                     [cn]第三方应用参数属性 */
        unsigned int concurrentParticipants;                    /**< [en]concurrent participants.
                                                                     [cn]会议方数，会议最大与会人数限制. 0: 无限制, 大于0: 会议最大与会人数*/
        char customInfo[HWM_MAX_CUSTOM_INFO_LEN];                   /**< [en]Indicates User-defined information
                                                                     [cn]用户自定义信息 */
        HwmConfResType confResType;                             /**< [en]Indicates conf resource type
                                                                     [cn]会议资源类型 */
        HwmSummaryState defaultSummaryState;                    /**< [en]Indicates record include summary by default
                                                                     [cn]会议云录制是否包含纪要，初始值（周期会议暂不支持）*/
        HwmAutoMuteType autoMuteMode;                           /**< [en]Indicates is auto mute of soft terminal
                                                                     [cn]来宾入会，软终端是否自动静音 */
        HwmAutoMuteType hardTerminalAutoMuteMode;               /**< [en]Indicates is auto mute of hard terminal
                                                                     [cn]来宾入会，硬终端是否自动静音 */
        bool supportMultiShare;                                 /**< [en]Indicates support multi-share.
                                                                     [cn]是否支持多人共享 */
        bool supportWatermark;                                  /**< [en]Indicates whether is support watermark.
                                                                     [cn]是否支持水印 */
        bool autoPublishSummary;                                /**< [en]Indicates Whether auto publish summary. 
                                                                     [cn]是否自动发布纪要（免人工审核）*/
        bool attendeesExcludeSelf;                              /**< [en]Indicates does the attendee list exclude self.
                                                                     [cn]与会者列表是否排除自己 */
        bool supportDataGuardAudio;                             /**< [en]Indicates whether is support data guard audio.
                                                                     [cn]是否支持音频暗水印 */
        bool supportDataGuardVideo;                             /**< [en]Indicates whether is support data guard video.
                                                                     [cn]是否支持视频暗水印 */
    } HwmBookConfParam;

    /** 
     * [en]Edit(modify) conf parameter.
     * [cn]编辑（修改）已经预约的会议
     */
    typedef struct tagHwmEditConfParam
    {
        char confId[HWM_MAX_CONF_ID_LEN];                       /**< [en]Indicates the conference id.
                                                                     [cn]会议id */
        char subject[HWM_MAX_SUBJECT_LEN];                      /**< [en]Indicates the conference subject.
                                                                     [cn]会议主题 */
        long long startTime;                                    /**< [en]Indicates the begin time of the conference. Timestamp(second)
                                                                     [cn]（已废弃，推荐使用字段startTimeStamp）会议开始时间，时间戳（单位：秒） */
        long long startTimeStamp;                               /**< [en]Indicates conference start time
                                                                     [cn]会议开始时间，startTimeStamp、startTime二选一，优先使用startTimeStamp(时间戳，单位秒) */
        int duration;                                           /**< [en]Indicates the duration of the conference. (minute)
                                                                     [cn]会议时长（单位：分钟） */
        bool vmrFlag;                                           /**< [en]Indicates whether vmr is used.
                                                                     [cn]（废弃）是否使用云会议室 */
        char vmrId[HWM_MAX_VMR_CONF_ID_LEN];                    /**< [en]Indicates the vmr if vmr used.
                                                                     [cn]云会议室id */   
        HwmConfMediaType confType;                              /**< [en]Indicates the type of the conference.
                                                                     [cn]会议类型 */
        bool isNeedConfPwd;                                     /**< [en]Indicates whether guest password is needed entering the conference.
                                                                     [cn]来宾入会是否需要密码 */
        int timeZone;                                           /**< [en]Indicates the time zone used in conference notification.
                                                                     [cn]时区码 */
        HwmJoinConfPermissionType joinConfRestrictionType;      /**< [en]Indicates the admission of the conference.
                                                                     [cn]会议接入限制 */ 
        bool isAutoRecordOn;                                    /**< [en]Indicates whether taking a autorecord of the conference.
                                                                     [cn]会议是否开启自动录制 */
        bool isRecordOn;                                        /**< [en]Indicates whether allow taking a record of the conference.
                                                                     [cn]会议是否允许开启录制 */
        bool isLiveOn;                                          /**< [en]Reserved.This field is used to transparently transmit conference details when editing a conference.
                                                                     [cn]保留。编辑会议时透传会议详情此字段 */
        bool isSmsOn;                                           /**< [en]Indicates if sms notification sent.
                                                                     [cn]是否发送短信通知 */
        bool isMailOn;                                          /**< [en]Indicates if email notification sent.
                                                                     [cn]是否发送邮件通知 */ 
        bool isEmailCalendarOn;                                 /**< [en]Indicates if calendar notification sent.
                                                                     [cn]是否发送日历通知 */ 
        HwmAttendeeInfo *attendees;                             /**< [en]Indicates attendees to the conference.
                                                                     [cn]与会者列表 */ 
        unsigned int attendeeLen;                               /**< [en]Indicates the size of the attendee list.
                                                                     [cn]与会者列表长度 */ 
        bool supportTerminal;                                   /**< [en]Indicates whether is support invite Terminal.
                                                                     [cn]是否邀请硬终端入会 */
        HwmVmrConfIdType vmrConfIdType;                         /**< [en]VMR conf Id type.
                                                                     [cn]云会议室的会议ID类型 */
        bool isOpenWaitingRoom;                                 /**< [en]Indicates whether is open waiting room.
                                                                     [cn]是否开启等候室 */
        bool allowGuestStartConf;                               /**< [en]Indicates whether is allow guest start conf.
                                                                     [cn]是否允许来宾提前启动会议*/
        bool forbiddenScreenShots;                              /**< [en] disable screenshots
                                                                     [cn] 是否禁止截屏 */
        unsigned int allowGuestStartConfTime;                     /**< [en]Indicates join before host time.
                                                                     [cn]允许来宾提前入会时间范围(单位：分钟) 0-随时，n-提前n分钟启动会议*/
        char applicationsAttribute[HWM_MAX_APP_ATTRIBUTE_LEN];  /**< [en]Indicates appServer attribute
                                                                     [cn]第三方应用参数属性 */
        unsigned int concurrentParticipants;                    /**< [en]concurrent participants.
                                                                     [cn]会议方数，会议最大与会人数限制. 0: 无限制, 大于0: 会议最大与会人数*/
        char customInfo[HWM_MAX_CUSTOM_INFO_LEN];                   /**< [en]Indicates User-defined extended information
                                                                     [cn]自定义扩展信息 */
        HwmSummaryState defaultSummaryState;                    /**< [en]Indicates record include summary by default
                                                                     [cn]会议云录制是否包含纪要，初始值（周期会议暂不支持 */
        HwmAutoMuteType autoMuteMode;                           /**< [en]Indicates is auto mute of soft terminal
                                                                     [cn]来宾入会，软终端是否自动静音 */
        HwmAutoMuteType hardTerminalAutoMuteMode;               /**< [en]Indicates is auto mute of hard terminal
                                                                     [cn]来宾入会，硬终端是否自动静音 */
        bool supportMultiShare;                                 /**< [en]Indicates support multi-share.
                                                                     [cn]是否支持多人共享 */
        bool supportWatermark;                                  /**< [en]Indicates whether is support watermark.
                                                                     [cn]是否支持水印 */
        bool autoPublishSummary;                                /**< [en]Indicates Whether auto publish summary.
                                                                     [cn]是否自动发布纪要（免人工审核）*/
        bool attendeesExcludeSelf;                              /**< [en]Indicates does the attendee list exclude self.
                                                                     [cn]与会者列表是否排除自己 */
        bool supportDataGuardAudio;                             /**< [en]Indicates whether is support data guard audio.
                                                                     [cn]是否支持音频暗水印 */
        bool supportDataGuardVideo;                             /**< [en]Indicates whether is support data guard video.
                                                                     [cn]是否支持视频暗水印 */
    } HwmEditConfParam;

    typedef struct tagChangeVmrInfoParam
    {
        char vmrId[HWM_MAX_VMR_ID_LEN];                         /**< [en]Indicates the id of the vmr.
                                                                     [cn]云会议室id */
        char vmrName[HWM_MAX_VMR_NAME_LEN];                     /**< [en]Indicates the name of the vmr.
                                                                     [cn]云会议室名称 */
        char guestPwd[HWM_MAX_PASSWORD_LEN];                    /**< [en]Indicates the guest password of the vmr.
                                                                     [cn]云会议室来宾密码 */
        char chairPwd[HWM_MAX_PASSWORD_LEN];                    /**< [en]Indicates the host password of the vmr.
                                                                     [cn]云会议室主席密码 */
        bool allowGuestStartConf;                               /**< [en]Indicates whether is allow guest start conf.
                                                                     [cn]是否允许来宾提前启动会议*/
        HwmJoinConfPermissionType joinConfRestrictionType;      /**< [en]Indicates the user type allowed to join the conference.
                                                                     [cn]允许入会范围限制。默认所有用户。*/
        bool isOpenWaitingRoom;                                 /**< [en]Indicates whether is open waiting room
                                                                     [cn]等候室开启开关。
                                                                     说明：需要等候室功能开通后才能生效*/
    } ChangeVmrInfoParam;

    typedef struct tagWaterMarkInfo
    {
        char content[HWM_MAX_WATERMARK_CONTENT_LEN];    //用户自定义水印注入内容
        HwmWaterMarkColorType color;                    /**< [en]WaterMark font color
                                                             [cn]水印字体颜色，0-浅色，1-标准色 #，2-深色 */
    } HwmWaterMarkInfo;

    /**
    * [en]Call contact info
    * [cn]呼叫人信息
    */
    typedef struct tagHwmCallContactInfo
    {
        char name[HWM_MAX_DISPLAY_NAME_LEN];                    /**< [en]The call contact's name
                                                                     [cn]呼叫人名称 */
        char number[HWM_MAX_NUMBER_LEN];                        /**< [en]The call contact's number
                                                                     [cn]呼叫人号码 */
        char thirdUserId[HWM_MAX_USER_ID_LEN];                  /**< [en]The call contact's third account
                                                                     [cn]呼叫人三方账号，appId鉴权时使用*/
        char userUuid[HWM_MAX_ACCOUNT_ID_LEN];                  /**< [en]The call contact's uuid.
                                                                     [cn]呼叫人uuid */
    } HwmCallContactInfo;

    /**
    * [en]Start call info
    * [cn]发起呼叫信息
    */
    typedef struct tagHwmStartCallInfo
    {
        char name[HWM_MAX_USER_NAME_LEN];                       /**< [en]caller name
                                                                     [cn]主叫名称 */
        HwmCallContactInfo callee;                              /**< [en]callee info
                                                                     [cn]被叫信息 */
        HwmCallType callType;                                   /**< [en]call type
                                                                     [cn]呼叫类型 */
    } HwmStartCallInfo;

    /**
    * [en]Caller info
    * [cn]主叫信息
    */
    typedef struct tagHwmCallerInfo
    {
        char nickName[HWM_MAX_DISPLAY_NAME_LEN];                    /**< [en]Caller nickName
                                                                         [cn]主叫姓名
                                                                         [cs_allowNull:Y] */
        char aliasCN[HWM_MAX_DISPLAY_NAME_LEN];                     /**< [en]Indicates chinese alias.
                                                                         [cn]中文别名（获取会中与会者信息使用）
                                                                         [cs_allowNull:Y] */
        char aliasEN[HWM_MAX_DISPLAY_NAME_LEN];                     /**< [en]Indicates english alias.
                                                                         [cn]英文别名（获取会中与会者信息使用）
                                                                         [cs_allowNull:Y] */
        char extendedField[HWM_MAX_EXTENDED_FIELD_LEN];             /**< [en]Indicates extended field.
                                                                         [cn]扩展字段
                                                                         [cs_allowNull:Y] */
    } HwmCallerInfo;

    /**
    * [en]Callee info
    * [cn]被叫信息
    */
    typedef struct tagHwmCalleeInfo
    {
        char nickName[HWM_MAX_DISPLAY_NAME_LEN];                    /**< [en]Callee nickName
                                                                         [cn]被叫姓名
                                                                         [cs_allowNull:Y] */
        char aliasCN[HWM_MAX_DISPLAY_NAME_LEN];                     /**< [en]Indicates chinese alias.
                                                                         [cn]中文别名（获取会中与会者信息使用）
                                                                         [cs_allowNull:Y] */
        char aliasEN[HWM_MAX_DISPLAY_NAME_LEN];                     /**< [en]Indicates english alias.
                                                                         [cn]英文别名（获取会中与会者信息使用）
                                                                         [cs_allowNull:Y] */
        char extendedField[HWM_MAX_EXTENDED_FIELD_LEN];             /**< [en]Indicates extended field.
                                                                         [cn]扩展字段
                                                                         [cs_allowNull:Y] */
        char number[HWM_MAX_NUMBER_LEN];                            /**< [en]Callee number
                                                                         [cn]被叫号码，如果填分配给账号的sip号码（如+99111244216210249）则呼叫该账号的App；
                                                                             如果填PSTN号码（如18700000000），则通过VoIP网关呼叫该号码，前提是该企业已开通PSTN呼叫权限。
                                                                             与thirdUserId二选一（优先级 : number>thirdUserId），账号密码鉴权时使用。
                                                                         [cs_allowNull:N] */
        char thirdUserId[HWM_MAX_USER_ID_LEN];                      /**< [en]Callee third account
                                                                         [cn]被叫三方账号，与number二选一（优先级 : number>thirdUserId），appId鉴权时使用
                                                                         [cs_allowNull:N] */
        char userUuid[HWM_MAX_ACCOUNT_ID_LEN];                      /**< [en]Indicates the uuid.
                                                                         [cn]与会者uuid */
        HwmConfAttendeeType type;                                   /**< [en]Indicates conference attendee type.
                                                                         [cn]与会者类型 */
        char orgId[HWM_MAX_ORG_ID_LENGTH];                          /**< [en]Indicates org id.
                                                                         [cn]企业id */
    } HwmCalleeInfo;

    /**
    * [en]Start call info
    * [cn]发起呼叫信息
    */
    typedef struct tagHwmP2PStartCallInfo
    {
        HwmCallerInfo callerInfo;                               /**< [en]caller info
                                                                     [cn]主叫信息
                                                                     [cs_allowNull:N] */
        HwmCalleeInfo calleeInfo;                               /**< [en]callee info
                                                                     [cn]被叫信息
                                                                     [cs_allowNull:N] */
        HwmMediaType mediaType;                                 /**< [en]Media type
                                                                     [cn]媒体类型
                                                                     [cs_allowNull:N] */
        bool isGuestJoinConfWithoutPwd;                         /**< [en]Indicates whether the guest joins the conference without password
                                                                     [cn]是否来宾免密 */
        HwmJoinConfPermissionType callInRestriction;            /**< [en]Call in Restriction type
                                                                     [cn]呼入限制 */
        char customInfo[HWM_MAX_CUSTOM_INFO_LEN];               /**< [en]Indicates User-defined extended information
                                                                     [cn]自定义扩展信息 */
    } HwmP2PStartCallInfo;

    /**
    * [en]Custom Button Position
    * [cn]自定义按键添加位置类型
    */
    typedef enum tagHwmCustomButtonPos
    {
        CUSTOM_BUTTON_POS_MORE_MENU,      //工具栏更多菜单中自定义菜单
        CUSTOM_BUTTON_POS_TOOL_BAR,       //工具栏中自定义按钮
        CUSTOM_BUTTON_POS_SHARE_TOOL_BAR, //共享工具栏中自定义按钮
    } HwmCustomButtonPos;

    /**
    * [en]Custom Button Info
    * [cn]自定义按钮信息
    */
    typedef struct tagHwmCustomButtonInfo
    {
        HwmCustomButtonPos buttonPos;
        char buttonKey[HWM_MAX_DISPLAY_NAME_LEN];
        char buttonTitle[HWM_MAX_DISPLAY_NAME_LEN];
        char buttonImg[HWM_MAX_FILE_PATH_LEN];
    } HwmCustomButtonInfo;

    /**
    * [en]This structure is used to describe auth server type
    * [cn]鉴权服务类型
    **/
    typedef enum tagAuthServerType
    {
        AUTH_SERVER_TYPE_OAUTH2      /**< [en]Indicates user has not binded. [cn]单点登录*/
    }AuthServerType;

    typedef struct tagHwmSSOAuthInfo
    {
        char code[HWM_MAX_SSO_CODE_LEN];                /**< [en]Indicates the account username.
                                                             [cn]授权码[cs_sensitive:hide] */
        char domain[HWM_MAX_SSO_DOMAIN_LEN];            /**< [en]sso domain.
                                                             [cn]企业域名[cs_sensitive:domain] */
        AuthServerType authServerType;                  /**< [en]authServerType.
                                                             [cn]oauth2 */
    }HwmSSOAuthInfo;

    /**
    * [en]This enumeration is used to invite btn inject info.
    * [cn]邀请按钮注入相关信息
    */
    typedef struct tagHwmInviteBtnInjectInfo
    {
        bool isInviteWndOnTop;              /**< [en]is invite wnd on top.
                                             <br>[cn]邀请界面是否显示在最前面   */
    }HwmInviteBtnInjectInfo;

    /**
    * [en]Custom btn inject info.
    * [cn]自定义按钮注入相关信息
    */
    typedef struct tagHwmCustomBtnInjectInfo
    {
        HwmBtnPosInfo posInfo;                          /**< [en]Buttton pos info
                                                             [cn]按钮位置信息 */
    }HwmCustomBtnInjectInfo;

    /**
    * [en]Modify cycle conf param
    * [cn]编辑周期会议参数
    */
    typedef struct tagModifyCycleConfParam
    {
        HwmEditConfParam editConfParam;            /**< [en] [cn]编辑会议参数
                                                         [cs_allowNull:Y] */
        HwmCycleConfParam cycleParam;                  /**< [en]Indicates sub conference param. [cn]子会议参数
                                                         [cs_allowNull:N]*/
    }HwmModifyCycleConfParam;


    /**
     *[en] [cn]预约周期会议参数
     *[cs_allowNull:Y]
     */
    typedef struct tagBookCycleConfParam
    {
        HwmBookConfParam bookConfParam;                /**< [en] [cn]预约会议参数
                                                         [cs_allowNull:Y]*/
        HwmCycleConfParam cycleParam;                  /**< [en]Indicates sub conference param. [cn]子会议参数
                                                         [cs_allowNull:N]*/
    }HwmBookCycleConfParam;

    /**
    * [en]Modify sub cycle conf param
    * [cn]编辑周期子会议参数
    */
    typedef struct tagModifySubCycleConfParam
    {
        char confId[HWM_MAX_CONF_ID_LEN];                 /**< [en]Indicates conference id. [cn]会议id
                                                                   [cs_allowNull:N]*/
        char subConfID[HWM_MAX_CONF_UUID_LEN];              /**< [en]Indicates conference uuid. [cn]子会议唯一标识UUID
                                                                   [cs_allowNull:N]*/
        HwmConfMediaType confMediaType;                          /**< [en]Indicates media type [cn]会议媒体类型
                                                                   [cs_allowNull:Y] */
        long long startTime;                                  /**< [en]Indicates conference start time.
                                                                   [cn]（已废弃，推荐使用字段startTimeStamp）会议开始时间（UTC时间），单位为秒。
                                                                   [cs_allowNull:Y] */
        long long startTimeStamp;                             /**< [en]Indicates conference start time
                                                                   [cn]会议开始时间，startTimeStamp、startTime二选一，优先使用startTimeStamp(时间戳，单位秒)
                                                                   [cs_allowNull:Y] */
        unsigned int confLen;                                   /**< [en]Indicates conference length of time, unit is minute  ,this param is optional. [cn]可选，会议持续时长，单位分钟
                                                                   [cs_allowNull:Y] */
        bool isAutoRecord;                                /**< [en]Indicates whether record automatically when conference begin. [cn]会议开始后是否自动开启录制
                                                                   [cs_allowNull:Y] */
        unsigned int recordAuthType;                            /**< [en]Indicates record auth type. [cn]录播鉴权方式  0:可通过连接观看/下载（不拼接nonce）1:企业用户可观看/下载 2:与会者可观看/下载
                                                                   [cs_allowNull:Y] */
        HwmJoinConfPermissionType callInRestriction;              /**< [en]Indicates call in restriction. [cn]呼入限制
                                                                   [cs_allowNull:Y] */
        bool allowGuestStartConf;                             /**< [en]Indicates allow guest start conf. [cn]允许来宾启动会议
                                                                   [cs_allowNull:Y] */
        unsigned int allowGuestStartConfTime;                   /**< [en]Indicates join before host time.
                                                                   [cn]允许来宾提前入会时间范围(单位：分钟) 0-随时，n-提前n分钟启动会议*/
    }HwmModifySubCycleConfParam;

    /*
    * [en]cancelCycleConfParam
    * [cn]取消周期会议参数
    */
    typedef struct tagCancelCycleConfParam
    {
        char confId[HWM_MAX_CONF_ID_LEN];                 /**< [en]Indicates conference id. [cn]会议id
                                                                   [cs_allowNull:N]*/
        bool endOnlineConf;                               /**< [en]Indicates is end online conf. [cn]是否需要结束在线会议
                                                                   [cs_allowNull:N]*/
    }HwmCancelCycleConfParam;

    /*
    * [en]cancelSubCycleConfParam
    * [cn]取消周期子会议参数
    */
    typedef struct tagCancelSubCycleConfParam
    {
        char confId[HWM_MAX_CONF_ID_LEN];                 /**< [en]Indicates conference id. [cn]会议id
                                                                   [cs_allowNull:N]*/
        char subConfID[HWM_MAX_CONF_UUID_LEN];            /**< [en]Indicates conference uuid. [cn]子会议唯一标识UUID
                                                                   [cs_allowNull:N]*/
        bool endOnlineConf;                               /**< [en]Indicates is end online conf. [cn]是否需要结束在线会议
                                                                   [cs_allowNull:N]*/
    }HwmCancelSubCycleConfParam;

    /*
    * [en]Setting Area info
    * [cn]设置区域信息
    */
    typedef struct tagHwmSettingArea
    {
        bool isHide;                                    /**< [en]Is hide.
                                                             [cn]是否隐藏 */
    }HwmSettingArea;

    /*
    * [en]Normal setting tab info
    * [cn]基本设置页信息
    */
    typedef struct tagHwmNormalSettingTab
    {
        bool isHide;                                       /**< [en]Is hide.
                                                                [cn]是否隐藏 */
        HwmSettingArea callOrConfArea;                     /**< [en]CallOrConf setting area.
                                                                [cn]会议呼叫设置区域 */
        HwmSettingArea chatNotifyArea;                     /**< [en]ChatNotify setting area.
                                                                [cn]聊天消息提醒设置区域 */
        HwmSettingArea subtitleArea;                       /**< [deprecated]字幕使用了独立设置页
                                                                [en]Subtitle setting area.
                                                                [cn]字幕设置区域 */
        HwmSettingArea shareArea;                          /**< [en]Share setting area.
                                                                [cn]共享设置区域 */
    }HwmNormalSettingTab;

    /*
    * [en]Video setting tab info
    * [cn]视频设置页信息
    */
    typedef struct taHwmVideoSettingTab
    {
        bool isHide;                                       /**< [en]Is hide.
                                                                [cn]是否隐藏 */
        HwmSettingArea cameraArea;                         /**< [en]Camera setting area.
                                                                [cn]摄像头设置区域 */
        HwmSettingArea videoViewArea;                      /**< [en]Video view setting area.
                                                                [cn]视频画面设置区域 */
    }HwmVideoSettingTab;

    /*
    * [en]Audio setting tab info
    * [cn]音频设置页信息
    */
    typedef struct tagHwmAudioSettingTab
    {
        bool isHide;                                       /**< [en]Is hide.
                                                                [cn]是否隐藏 */
        HwmSettingArea micArea;                            /**< [en]Mic setting area.
                                                                [cn]麦克风设置区域 */
        HwmSettingArea spkArea;                            /**< [en]Spk setting area.
                                                                [cn]扬声器设置区域 */
        HwmSettingArea advancedArea;                      /**<  [en]Advanced setting area.
                                                                [cn]高级设置区域 */
    }HwmAudioSettingTab;

    /*
    * [en]Screen setting tab info
    * [cn]屏幕设置页信息
    */
    typedef struct tagHwmScreenSettingTab
    {
        bool isHide;                                       /**< [en]Is hide.
                                                                [cn]是否隐藏 */
        HwmSettingArea dualScreenArea;                      /**<[en]Dual screen setting area.
                                                                [cn]双屏设置区域 */
        HwmSettingArea displayTopologyArea;                 /**<[en]Display topology setting area
                                                                [cn]显示器显示模式设置区域 */
        HwmSettingArea swapContentArea;                     /**<[en]Swap screen content setting area
                                                                [cn]切换显示内容设置区域 */
    }HwmScreenSettingTab;

    /*
    * [en]Subtitle setting tab info
    * [cn]字幕设置页信息
    */
    typedef struct tagHwmSubtitleSettingTab
    {
        bool isHide;                                       /**< [en]Is hide.
                                                                [cn]是否隐藏 */
        HwmSettingArea subtitleSettingArea;                /**< [en]Dual screen setting area.
                                                                [cn]字幕基础设置区域 */
        HwmSettingArea translateSettingArea;               /**< [en]Display topology setting area
                                                                [cn]字幕翻译设置区域 */
    }HwmSubtitleSettingTab;

    /*
    * [en]Record setting tab info
    * [cn]录制设置页信息
    */
    typedef struct tagHwmRecordSettingTab
    {
        bool isHide;                                       /**< [en]Is hide.
                                                                [cn]是否隐藏 */
        HwmSettingArea localRecordArea;                    /**< [en]Local setting area.
                                                                [cn]本地录制设置区域 */
    }HwmRecordSettingTab;

    /*
    * [en]VirtualBackgroud setting tab info
    * [cn]虚拟背景设置页信息
    */
    typedef struct tagHwmVirtualBackgroudSettingTab
    {
        bool isHide;                                       /**< [en]Is hide.
                                                                [cn]是否隐藏 */
        HwmSettingArea virtualBackgroudArea;               /**< [en]Virtual backgroud setting area.
                                                                [cn]虚拟背景设置区域 */
    }HwmVirtualBackgroudSettingTab;

    /*
    * [en]Hotkey setting tab info
    * [cn]快捷键设置页信息
    */
    typedef struct tagHwmHotKeySettingTab
    {
        bool isHide;                                       /**< [en]Is hide.
                                                                [cn]是否隐藏 */
        HwmSettingArea hotkeyArea;                         /**< [en]Hotkey setting area.
                                                                [cn]快捷键设置区域 */
    }HwmHotKeySettingTab;

    /*
    * [en]Setting window config info
    * [cn]设置界面配置信息
    */
    typedef struct tagHwmSettingWindowConfig
    {
        HwmNormalSettingTab normalTab;                                 /**< [en]Normal setting tab.
                                                                            [cn]基本设置页 */
        HwmVideoSettingTab videoTab;                                   /**< [en]Video setting tab.
                                                                            [cn]视频设置页 */
        HwmAudioSettingTab audioTab;                                   /**< [en]Audio setting tab.
                                                                            [cn]音频设置页 */
        HwmScreenSettingTab screenTab;                                 /**< [en]Screen setting tab.
                                                                            [cn]屏幕设置页信息 */
        HwmSubtitleSettingTab subtitleTab;                             /**< [en]Subtitle setting tab.
                                                                            [cn]字幕设置页信息 */
        HwmRecordSettingTab recordTab;                                 /**< [en]Record setting tab.
                                                                            [cn]录制设置页 */
        HwmVirtualBackgroudSettingTab virtualBackgroudTab;             /**< [en]Virtual backgroud setting tab.
                                                                            [cn]虚拟背景设置页 */
        HwmHotKeySettingTab hotkeyTab;                                 /**< [en]Hotkey setting tab.
                                                                            [cn]快捷键设置页 */
    }HwmSettingWindowConfig;

    /*
    * [en]Indicates network access type
    * [cn]网络接入类型
    */
    typedef enum tagHwmNetworkAccessType
    {
        ACCESS_TYPE_AUTO = 0,                 /**< [en]Indicates auto
                                                   [cn]默认值 */
        ACCESS_TYPE_INNER,                    /**< [en]Indicates inner
                                                   [cn]公司内网接入, 用于媒体和信令下沉场景，app从内网接入，信令/媒体消息走代理服务器才能访问云会议服务端 */
        ACCESS_TYPE_OUTER,                    /**< [en]Indicates outter
                                                   [cn]公司外网接入, 用于非媒体和信令下沉场景，互联网接入，信令/媒体消息可以直接访问云会议服务端 */
    } HwmNetworkAccessType;

    /*
    * [en]Indicates network connect type
    * [cn]网络连接类型
    */
    typedef enum tagHwmNetConnectType {
        NET_CONNECT_AUTO = 0,                 /**< [en]Indicates auto
                                                   [cn]自动模式 */
        NET_CONNECT_FORCE_VPN,                /**< [en]Indicates force vpn
                                                   [cn]强制vpn */
        NET_CONNECT_FORCE_DIRECT,             /**< [en]Indicates force direct
                                                   [cn]强制直连 */
    }HwmNetConnectType;

    /*
    * [en]network mode info
    * [cn]网络模式信息
    */
    typedef struct tagHwmNetWorkModeInfo {
        HwmNetworkAccessType accessType;                       /**< [en]Indicates network access type
                                                                    [cn]网络接入类型 */
        HwmNetConnectType netConnectType;                      /**< [en]Indicates network connect type
                                                                    [cn]网络连接类型 */
    }HwmNetWorkModeInfo;

    /*
    * [en]raw data external config.
    * [cn]端到端加解密接口配置信息
    */
    typedef struct tagHwmRawDataExternalConfigInfo
    {
        bool enableE2EE;                                              /**< [en]Indicates enableE2EE. 
                                                                           [cn]是否开启端到端加密(默认值: false)，开启后下面的设置项才生效 */
        char libraryPath[HWM_MAX_RAW_DATA_LIBRARY_PATH_LEN];          /**< [en]library path
                                                                           [cn]库目录 */
        char audioEncrtptFuncName[HWM_MAX_ENCRYPT_FUNC_NAME_LEN];     /**< [en]audio encrypt func name
                                                                           [cn]语音加密函数名 */
        char audioDecrtptFuncName[HWM_MAX_ENCRYPT_FUNC_NAME_LEN];     /**< [en]audio decrypt func name
                                                                           [cn]语音解密函数名 */
        char videoEncrtptFuncName[HWM_MAX_ENCRYPT_FUNC_NAME_LEN];     /**< [en]video encrypt func name
                                                                           [cn]视频加密函数名 */
        char videoDecrtptFuncName[HWM_MAX_ENCRYPT_FUNC_NAME_LEN];     /**< [en]video decrypt func name
                                                                           [cn]视频解密函数名 */
    }HwmRawDataExternalConfigInfo;

    typedef struct tagHwmMediaFrameBuffer
    {
        unsigned char *buffer;     /**< [en]Indicates media frame buffer. 
                                        [cn]音视频码流缓冲区地址 */
        unsigned int dataSize;     /**< [en]Indicates size of media frame buffer. 
                                        [cn]音视频原始码流长度 */
        unsigned int bufferSize;   /**< [en]Indicates size of media frame buffer. 
                                        [cn]音视频码流缓冲区长度，加解密后不能超出这个长度 */
    }HwmMediaFrameBuffer;

    /**
    * [en] Setting page
    * [cn] 设置界面页签
    */
    typedef enum tagHwmSettingPage
    {
        SETTING_PAGE_NORMAL,                        /**< [en]Normal setting page.
                                                    [cn]基本设置页 */
        SETTING_PAGE_VIDEO,                         /**< [en]Video setting page.
                                                    [cn]视频设置页 */
        SETTING_PAGE_AUDIO,                         /**< [en]Audio setting page.
                                                    [cn]音频设置页 */
        SETTING_PAGE_RECORD,                        /**< [en]Record setting page.
                                                    [cn]录制设置页 */
        SETTING_PAGE_HOTKEY,                        /**< [en]Hotkey setting page.
                                                    [cn]快捷键设置页 */
        SETTING_PAGE_SCREEN,                        /**< [en]Screen setting page.
                                                    [cn]屏幕设置页 */
        SETTING_PAGE_COUNT,                        /**< [en]Count of setting pages.
                                                    [cn]设置页数量 */
    }HwmSettingPage;

    /**
    * [en] Show setting window info
    * [cn] 显示设置窗口信息
    */
    typedef struct tagHwmShowSettingWindowInfo
    {
        bool isShow;                         /**< [en]Is show.
                                             [cn]是否显示 */
        HwmSettingPage defaultPage;          /**< [en]Default page
                                             [cn]默认显示标签页 */
    }HwmShowSettingWindowInfo;

    /**
    * [en] Voice prompts info
    * [cn] 提示音信息
    */
    typedef struct tagHwmVoicePromptsInfo
    {
        bool isDisableMutePrompt;       /**< [en]Is disable mute prompts.
                                             [cn]是否禁用静音提示 */
    }HwmVoicePromptsInfo;

    /**
    * [en] Call call info
    * [cn] 全部呼叫信息
    */
    typedef struct tagHwmCallAllInfo
    {
        HwmConfAttendee* list;          /**< [en] Called attendee list
                                             [cn] 被叫列表 */
        unsigned int count;             /**< [en] Count of attendee(s) in the list
                                             [en] 被叫数量 */
    }HwmCallAllInfo;
}

#endif /* _HWM_AGENT_DEF_H_ */
