﻿/**
 * @file HwmAgentCallback.h
 *
 * Copyright(C), 2019-2020, Huawei Tech. Co., Ltd. ALL RIGHTS RESERVED.
 *
 * @brief [en] Callback class.
 *        [cn] 回调类文件
 */

#ifndef _HWM_AGENT_CALLBACK_H_
#define _HWM_AGENT_CALLBACK_H_

#include "HwmAgentNotifyDef.h"
#include "HwmErrorCodeDef.h"
namespace hwmsdkagent
{

    class HwmAgentCallback
    {
    public:
        /**
         * @brief [en] Callback of Init interface.
         *        [cn] 初始化回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @param [in] HwmSdkInfo *sdkInfo       [en] Sdk info
         *                                       [cn] sdk信息
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnInitResult(hwmsdk::HwmErrCode ret, const char* reason, HwmSdkInfo *sdkInfo) {};

        /**
         * @brief [en] Callback of Exit interface.
         *        [cn] 退出回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnExitResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
         * @brief [en] Callback of Config interface.
         *        [cn] 配置定制化参数回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnConfigResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
         * @brief [en] Callback of CleanConfig interface.
         *        [cn] 清除定制化参数回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnCleanConfigResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of EnableFeature interface.
        *        [cn] 会议相关功能开关配置接口回调
        *
        * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
        *                                       [cn] 返回值
        * @param [in] const char* reason        [en] Fail reason
        *                                       [cn] 失败原因描述
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnEnableFeature(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of SetLanguage interface.
        *        [cn] 设置语言回调
        *
        * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
        *                                       [cn] 返回值
        * @param [in] const char* reason        [en] Fail reason
        *                                       [cn] 失败原因描述
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnSetLanguageResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of SetShareScreenPermission interface.
        *        [cn] 设置共享权限回调
        *
        * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
        *                                       [cn] 返回值
        * @param [in] const char* reason        [en] Fail reason
        *                                       [cn] 失败原因描述
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnSetShareScreenPermissionResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of RenameSelf interface.
        *        [cn] 设置会场名回调
        *
        * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
        *                                       [cn] 返回值
        * @param [in] const char* reason        [en] Fail reason
        *                                       [cn] 失败原因描述
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnRenameSelfResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
         * @brief [en] Callback of Login interface.
         *        [cn] 登录回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @param [in] const HwmLoginResult* loginResult    [en] login result info
         *                                                  [cn] 登录结果信息
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnLoginResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmLoginResult* loginResult) {};
        /**
         * @brief [en] Callback of Logout interface.
         *        [cn] 登出回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @param [in] const HwmLogoutResult* logoutesult   [en] logout result info
         *                                                  [cn] 登出结果信息
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnLogoutResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmLogoutResult* logoutResult) {};

        /**
         * @brief [en] Callback of CreateConf interface.
         *        [cn] 创建会议回调
         *
         * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
         *                                               [cn] 返回值
         * @param [in] const char* reason                [en] Fail reason
         *                                               [cn] 失败原因描述
         * @param [in] HwmConfInfo *confInfo             [en] Conf info
         *                                               [cn] 会议信息
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnCreateConfResult(hwmsdk::HwmErrCode ret, const char* reason, HwmConfInfo *confInfo) {};

        /**
         * @brief [en] Callback of JoinConfById interface.
         *        [cn] 通过会议id入会回调
         *
         * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
         *                                               [cn] 返回值
         * @param [in] const char* reason                [en] Fail reason
         *                                               [cn] 失败原因描述
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnJoinConfByIdResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of JoinConfByRandom interface.
        *        [cn] 通过会议random入会回调
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnJoinConfByRandomResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
         * @brief [en] Callback of AddAttendee interface.
         *        [cn] 添加与会者回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnAddAttendeeResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
         * @brief [en] Callback of LeaveConf interface.
         *        [cn] 离开会议回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnLeaveConfResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
         * @brief [en] Callback of EndConf interface.
         *        [cn] 结束会议回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnEndConfResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
         * @brief [en] Callback of StartShare interface.
         *        [cn] 开启共享回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnStartShareResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
         * @brief [en] Callback of StopShare interface.
         *        [cn] 停止共享回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnStopShareResult(hwmsdk::HwmErrCode ret, const char* reason) {};

         /**
         * @brief [en] Callback of BookConf interface.
         *        [cn] 预约会议回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @param [in] const HwmConfDetail* confDetail    [en] conf detail
         *                                                [cn] 会议详情
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnBookConfResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmConfDetail* confDetail) {};

        /**
         * @brief [en] Callback of CancelConf interface.
         *        [cn] 取消会议预约回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnCancelConfResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
         * @brief [en] Callback of EditConf interface.
         *        [cn] 修改会议预约回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnEditConfResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
         * @deprecated
         * [en] Deprecated API.
         * [cn] 废弃接口
         *
         * @brief [en] 
         *        [cn] Vmr列表通知，get vmr list 请求返回
         *
         * @param [in] hwmsdk::HwmErrCode ret                    [en] Return code
         *                                                       [cn] 返回值
         * @param [in] const char* reason                        [en] Fail reason
         *                                                       [cn] 失败原因描述
         * @param [in] const HwmVmrInfo* vmrInfoList             [en] the list of vmrinfo
         *                                                       [cn] 云会议室信息列表
         * @param [in] unsigned int listLen                      [en] the size of the list
         *                                                       [cn] 列表长度                         
         */
        virtual void OnGetVmrListResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmVmrInfo* vmrInfoList, unsigned int listLen) {};

        /**
         * @brief [en]
         *        [cn] Vmr列表通知，get vmr list 请求返回
         *
         * @param [in] hwmsdk::HwmErrCode ret                    [en] Return code
         *                                                       [cn] 返回值
         * @param [in] const char* reason                        [en] Fail reason
         *                                                       [cn] 失败原因描述
         * @param [in] const HwmVmrListInfo* vmrListInfo         [en] the info of vmr list
         *                                                       [cn] vmr列表信息
         */
        virtual void OnGetVmrInfoListResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmVmrListInfo* vmrListInfo) {};

        /**
         * @brief [en] 
         *        [cn] 会议详情通知，获取会议详情返回
         *
         * @param [in] hwmsdk::HwmErrCode ret                      [en] Return code
         *                                                         [cn] 返回值
         * @param [in] const char* reason                          [en] Fail reason
         *                                                         [cn] 失败原因描述
         * @param [in] const HwmConfDetail* confDetail             [en] conference information in detail
         *                                                         [cn] 会议详细信息
         */
        virtual void OnGetConfDetailResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmConfDetail* confDetail) {};
         /**
         * @brief [en] Callback of ChangeVmrInfo interface.
         *        [cn] 修改虚拟会议室信息回调
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         **/
        virtual void OnChangeVmrInfoResult(hwmsdk::HwmErrCode ret, const char* reason) {};
         /**
         * @brief [en] Callback of SetWaterMarkInjectInfo interface.
         *        [cn] 设置水印注入信息回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnSetWaterMarkInjectInfoResult(hwmsdk::HwmErrCode ret, const char* reason) {};

         /**
         * @brief [en] Callback of StartCall interface.
         *        [cn] 发起呼叫回调
         *
         * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
         *                                               [cn] 返回值
         * @param [in] const char* reason                [en] Fail reason
         *                                               [cn] 失败原因描述
         * @attention [en] NA
         *            [cn] NA
         **/
        virtual void OnStartCallResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of StartP2pConf interface.
        *        [cn] 发起立即会议回调
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        * @param [in] HwmConfInfo *confInfo             [en] Conf info
        *                                               [cn] 会议信息
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnStartP2pConfResult(hwmsdk::HwmErrCode ret, const char* reason, HwmConfInfo *confInfo) {};

        /**
        * @brief [en] Callback of SetCallIncomingAnswerResult interface.
        *        [cn] 设置呼叫来电应答结果回调
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnSetCallIncomingAnswerResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of SetConfIncomingAnswerResult interface.
        *        [cn] 设置会议来电应答结果回调
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnSetConfIncomingAnswerResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of set file save path.
        *        [cn] 设置用户保存路径结果回调
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnSetUserSavePath(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of GetUserState interface.
        *        [cn] 获取用户状态结果回调
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        * @param [in] HwmUserState userState            [en] user state
        *                                               [cn] 用户状态
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnGetUserStateResult(hwmsdk::HwmErrCode ret, const char* reason, HwmUserState userState) {};

        /**
        * @brief [en]  Callback of SetShareStrategy interface.
        *        [cn]  设置屏幕共享策略接口回调
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnSetShareStrategyResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en]  Callback of SetShareAudio interface.
        *        [cn]  设置共享音频接口回调
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnSetShareAudioResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en]  Callback of AVDeviceCtrl interface.
        *        [cn]  音视频开关控制接口回调
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        * @param [in] HwmDeviceType deviceType          [en] Device type
        *                                               [cn] 设备类型
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnAVDeviceCtrlResult(hwmsdk::HwmErrCode ret, const char* reason, HwmDeviceType deviceType) {};

        /**
        * @brief [en] Callback of GetAttendeeList interface.
        *        [cn] 获取与会者列表接口回调
        * @param [in] HwmAttendeeInfo audidenceList                 [en] 与会者列表
        *                                                           [cn] attendeeList
        * @param [in] unsigned int length                           [en] 与会者列表个数
        *                                                           [cn] attendee Number
        **/
        virtual void OnGetAttendeeListResult(const HwmAttendeeInfo* audidenceList, unsigned int length) {};

        /**
        * @brief [en] Callback of GetAudienceList interface.
        *        [cn] 获取观众列表接口回调
        * @param [in] HwmAttendeeInfo* audidenceList                [en] 观众列表
        *                                                           [cn] audidenceList
        * @param [in] unsigned int length                           [en] 观众个数
        *                                                           [cn] audidence Number
        **/
        virtual void OnGetAudienceListResult(const HwmAttendeeInfo* audidenceList, unsigned int length) {};

        /**
        * @brief [en] Callback of OnSetAudioRawDataOutputConfigResult interface.
        *        [cn] 开启或关闭接收音频流接口回调
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        **/
        virtual void OnSetAudioRawDataOutputConfigResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en]  Callback of UpdateCustomButtonInfo interface.
        *        [cn]  更新自定义按钮接口回调
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnUpdateCustomButtonInfoResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en]
        * [cn]创建周期会议的结果回调
        *
        * @param result [type:HwmErrCode][cs_allowNull:N]
        * [en]
        * [cn]错误码
        * @see HwmErrCode
        *
        * @param reasonDesc [type:char*][cs_allowNull:N]
        * [en]
        * [cn]错误描述
        */
        virtual void OnBookCycleConfResult(hwmsdk::HwmErrCode result, const char* reasonDesc, const HwmConfDetail* confDetail) {};

        /**
        * @brief
        * [en]
        * [cn]编辑周期会议的结果回调
        *
        * @param result [type:HwmErrCode][cs_allowNull:N]
        * [en]
        * [cn]错误码
        * @see HwmErrCode
        *
        * @param reasonDesc [type:char*][cs_allowNull:N]
        * [en]
        * [cn]错误描述
        */
        virtual void OnModifyCycleConfResult(hwmsdk::HwmErrCode  result, const char* reasonDesc) {};

        /**
        * @brief
        * [en]
        * [cn]编辑周期子会议的结果回调
        *
        * @param result [type:HwmErrCode][cs_allowNull:N]
        * [en]
        * [cn]错误码
        * @see HwmErrCode
        *
        * @param reasonDesc [type:char*][cs_allowNull:N]
        * [en]
        * [cn]错误描述
        */
        virtual void OnModifySubCycleConfResult(hwmsdk::HwmErrCode result, const char* reasonDesc) {};

        /**
        * @brief
        * [en]
        * [cn]取消周期会议的结果回调
        *
        * @param result [type:HwmErrCode][cs_allowNull:N]
        * [en]
        * [cn]错误码
        * @see HwmErrCode
        *
        * @param reasonDesc [type:char*][cs_allowNull:N]
        * [en]
        * [cn]错误描述
        */
        virtual void OnCancelCycleConfResult(hwmsdk::HwmErrCode result, const char* reasonDesc) {};

        /**
        * @brief
        * [en]
        * [cn]取消周期子会议的结果回调
        *
        * @param result [type:HwmErrCode][cs_allowNull:N]
        * [en]
        * [cn]错误码
        * @see HwmErrCode
        *
        * @param reasonDesc [type:char*][cs_allowNull:N]
        * [en]
        * [cn]错误描述
        */
        virtual void OnCancelSubCycleConfResult(hwmsdk::HwmErrCode result, const char* reasonDesc) {};

        /**
        * @brief
        * [en] Callback of ConfigSettingWindow interface.
        * [cn] 定制设置界面的结果回调
        * 
        * @param result [type:HwmErrCode][cs_allowNull:N]
        * [en]
        * [cn]错误码
        * @see HwmErrCode
        *
        * @param reasonDesc [type:char*][cs_allowNull:N]
        * [en]
        * [cn]错误描述
        */
        virtual void OnConfigSettingWindowResult(hwmsdk::HwmErrCode result, const char* reasonDesc) {};
        
        /**
        * @brief [en] Callback of GetSSOAuthorizeUrl interface.
        *        [cn] 根据企业域名获取鉴权中心Url 接口 结果回调
        *
        * @param [in] hwmsdk::HwmErrCode ret                    [en] Return code
        *                                                       [cn] 返回值
        * @param [in] const char* reason                        [en] Fail reason
        *                                                       [cn] 失败原因描述
        * @param [in] const char* ssoAuthorizeUrl               [en] SSO Authorize Url
        *                                                       [cn] 鉴权中心Url
        */
        virtual void OnGetSSOAuthorizeUrlResult(hwmsdk::HwmErrCode ret, const char* reason, const char* ssoAuthorizeUrl) {};

        /**
        * @brief [en] Callback of LoginBySSO interface.
        *        [cn] SSO登录结果回调
        *
        * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
        *                                       [cn] 返回值
        * @param [in] const char* reason        [en] Fail reason
        *                                       [cn] 失败原因描述
        * @param [in] const HwmLoginResult* loginResult    [en] login result info
        *                                                  [cn] 登录结果信息
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnLoginBySSOResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmLoginResult* loginResult) {};
        
        /**
        * @brief [en] Callback of SetNetworkMode interface.
        *        [cn] 设置网络模式结果回调
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        **/
        virtual void OnSetNetworkModeResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of SetRawDataExternalConfig interface.
        *        [cn] 设置端到端加密接口配置信息回调
        *
        * @param [in] hwmsdk::HwmErrCode ret             [en] Return code
        *                                                [cn] 返回值
        * @param [in] const char* reason                 [en] Fail reason
        *                                                [cn] 失败原因描述
        **/
        virtual void OnSetRawDataExternalConfigResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of ShowSettingWindow interface.
        *        [cn] 显示设置界面接口结果回调
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnShowSettingWindowResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of DisableVoicePrompts interface.
        *        [cn] 屏蔽语音提示接口结果回调
        *
        * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
        *                                       [cn] 返回值
        * @param [in] const char* reason        [en] Fail reason
        *                                       [cn] 失败原因描述
        * @attention [en] NA
        *            [cn] NA
        **/
        virtual void OnDisableVoicePromptsResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief 
        * [en] Callback of UploadSelfAvatar interface.
        * [cn] 上传个人头像结果回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnUploadSelfAvatarResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of AgreeVoiceprintStatement interface.
        *        [cn] 签署声纹功能声明回调
        *
        * @param [in] SDKERR ret    [en] Return code
        *                                       [cn] 返回值
        * @param [in] const char* reason        [en] Fail reason
        *                                       [cn] 失败原因描述
        *
        * @mac:disable
        **/
        virtual void OnAgreeVoiceprintStatementResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of ConfigVoiceprintRecordWindow interface.
        *        [cn] 定制声纹录入界面回调
        *
        * @param [in] SDKERR ret    [en] Return code
        *                                       [cn] 返回值
        * @param [in] const char* reason        [en] Fail reason
        *                                       [cn] 失败原因描述
        *
        * @mac:disable
        **/
        virtual void OnConfigVoiceprintRecordWindowResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of ShowVoiceprintRecordWindow interface.
        *        [cn] 显示声纹录入窗口回调
        *
        * @param [in] SDKERR ret    [en] Return code
        *                                       [cn] 返回值
        * @param [in] const char* reason        [en] Fail reason
        *                                       [cn] 失败原因描述
        *
        * @mac:disable
        **/
        virtual void OnShowVoiceprintRecordWindowResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief 
        * [en] Callback of set file save path.
        * [cn] 设置用户保存路径结果回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        * @param pathInfo [type:HwmUserSavePathInfo*]
        * [en] Save path info
        * [cn] 保存路径信息
        *
        * @mac:disable
        **/
        virtual void OnGetUserSavePathResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmUserSavePathInfo* pathInfo) {};

        /**
        * @brief 
        * [en] Callback of DeleteSelfAvatar interface.
        * [cn] 删除个人头像结果回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnDeleteSelfAvatarResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of set log save path.
        * [cn] 设置日志路径结果回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnSetLogPathResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of GeneralWatch interface.
        * [cn] 选看接口回调
        *
        * @param ret [type:SDKERR][cs_allowNull:N]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*][cs_allowNull:N]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnGeneralWatchResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of ShowToast interface.
        * [cn] 显示提示信息回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnShowToastResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of GetMainWindowInfo interface
        * [cn] 主窗口信息通知，获取主窗口信息请求返回
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        * @param mainWindowInfo [type:HwmMainWindowInfo*]
        * [en] Main window info
        * [cn] 主窗口信息
        */
        virtual void OnGetMainWindowInfoResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmMainWindowInfo* mainWindowInfo) {};

        /**
        * @brief
        * [en] Callback of SetAnnotationWindow interface.
        * [cn] 设置标注窗口接口回调
        *
        * @param ret [type:SDKERR][cs_allowNull:N]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*][cs_allowNull:N]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnSetAnnotationWindowResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of StartAnnotation interface.
        * [cn] 开启标注接口回调
        *
        * @param ret [type:SDKERR][cs_allowNull:N]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*][cs_allowNull:N]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnStartAnnotationResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of SaveAnnotation interface.
        * [cn] 保存标注接口回调
        *
        * @param ret [type:SDKERR][cs_allowNull:N]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*][cs_allowNull:N]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnSaveAnnotationResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of StopAnnotation interface.
        * [cn] 停止标注接口回调
        *
        * @param ret [type:SDKERR][cs_allowNull:N]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*][cs_allowNull:N]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnStopAnnotationResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of SetTransparentWnd interface.
        *        [cn] 设置透明窗口接口回调
        *
        * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
        *                                       [cn] 返回值
        * @param [in] const char* reason        [en] Fail reason
        *                                       [cn] 失败原因描述
        **/
        virtual void OnSetTransparentWndResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of SetServerAddress interface.
        *        [cn] 设置登录地址信息回调
        *
        * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
        *                                       [cn] 返回值
        * @param [in] const char* reason        [en] Fail reason
        *                                       [cn] 失败原因描述
        **/
        virtual void OnSetServerAddressResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of SetProxyInfo interface.
        *        [cn] 设置代理信息回调
        *
        * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
        *                                       [cn] 返回值
        * @param [in] const char* reason        [en] Fail reason
        *                                       [cn] 失败原因描述
        **/
        virtual void OnSetProxyInfoResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of SetCAVerifyInfo interface.
        *        [cn] 设置证书校验信息回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        **/
        virtual void OnSetCAVerifyInfoResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of SetCustomUIText interface.
        *        [cn] 设置用户自定义场景文案
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnSetCustomUITextResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of SetCustomUIImage interface.
        *        [cn] 设置用户自定义场景图标回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnSetCustomUIImageResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of SetAnswerType interface.
        * [cn] 设置来电应答结果回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnSetAnswerTypeResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of AcceptConf interface.
        * [cn] 接受会议邀请回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        */
        virtual void OnAcceptConfResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of RejectConf interface.
        * [cn] 拒绝会议邀请回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        */
        virtual void OnRejectConfResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of SetGlobalGuaranteedIpAddresses interface.
        *        [cn] 设置管理面域名对应的保底ip地址回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        **/
        virtual void OnSetGlobalGuaranteedIpAddressesResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of GetSelfShareState interface.
        * [cn] 获取自己的开启共享状态回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        * @param selfShareStateInfo [type:HwmSelfShareStateInfo]
        * [en] Self share state info
        * [cn] 自己开启共享的状态信息
        **/
        virtual void OnGetSelfShareStateResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmSelfShareStateInfo* selfShareStateInfo) {};

        /**
        * @brief [en] Callback of GetLoginState interface.
        *        [cn] 获取登录状态接口回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        * @param loginStateInfo [type:HwmLoginStateInfo*]
        * [en] login state info
        * [cn] 登录状态信息
        **/
        virtual void OnGetLoginStateResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmLoginStateInfo* loginStateInfo) {};

        /**
        * @brief
        * [en] Callback of SetAvatarStyle interface.
        * [cn] 设置头像风格回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnSetAvatarStyleResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of GetLoginToken interface.
        *        [cn] 获取登录token接口回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        * @param loginTokenInfo [type:HwmLoginTokenInfo*]
        * [en] login token info
        * [cn] 登录token信息
        **/
        virtual void OnGetLoginTokenResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmLoginTokenInfo* loginTokenInfo) {};

        /**
        * @brief
        * [en] Callback of OpenSubtitle interface.
        * [cn] 开启/关闭字幕回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        */
        virtual void OnOpenSubtitleResult(hwmsdk::HwmErrCode ret, const char* reason, bool isOpen) {};

        /**
        * @brief
        * [en] Callback of OpenWaitingRoom interface.
        * [cn] 开启/关闭等候室回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        */
        virtual void OnOpenWaitingRoomResult(hwmsdk::HwmErrCode ret, const char* reason, bool isOpen) {};

        /**
        * @brief
        * [en] Callback of OperateLocalRecord interface.
        * [cn] 操作本地录制回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        */
        virtual void OnOperateLocalRecordResult(hwmsdk::HwmErrCode ret, const char* reason, HwmLocalRecordOperateType type) {};

        /**
        * @brief
        * [en] Callback of OpenVirtualBackgroundSettingPage interface.
        * [cn] 打开虚拟背景设置回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        */
        virtual void OnOpenVirtualBackgroundSettingPageResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of GetConfLocalRecordStatus interface.
        * [cn] 获取本地录制状态结果回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        * @param state [type:HwmLocalRecordStatusInfo]
        * [en] Local record status
        * [cn] 本地录制状态
        **/
        virtual void OnGetConfLocalRecordStatusResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmLocalRecordStatusInfo* state) {};

        /**
        * @brief
        * [en] Callback of GetConfLocalRecordPermission interface.
        * [cn] 获取本地录制权限结果回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        * @param permission [type:HwmLocalRecordPermissionInfo]
        * [en] Local record permission
        * [cn] 本地录制状态
        **/
        virtual void OnGetSelfHasLocalRecordPermissionResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmLocalRecordPermissionInfo* permission) {};

        /**
        * @brief
        * [en] Callback of RequestLocalRecordPermission interface.
        * [cn] 申请本地录制权限回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        */
        virtual void OnRequestLocalRecordPermissionResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of ApplyHostRole interface.
        * [cn] 申请主持人回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        */
        virtual void OnApplyHostRoleResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief
        * [en] Callback of GetSubtitlePageOpened interface.
        * [cn] 获取字幕状态回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        * @param isOpen [type:HWM_BOOL]
        * [en] Is open
        * [cn] 是否开启
        */
        virtual void OnGetSubtitlePageOpenedResult(hwmsdk::HwmErrCode ret, const char* reason, bool isOpen) {};

        /**
        * @brief
        * [en] Callback of GetWaitingRoomOpened interface.
        * [cn] 获取等候室状态回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        * @param isOpen [type:HWM_BOOL]
        * [en] Is open
        * [cn] 是否开启
        */
        virtual void OnGetWaitingRoomOpenedResult(hwmsdk::HwmErrCode ret, const char* reason, bool isOpen) {};

        /**
        * @brief
        * [en] Callback of GetVirtualBackgroundSettingPageOpened interface.
        * [cn] 获取虚拟背景设置界面状态回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        * @param isOpen [type:HWM_BOOL]
        * [en] Is open
        * [cn] 是否开启
        */
        virtual void OnGetVirtualBackgroundSettingPageOpenedResult(hwmsdk::HwmErrCode ret, const char* reason, bool isOpen) {};

        /**
        * @brief
        * [en] Callback of SubscribeSubtitleLanguage interface.
        * [cn] 订阅字幕语言
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        */
        virtual void OnSubscribeSubtitleLanguageResult(hwmsdk::HwmErrCode ret, const char* reason, HwmTargetLanguageType targetLanguageType) {};

        /**
        * @brief
        * [en] Callback of SubscribeSubtitle interface.
        * [cn] 订阅/取消订阅字幕
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        * @param isSubscirbe [type:HWM_BOOL]
        * [en] Is Subscirbe
        * [cn] 是否订阅
        */
        virtual void OnSubscribeSubtitleResult(hwmsdk::HwmErrCode ret, const char* reason, bool isSubscirbe) {};

        /**
        * @brief
        * [en] Callback of GetSubtitleSupportLanguage interface.
        * [cn] 获取字幕语言列表
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        * @param targetLanguageTypeList [type:HwmTargetLanguageTypeList*]
        * [en] Target Language Type List
        * [cn] 目标语言类型列表
        */
        virtual void OnGetSubtitleSupportLanguageResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmTargetLanguageTypeList* targetLanguageTypeList) {};

/* auto generate code insert, forbit delete */

        /*********************************************以下是保留的接口回调*********************************************/

        /**
        * @brief [en] Callback of LoginByToken interface.
        *        [cn] token登录接口回调
        *
        * @param [in] hwmsdk::HwmErrCode ret               [en] Return code
        *                                                  [cn] 返回值
        * @param [in] const char* reason                   [en] Fail reason
        *                                                  [cn] 失败原因描述
        * @param [in] const HwmLoginResult* loginResult    [en] login result info
        *                                                  [cn] 登录结果信息
        * @attention [en] Reserved
        *            [cn] 保留接口回调
        **/
        virtual void OnLoginByTokenResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmLoginResult* loginResult) {};

        /**
         * @brief [en] Callback of EnablePair interface.
         *        [cn] 开启或关闭配对功能回调
         *
         * @param [in] hwmsdk::HwmErrCode ret    [en] Return code
         *                                       [cn] 返回值
         * @param [in] const char* reason        [en] Fail reason
         *                                       [cn] 失败原因描述
         * @attention [en] Reserved
         *            [cn] 保留接口回调
         **/
        virtual void OnEnablePairResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] Callback of JoinConfByLink interface.
        *        [cn] 通过会议链接入会回调
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        * @attention [en] Reserved
        *            [cn] 保留接口回调
        **/
        virtual void OnJoinConfByLinkResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en]  Callback of ShowMeetingWindow interface.
        *        [cn]  显示会议窗口
        *
        * @param [in] hwmsdk::HwmErrCode ret            [en] Return code
        *                                               [cn] 返回值
        * @param [in] const char* reason                [en] Fail reason
        *                                               [cn] 失败原因描述
        * @attention [en] Reserved
        *            [cn] 保留接口回调
        **/
        virtual void OnShowMeetingWindowResult(hwmsdk::HwmErrCode ret, const char* reason) {};

        /**
        * @brief [en] window handle list callback.when call GetHwndInfoList, will receive this calback
        *        [cn] 窗口句柄列表回调，GetRenderWindowInfoList 请求返回
        *
        * @param [in] hwmsdk::HwmErrCode ret                    [en] Return code
        *                                                       [cn] 返回值
        * @param [in] const char* reason                        [en] Fail reason
        *                                                       [cn] 失败原因描述
        * @param [in] const HwmHwndListInfo* hwndInfo           [en] window handle info
        *                                                       [cn] 窗口句柄信息
        */
        virtual void OnGetRenderWindowInfoListResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmHwndListInfo* hwndInfo) {};

        /*********************************************以上是保留的接口回调*********************************************/

        /**
        * @brief [en] Callback of StartProjection interface.
        *        [cn] 发起投屏回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        *
        * @param startProjectionResult [type:HwmStartProjectionResult]
        * [en] Start projection result
        * [cn] 发起投屏结果
        **/
        virtual void OnStartProjectionResult(hwmsdk::HwmErrCode ret, const char* reason, const HwmStartProjectionResult* startProjectionResult) {};

        /**
        * @brief [en] Callback of SetVirtualBackgroundDefaultImages interface.
        *        [cn] 设置虚拟背景默认图片接口回调
        *
        * @param ret [type:SDKERR]
        * [en] Error code
        * [cn] 错误码
        *
        * @param reason [type:HWM_CHAR*]
        * [en] Reason
        * [cn] 错误描述
        **/
        virtual void OnSetVirtualBackgroundDefaultImagesResult(hwmsdk::HwmErrCode ret, const char* reason) {};
    };
}

#endif /* _HWM_AGENT_CALLBACK_H_ */
